/**
 * @file GLViewport.cpp
 * @brief OpenGL viewport widget for 3D mesh rendering and visualization
 *
 * This file implements the GLViewport class which provides a Qt OpenGL widget
 * for rendering 3D meshes with various rendering modes, lighting setups, and
 * camera controls. Supports desktop and WebAssembly platforms.
 *
 * Copyright © 2025 Linus Suter
 * Released under the GNU/GPL License
 */

#include "GLViewport.h"
#include <cmath>

/**
 * @brief Constructor - Initializes the OpenGL viewport with default settings
 * @param parent Parent Qt widget
 *
 * Sets up initial camera position (30° X rotation, 45° Y rotation, zoom 10),
 * rendering mode (SOLID), and lighting mode (STUDIO).
 */
GLViewport::GLViewport(QWidget *parent)
    : QOpenGLWidget(parent), rotationX(30.0f), rotationY(45.0f), zoom(10.0f),
      wireframe(false), renderMode(SOLID), animTime(0.0f),
      showGrid(false), showAxis(false), backFaceCulling(false),
      autoFrameAll(true), meshNeedsUpdate(true), currentLightingMode(LIGHTING_STUDIO)
#ifdef __EMSCRIPTEN__
      , shaderProgram(nullptr), lineShaderProgram(nullptr), vbo(nullptr), vertexCount(0)
#endif
{
}

/**
 * @brief Destructor - Cleans up OpenGL resources
 */
GLViewport::~GLViewport() {
#ifdef __EMSCRIPTEN__
    delete shaderProgram;
    delete lineShaderProgram;
    delete vbo;
#endif
}

void GLViewport::clearMesh() {
    // Clear the current mesh completely
    currentMesh.clear();
    
    // Force update
    update();
}

/**
 * @brief Initializes OpenGL context and sets up rendering state
 *
 * Called once when the OpenGL context is first created. Enables depth testing,
 * lighting, color materials, and smooth shading. Sets background color to
 * dark gray (RGB: 0.2, 0.2, 0.21) and applies default studio lighting.
 */
void GLViewport::initializeGL() {
    initializeOpenGLFunctions();

    // Enable depth testing for proper 3D rendering
    glEnable(GL_DEPTH_TEST);
    glDepthFunc(GL_LESS);  // Pass depth test if fragment is closer

#ifndef __EMSCRIPTEN__
    // Enable lighting for realistic shading (not available in WebGL)
    glEnable(GL_LIGHTING);

    // Enable color materials to use vertex colors with lighting
    glEnable(GL_COLOR_MATERIAL);

    // Enable normal vector normalization for correct lighting
    glEnable(GL_NORMALIZE);

    // Configure color material mode for ambient and diffuse components
    glColorMaterial(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE);

    // Use smooth (Gouraud) shading by default
    glShadeModel(GL_SMOOTH);
#endif

    // Set background color to dark gray
    glClearColor(0.2f, 0.2f, 0.21f, 1.0f);

#ifndef __EMSCRIPTEN__
    // Apply the default lighting configuration
    applyLighting();
#else
    // Initialize shaders for WebGL
    initializeShaders();
#endif
}

void GLViewport::paintGL() {
    // Ensure background color is set (especially important for WebAssembly)
    glClearColor(0.2f, 0.2f, 0.21f, 1.0f);

    // Clear both color and depth buffers
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

    // Enable depth testing with proper settings
    glEnable(GL_DEPTH_TEST);
    glDepthFunc(GL_LEQUAL);  // Use LEQUAL instead of LESS for better precision
    glDepthMask(GL_TRUE);

    // Enable polygon offset to prevent z-fighting
    glEnable(GL_POLYGON_OFFSET_FILL);
    glPolygonOffset(1.0f, 1.0f);

    // Back-face culling controlled by checkbox
    if (backFaceCulling) {
        glEnable(GL_CULL_FACE);
        glCullFace(GL_BACK);
    } else {
        glDisable(GL_CULL_FACE);
    }

#ifndef __EMSCRIPTEN__
    // Setup projection matrix manually (instead of gluPerspective)
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();

    float aspect = (float)width() / (float)height();
    float fov = 45.0f * M_PI / 180.0f;
    float near = 0.1f;
    float far = 100.0f;

    float f = 1.0f / std::tan(fov / 2.0f);
    float projMatrix[16] = {
        f / aspect, 0.0f, 0.0f, 0.0f,
        0.0f, f, 0.0f, 0.0f,
        0.0f, 0.0f, (far + near) / (near - far), -1.0f,
        0.0f, 0.0f, (2.0f * far * near) / (near - far), 0.0f
    };
    glLoadMatrixf(projMatrix);

    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();
    glTranslatef(0.0f, 0.0f, -zoom);
    glRotatef(rotationX, 1.0f, 0.0f, 0.0f);
    glRotatef(rotationY, 0.0f, 1.0f, 0.0f);
#endif

    // Render grid and axis first (behind mesh)
    if (showGrid) renderGrid();
    if (showAxis) renderAxis();

#ifdef __EMSCRIPTEN__
    renderMeshModern();
#else
    renderMesh();
#endif
}

void GLViewport::resizeGL(int w, int h) {
    glViewport(0, 0, w, h);
}

void GLViewport::renderMesh() {
    if (currentMesh.getTriangleCount() == 0) return;

    if (meshNeedsUpdate) {
        updateMeshData();
    }

    if (vertexData.empty()) return;

    // Setup rendering based on mode
    bool useLighting = (renderMode == SOLID || renderMode == SOLID_WIREFRAME ||
                       renderMode == FLAT || renderMode == HIDDEN_LINE ||
                       renderMode == SMOOTH_SHADED || renderMode == FLAT_LINES);

#ifndef __EMSCRIPTEN__
    if (useLighting) {
        glEnable(GL_LIGHTING);
    } else {
        glDisable(GL_LIGHTING);
    }
#endif

#ifndef __EMSCRIPTEN__
    // For desktop, we can use polygon mode
    if (renderMode == WIREFRAME || wireframe) {
        glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
    } else {
        glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
    }
#endif

#ifndef __EMSCRIPTEN__
    // Shading model selection
    if (renderMode == FLAT || renderMode == FLAT_LINES) {
        glShadeModel(GL_FLAT);
    } else {
        glShadeModel(GL_SMOOTH);
    }
#endif

    // X-Ray mode setup
    if (renderMode == X_RAY) {
        glEnable(GL_BLEND);
        glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
        glDepthMask(GL_FALSE);
#ifndef __EMSCRIPTEN__
        glColor4f(0.5f, 0.7f, 1.0f, 0.3f);
#endif
    }

#ifndef __EMSCRIPTEN__
    // Enable client states for vertex arrays
    glEnableClientState(GL_VERTEX_ARRAY);
    glEnableClientState(GL_NORMAL_ARRAY);
    glEnableClientState(GL_COLOR_ARRAY);

    // Set up pointers (interleaved: pos(3) + normal(3) + color(3) = 9 floats per vertex)
    int stride = 9 * sizeof(float);
    glVertexPointer(3, GL_FLOAT, stride, &vertexData[0]);
    glNormalPointer(GL_FLOAT, stride, &vertexData[3]);
    glColorPointer(3, GL_FLOAT, stride, &vertexData[6]);

    // ✅ Use the *actual* number of vertices that were written
    const int drawVertexCount = static_cast<int>(vertexData.size() / 9);
    const int drawTriangleCount = drawVertexCount / 3;
    if (drawVertexCount <= 0) {
        // Clean up client state and return early
        glDisableClientState(GL_COLOR_ARRAY);
        glDisableClientState(GL_NORMAL_ARRAY);
        glDisableClientState(GL_VERTEX_ARRAY);
        return;
    }

    // Draw based on render mode
    switch (renderMode) {
        case POINTS:
            glPointSize(3.0f);
            glDrawArrays(GL_POINTS, 0, drawVertexCount);
            break;

        case WIREFRAME:
            glDisable(GL_LIGHTING);
            glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
            glDrawArrays(GL_TRIANGLES, 0, drawVertexCount);
            glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
            break;

        case FLAT:
        case SOLID:
        case SMOOTH_SHADED:
            glDrawArrays(GL_TRIANGLES, 0, drawVertexCount);
            break;

        case SOLID_WIREFRAME:
            // Draw filled with slight polygon offset
            glEnable(GL_POLYGON_OFFSET_FILL);
            glPolygonOffset(1.0f, 1.0f);
            glDrawArrays(GL_TRIANGLES, 0, drawVertexCount);
            glDisable(GL_POLYGON_OFFSET_FILL);

            // Draw wireframe on top
            glDisable(GL_LIGHTING);
            glColor3f(0.2f, 0.2f, 0.2f);
            for (int i = 0; i < drawTriangleCount; ++i) {
                glDrawArrays(GL_LINE_LOOP, i * 3, 3);
            }
            // Re-enable lighting only if it was used (your code already handles this around here)
            break;

        case HIDDEN_LINE:
            // Draw white filled polygons first (hidden surface removal)
            glDisable(GL_LIGHTING);
            glColor3f(1.0f, 1.0f, 1.0f);
            glDrawArrays(GL_TRIANGLES, 0, drawVertexCount);

            // Draw black wireframe on top
            glEnable(GL_POLYGON_OFFSET_LINE);
            glPolygonOffset(-1.0f, -1.0f);
            glColor3f(0.0f, 0.0f, 0.0f);
            for (int i = 0; i < drawTriangleCount; ++i) {
                glDrawArrays(GL_LINE_LOOP, i * 3, 3);
            }
            glDisable(GL_POLYGON_OFFSET_LINE);
            break;

        case FLAT_LINES:
            // (your fill pass)
            glDrawArrays(GL_TRIANGLES, 0, drawVertexCount);
            // (your line pass)
            for (int i = 0; i < drawTriangleCount; ++i) {
                glDrawArrays(GL_LINE_LOOP, i * 3, 3);
            }
            break;

        case X_RAY:
            glDrawArrays(GL_TRIANGLES, 0, drawVertexCount);
            break;
    }

    // Disable client states
    glDisableClientState(GL_COLOR_ARRAY);
    glDisableClientState(GL_NORMAL_ARRAY);
    glDisableClientState(GL_VERTEX_ARRAY);
#endif

    // Cleanup
    if (renderMode == X_RAY) {
        glDisable(GL_BLEND);
        glDepthMask(GL_TRUE);
    }
}

void GLViewport::setMesh(const Mesh& mesh) {
    makeCurrent();
    // Clear any existing mesh data first
    currentMesh.clear();
    
    // Set the new mesh
    currentMesh = mesh;
    meshNeedsUpdate = true;

    // Only rebuild VBOs if mesh has data
    if (mesh.getVertexCount() > 0) {
        // Rebuild VBOs with new data
        //setupVBOs();
        
        // Frame the mesh if auto-frame is enabled
        if (autoFrameAll) {
            frameAll();
        }
    }
    
    doneCurrent();

    // Update display
    update();
}

void GLViewport::updateMeshData() {
    vertexData.clear();

    for (size_t i = 0; i < currentMesh.getTriangleCount(); i++) {
        const Triangle& tri = currentMesh.getTriangle(i);

        // Get all three vertices to check for degenerate triangles
        const Vertex& v0 = currentMesh.getVertex(tri.v0);
        const Vertex& v1 = currentMesh.getVertex(tri.v1);
        const Vertex& v2 = currentMesh.getVertex(tri.v2);

        // Check if triangle is degenerate (all vertices at same position)
        float dx1 = v1.position.x - v0.position.x;
        float dy1 = v1.position.y - v0.position.y;
        float dz1 = v1.position.z - v0.position.z;
        float dx2 = v2.position.x - v0.position.x;
        float dy2 = v2.position.y - v0.position.y;
        float dz2 = v2.position.z - v0.position.z;

        // Cross product for triangle normal (geometric normal)
        float cnx = dy1 * dz2 - dz1 * dy2;
        float cny = dz1 * dx2 - dx1 * dz2;
        float cnz = dx1 * dy2 - dy1 * dx2;
        float triangleArea = std::sqrt(cnx * cnx + cny * cny + cnz * cnz);

        // Skip degenerate triangles (zero area)
        if (triangleArea < 0.00001f) {
            continue;
        }

        // Normalize geometric normal
        cnx /= triangleArea;
        cny /= triangleArea;
        cnz /= triangleArea;

        // Add vertices in original winding order
        for (int j = 0; j < 3; j++) {
            unsigned int idx = (j == 0) ? tri.v0 : ((j == 1) ? tri.v1 : tri.v2);
            const Vertex& v = currentMesh.getVertex(idx);

            // Position
            vertexData.push_back(v.position.x);
            vertexData.push_back(v.position.y);
            vertexData.push_back(v.position.z);

            // Normal - check for NaN or invalid values
            float nx = std::isnan(v.normal.x) ? 0.0f : v.normal.x;
            float ny = std::isnan(v.normal.y) ? 0.0f : v.normal.y;
            float nz = std::isnan(v.normal.z) ? 0.0f : v.normal.z;
            float nlen = std::sqrt(nx * nx + ny * ny + nz * nz);

            // If normal is invalid, use the geometric normal
            if (nlen < 0.0001f) {
                vertexData.push_back(cnx);
                vertexData.push_back(cny);
                vertexData.push_back(cnz);
            } else {
                // Use vertex normal as-is
                vertexData.push_back(nx);
                vertexData.push_back(ny);
                vertexData.push_back(nz);
            }

            // Color
            vertexData.push_back(v.color.x);
            vertexData.push_back(v.color.y);
            vertexData.push_back(v.color.z);
        }
    }

    meshNeedsUpdate = false;
}

void GLViewport::mousePressEvent(QMouseEvent *event) {
    lastMousePos = event->pos();
}

void GLViewport::mouseMoveEvent(QMouseEvent *event) {
    int dx = event->position().x() - lastMousePos.x();
    int dy = event->position().y() - lastMousePos.y();

    if (event->buttons() & Qt::LeftButton) {
        rotationX += dy * 0.5f;
        rotationY += dx * 0.5f;
        update();
    }

    lastMousePos = event->pos();
}

void GLViewport::wheelEvent(QWheelEvent *event) {
    zoom -= event->angleDelta().y() * 0.01f;
    zoom = std::max(1.0f, std::min(zoom, 50.0f));
    update();
}

void GLViewport::resetView() {
    rotationX = 30.0f;
    rotationY = 45.0f;
    zoom = 10.0f;
    update();
}

void GLViewport::setFrontView() {
    rotationX = 0.0f;
    rotationY = 0.0f;
    update();
}

void GLViewport::setTopView() {
    rotationX = 90.0f;
    rotationY = 0.0f;
    update();
}

void GLViewport::setSideView() {
    rotationX = 0.0f;
    rotationY = 90.0f;
    update();
}

void GLViewport::resetAll() {
    makeCurrent();

    currentMesh.clear();
    vertexData.clear();
    meshNeedsUpdate = false;

    rotationX = 30.0f;
    rotationY = 45.0f;
    zoom = 10.0f;
    lastMousePos = QPoint();

    renderMode = SOLID;
    currentLightingMode = LIGHTING_STUDIO;
    showGrid = false;
    showAxis = false;
    backFaceCulling = false;
    autoFrameAll = false;
    animTime = 0.0f;

#ifndef __EMSCRIPTEN__
    glDisable(GL_BLEND);
    glDisable(GL_CULL_FACE);
    glDepthMask(GL_TRUE);
    glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);

    applyLighting();
#endif

    doneCurrent();

    update();
    repaint();
}

void GLViewport::renderGrid() {
#ifdef __EMSCRIPTEN__
    if (!lineShaderProgram) return;

    // Create grid lines with position and color (gray for all)
    std::vector<float> gridData;
    for (int i = -10; i <= 10; i++) {
        // Lines parallel to X axis
        gridData.push_back(-10.0f); gridData.push_back(0.0f); gridData.push_back(i);
        gridData.push_back(0.3f); gridData.push_back(0.3f); gridData.push_back(0.3f);
        gridData.push_back(10.0f); gridData.push_back(0.0f); gridData.push_back(i);
        gridData.push_back(0.3f); gridData.push_back(0.3f); gridData.push_back(0.3f);

        // Lines parallel to Z axis
        gridData.push_back(i); gridData.push_back(0.0f); gridData.push_back(-10.0f);
        gridData.push_back(0.3f); gridData.push_back(0.3f); gridData.push_back(0.3f);
        gridData.push_back(i); gridData.push_back(0.0f); gridData.push_back(10.0f);
        gridData.push_back(0.3f); gridData.push_back(0.3f); gridData.push_back(0.3f);
    }

    // Upload to VBO
    vbo->bind();
    vbo->allocate(gridData.data(), gridData.size() * sizeof(float));

    // Use line shader
    lineShaderProgram->bind();

    // Calculate MVP
    QMatrix4x4 projection, view, model;
    float nearPlane = zoom * 0.1f;
    float farPlane = zoom * 10.0f;
    projection.perspective(45.0f, (float)width() / (float)height(), nearPlane, farPlane);
    view.translate(0.0f, 0.0f, -zoom);
    view.rotate(rotationX, 1.0f, 0.0f, 0.0f);
    view.rotate(rotationY, 0.0f, 1.0f, 0.0f);
    QMatrix4x4 mvp = projection * view * model;

    lineShaderProgram->setUniformValue("uMVP", mvp);

    // Set attributes (position and color interleaved)
    int stride = 6 * sizeof(float); // 3 pos + 3 color
    int posLoc = lineShaderProgram->attributeLocation("aPosition");
    int colorLoc = lineShaderProgram->attributeLocation("aColor");

    lineShaderProgram->enableAttributeArray(posLoc);
    lineShaderProgram->setAttributeBuffer(posLoc, GL_FLOAT, 0, 3, stride);

    lineShaderProgram->enableAttributeArray(colorLoc);
    lineShaderProgram->setAttributeBuffer(colorLoc, GL_FLOAT, 3 * sizeof(float), 3, stride);

    glDrawArrays(GL_LINES, 0, gridData.size() / 6);

    lineShaderProgram->disableAttributeArray(posLoc);
    lineShaderProgram->disableAttributeArray(colorLoc);
    vbo->release();
    lineShaderProgram->release();
#else
    glDisable(GL_LIGHTING);
    glColor3f(0.3f, 0.3f, 0.3f);
    glLineWidth(1.0f);

    glBegin(GL_LINES);
    for (int i = -10; i <= 10; i++) {
        // Lines parallel to X axis
        glVertex3f(-10.0f, 0.0f, i);
        glVertex3f(10.0f, 0.0f, i);

        // Lines parallel to Z axis
        glVertex3f(i, 0.0f, -10.0f);
        glVertex3f(i, 0.0f, 10.0f);
    }
    glEnd();

    glEnable(GL_LIGHTING);
#endif
}

void GLViewport::renderAxis() {
#ifdef __EMSCRIPTEN__
    if (!lineShaderProgram) return;

    // Create axis lines with colors (position + color)
    std::vector<float> axisData;

    // X axis - Red
    axisData.push_back(0.0f); axisData.push_back(0.0f); axisData.push_back(0.0f);
    axisData.push_back(1.0f); axisData.push_back(0.0f); axisData.push_back(0.0f); // red
    axisData.push_back(3.0f); axisData.push_back(0.0f); axisData.push_back(0.0f);
    axisData.push_back(1.0f); axisData.push_back(0.0f); axisData.push_back(0.0f); // red

    // Y axis - Green
    axisData.push_back(0.0f); axisData.push_back(0.0f); axisData.push_back(0.0f);
    axisData.push_back(0.0f); axisData.push_back(1.0f); axisData.push_back(0.0f); // green
    axisData.push_back(0.0f); axisData.push_back(3.0f); axisData.push_back(0.0f);
    axisData.push_back(0.0f); axisData.push_back(1.0f); axisData.push_back(0.0f); // green

    // Z axis - Blue
    axisData.push_back(0.0f); axisData.push_back(0.0f); axisData.push_back(0.0f);
    axisData.push_back(0.0f); axisData.push_back(0.0f); axisData.push_back(1.0f); // blue
    axisData.push_back(0.0f); axisData.push_back(0.0f); axisData.push_back(3.0f);
    axisData.push_back(0.0f); axisData.push_back(0.0f); axisData.push_back(1.0f); // blue

    // Upload to VBO
    vbo->bind();
    vbo->allocate(axisData.data(), axisData.size() * sizeof(float));

    // Use line shader
    lineShaderProgram->bind();

    // Calculate MVP
    QMatrix4x4 projection, view, model;
    float nearPlane = zoom * 0.1f;
    float farPlane = zoom * 10.0f;
    projection.perspective(45.0f, (float)width() / (float)height(), nearPlane, farPlane);
    view.translate(0.0f, 0.0f, -zoom);
    view.rotate(rotationX, 1.0f, 0.0f, 0.0f);
    view.rotate(rotationY, 0.0f, 1.0f, 0.0f);
    QMatrix4x4 mvp = projection * view * model;

    lineShaderProgram->setUniformValue("uMVP", mvp);

    // Set attributes (position and color interleaved)
    int stride = 6 * sizeof(float); // 3 pos + 3 color
    int posLoc = lineShaderProgram->attributeLocation("aPosition");
    int colorLoc = lineShaderProgram->attributeLocation("aColor");

    lineShaderProgram->enableAttributeArray(posLoc);
    lineShaderProgram->setAttributeBuffer(posLoc, GL_FLOAT, 0, 3, stride);

    lineShaderProgram->enableAttributeArray(colorLoc);
    lineShaderProgram->setAttributeBuffer(colorLoc, GL_FLOAT, 3 * sizeof(float), 3, stride);

    glDrawArrays(GL_LINES, 0, 6); // 6 vertices (3 lines x 2 vertices)

    lineShaderProgram->disableAttributeArray(posLoc);
    lineShaderProgram->disableAttributeArray(colorLoc);
    vbo->release();
    lineShaderProgram->release();
#else
    glDisable(GL_LIGHTING);
    glLineWidth(2.0f);

    glBegin(GL_LINES);
    // X axis - Red
    glColor3f(1.0f, 0.0f, 0.0f);
    glVertex3f(0.0f, 0.0f, 0.0f);
    glVertex3f(3.0f, 0.0f, 0.0f);

    // Y axis - Green
    glColor3f(0.0f, 1.0f, 0.0f);
    glVertex3f(0.0f, 0.0f, 0.0f);
    glVertex3f(0.0f, 3.0f, 0.0f);

    // Z axis - Blue
    glColor3f(0.0f, 0.0f, 1.0f);
    glVertex3f(0.0f, 0.0f, 0.0f);
    glVertex3f(0.0f, 0.0f, 3.0f);
    glEnd();

    glEnable(GL_LIGHTING);
#endif
}

void GLViewport::setLightingMode(LightingMode mode) {
    currentLightingMode = mode;
#ifndef __EMSCRIPTEN__
    makeCurrent();
    applyLighting();
    doneCurrent();
#endif
    update();
}

void GLViewport::applyLighting() {
#ifndef __EMSCRIPTEN__
    // Disable all lights first
    glDisable(GL_LIGHT0);
    glDisable(GL_LIGHT1);
    glDisable(GL_LIGHT2);

    switch (currentLightingMode) {
        case LIGHTING_DEFAULT:
            // Enhanced three-light default setup with better illumination
            glEnable(GL_LIGHT0);
            glEnable(GL_LIGHT1);
            glEnable(GL_LIGHT2);
            {
                // Main key light from top-right
                GLfloat light0Pos[] = {10.0f, 10.0f, 10.0f, 1.0f};
                GLfloat light0Ambient[] = {0.1f, 0.1f, 0.1f, 1.0f};
                GLfloat light0Diffuse[] = {0.7f, 0.7f, 0.68f, 1.0f};
                GLfloat light0Specular[] = {0.5f, 0.5f, 0.5f, 1.0f};

                glLightfv(GL_LIGHT0, GL_POSITION, light0Pos);
                glLightfv(GL_LIGHT0, GL_AMBIENT, light0Ambient);
                glLightfv(GL_LIGHT0, GL_DIFFUSE, light0Diffuse);
                glLightfv(GL_LIGHT0, GL_SPECULAR, light0Specular);

                // Fill light from left side
                GLfloat light1Pos[] = {-8.0f, 5.0f, -8.0f, 1.0f};
                GLfloat light1Ambient[] = {0.05f, 0.05f, 0.05f, 1.0f};
                GLfloat light1Diffuse[] = {0.4f, 0.4f, 0.42f, 1.0f};
                GLfloat light1Specular[] = {0.2f, 0.2f, 0.2f, 1.0f};

                glLightfv(GL_LIGHT1, GL_POSITION, light1Pos);
                glLightfv(GL_LIGHT1, GL_AMBIENT, light1Ambient);
                glLightfv(GL_LIGHT1, GL_DIFFUSE, light1Diffuse);
                glLightfv(GL_LIGHT1, GL_SPECULAR, light1Specular);

                // Back/rim light for depth
                GLfloat light2Pos[] = {0.0f, 8.0f, -10.0f, 1.0f};
                GLfloat light2Ambient[] = {0.0f, 0.0f, 0.0f, 1.0f};
                GLfloat light2Diffuse[] = {0.3f, 0.3f, 0.32f, 1.0f};
                GLfloat light2Specular[] = {0.3f, 0.3f, 0.3f, 1.0f};

                glLightfv(GL_LIGHT2, GL_POSITION, light2Pos);
                glLightfv(GL_LIGHT2, GL_AMBIENT, light2Ambient);
                glLightfv(GL_LIGHT2, GL_DIFFUSE, light2Diffuse);
                glLightfv(GL_LIGHT2, GL_SPECULAR, light2Specular);

                GLfloat matAmbient[] = {0.15f, 0.15f, 0.15f, 1.0f};
                GLfloat matDiffuse[] = {0.8f, 0.8f, 0.8f, 1.0f};
                GLfloat matSpecular[] = {0.4f, 0.4f, 0.4f, 1.0f};
                GLfloat matShininess[] = {32.0f};

                glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT, matAmbient);
                glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE, matDiffuse);
                glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, matSpecular);
                glMaterialfv(GL_FRONT_AND_BACK, GL_SHININESS, matShininess);
            }
            break;

        case LIGHTING_STUDIO:
            // Professional three-point lighting with enhanced brightness
            glEnable(GL_LIGHT0);
            glEnable(GL_LIGHT1);
            glEnable(GL_LIGHT2);
            {
                // Key light - brighter main light from top-right
                GLfloat light0Pos[] = {12.0f, 12.0f, 10.0f, 1.0f};
                GLfloat light0Ambient[] = {0.15f, 0.15f, 0.15f, 1.0f};
                GLfloat light0Diffuse[] = {1.0f, 0.98f, 0.95f, 1.0f};
                GLfloat light0Specular[] = {1.0f, 1.0f, 1.0f, 1.0f};

                glLightfv(GL_LIGHT0, GL_POSITION, light0Pos);
                glLightfv(GL_LIGHT0, GL_AMBIENT, light0Ambient);
                glLightfv(GL_LIGHT0, GL_DIFFUSE, light0Diffuse);
                glLightfv(GL_LIGHT0, GL_SPECULAR, light0Specular);

                // Fill light - softer from left
                GLfloat light1Pos[] = {-10.0f, 6.0f, 8.0f, 1.0f};
                GLfloat light1Ambient[] = {0.05f, 0.05f, 0.05f, 1.0f};
                GLfloat light1Diffuse[] = {0.5f, 0.52f, 0.55f, 1.0f};
                GLfloat light1Specular[] = {0.3f, 0.3f, 0.3f, 1.0f};

                glLightfv(GL_LIGHT1, GL_POSITION, light1Pos);
                glLightfv(GL_LIGHT1, GL_AMBIENT, light1Ambient);
                glLightfv(GL_LIGHT1, GL_DIFFUSE, light1Diffuse);
                glLightfv(GL_LIGHT1, GL_SPECULAR, light1Specular);

                // Back light - rim light for depth
                GLfloat light2Pos[] = {0.0f, 8.0f, -12.0f, 1.0f};
                GLfloat light2Ambient[] = {0.0f, 0.0f, 0.0f, 1.0f};
                GLfloat light2Diffuse[] = {0.4f, 0.42f, 0.45f, 1.0f};
                GLfloat light2Specular[] = {0.6f, 0.6f, 0.65f, 1.0f};

                glLightfv(GL_LIGHT2, GL_POSITION, light2Pos);
                glLightfv(GL_LIGHT2, GL_AMBIENT, light2Ambient);
                glLightfv(GL_LIGHT2, GL_DIFFUSE, light2Diffuse);
                glLightfv(GL_LIGHT2, GL_SPECULAR, light2Specular);

                // Enhanced material properties
                GLfloat matAmbient[] = {0.25f, 0.25f, 0.25f, 1.0f};
                GLfloat matDiffuse[] = {0.9f, 0.9f, 0.9f, 1.0f};
                GLfloat matSpecular[] = {0.7f, 0.7f, 0.7f, 1.0f};
                GLfloat matShininess[] = {40.0f};

                glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT, matAmbient);
                glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE, matDiffuse);
                glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, matSpecular);
                glMaterialfv(GL_FRONT_AND_BACK, GL_SHININESS, matShininess);
            }
            break;

        case LIGHTING_BRIGHT:
            // Bright, evenly lit scene
            glEnable(GL_LIGHT0);
            glEnable(GL_LIGHT1);
            {
                GLfloat light0Pos[] = {10.0f, 10.0f, 10.0f, 1.0f};
                GLfloat light0Ambient[] = {0.3f, 0.3f, 0.3f, 1.0f};
                GLfloat light0Diffuse[] = {1.0f, 1.0f, 1.0f, 1.0f};
                GLfloat light0Specular[] = {1.0f, 1.0f, 1.0f, 1.0f};

                glLightfv(GL_LIGHT0, GL_POSITION, light0Pos);
                glLightfv(GL_LIGHT0, GL_AMBIENT, light0Ambient);
                glLightfv(GL_LIGHT0, GL_DIFFUSE, light0Diffuse);
                glLightfv(GL_LIGHT0, GL_SPECULAR, light0Specular);

                GLfloat light1Pos[] = {-10.0f, 10.0f, -10.0f, 1.0f};
                GLfloat light1Ambient[] = {0.2f, 0.2f, 0.2f, 1.0f};
                GLfloat light1Diffuse[] = {0.8f, 0.8f, 0.8f, 1.0f};
                GLfloat light1Specular[] = {0.5f, 0.5f, 0.5f, 1.0f};

                glLightfv(GL_LIGHT1, GL_POSITION, light1Pos);
                glLightfv(GL_LIGHT1, GL_AMBIENT, light1Ambient);
                glLightfv(GL_LIGHT1, GL_DIFFUSE, light1Diffuse);
                glLightfv(GL_LIGHT1, GL_SPECULAR, light1Specular);

                GLfloat matAmbient[] = {0.3f, 0.3f, 0.3f, 1.0f};
                GLfloat matDiffuse[] = {1.0f, 1.0f, 1.0f, 1.0f};
                GLfloat matSpecular[] = {0.3f, 0.3f, 0.3f, 1.0f};
                GLfloat matShininess[] = {20.0f};

                glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT, matAmbient);
                glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE, matDiffuse);
                glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, matSpecular);
                glMaterialfv(GL_FRONT_AND_BACK, GL_SHININESS, matShininess);
            }
            break;

        case LIGHTING_DRAMATIC:
            // Single strong directional light with high contrast
            glEnable(GL_LIGHT0);
            {
                GLfloat light0Pos[] = {15.0f, 10.0f, 5.0f, 1.0f};
                GLfloat light0Ambient[] = {0.05f, 0.05f, 0.05f, 1.0f};
                GLfloat light0Diffuse[] = {1.0f, 0.95f, 0.9f, 1.0f};
                GLfloat light0Specular[] = {1.0f, 1.0f, 1.0f, 1.0f};

                glLightfv(GL_LIGHT0, GL_POSITION, light0Pos);
                glLightfv(GL_LIGHT0, GL_AMBIENT, light0Ambient);
                glLightfv(GL_LIGHT0, GL_DIFFUSE, light0Diffuse);
                glLightfv(GL_LIGHT0, GL_SPECULAR, light0Specular);

                GLfloat matAmbient[] = {0.05f, 0.05f, 0.05f, 1.0f};
                GLfloat matDiffuse[] = {0.8f, 0.8f, 0.8f, 1.0f};
                GLfloat matSpecular[] = {0.9f, 0.9f, 0.9f, 1.0f};
                GLfloat matShininess[] = {64.0f};

                glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT, matAmbient);
                glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE, matDiffuse);
                glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, matSpecular);
                glMaterialfv(GL_FRONT_AND_BACK, GL_SHININESS, matShininess);
            }
            break;

        case LIGHTING_SOFT:
            // Soft, diffuse lighting with minimal specular
            glEnable(GL_LIGHT0);
            glEnable(GL_LIGHT1);
            {
                GLfloat light0Pos[] = {8.0f, 10.0f, 8.0f, 1.0f};
                GLfloat light0Ambient[] = {0.2f, 0.2f, 0.2f, 1.0f};
                GLfloat light0Diffuse[] = {0.6f, 0.6f, 0.6f, 1.0f};
                GLfloat light0Specular[] = {0.2f, 0.2f, 0.2f, 1.0f};

                glLightfv(GL_LIGHT0, GL_POSITION, light0Pos);
                glLightfv(GL_LIGHT0, GL_AMBIENT, light0Ambient);
                glLightfv(GL_LIGHT0, GL_DIFFUSE, light0Diffuse);
                glLightfv(GL_LIGHT0, GL_SPECULAR, light0Specular);

                GLfloat light1Pos[] = {-8.0f, 10.0f, -8.0f, 1.0f};
                GLfloat light1Ambient[] = {0.15f, 0.15f, 0.15f, 1.0f};
                GLfloat light1Diffuse[] = {0.5f, 0.5f, 0.5f, 1.0f};
                GLfloat light1Specular[] = {0.1f, 0.1f, 0.1f, 1.0f};

                glLightfv(GL_LIGHT1, GL_POSITION, light1Pos);
                glLightfv(GL_LIGHT1, GL_AMBIENT, light1Ambient);
                glLightfv(GL_LIGHT1, GL_DIFFUSE, light1Diffuse);
                glLightfv(GL_LIGHT1, GL_SPECULAR, light1Specular);

                GLfloat matAmbient[] = {0.25f, 0.25f, 0.25f, 1.0f};
                GLfloat matDiffuse[] = {0.7f, 0.7f, 0.7f, 1.0f};
                GLfloat matSpecular[] = {0.1f, 0.1f, 0.1f, 1.0f};
                GLfloat matShininess[] = {10.0f};

                glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT, matAmbient);
                glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE, matDiffuse);
                glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, matSpecular);
                glMaterialfv(GL_FRONT_AND_BACK, GL_SHININESS, matShininess);
            }
            break;

        case LIGHTING_RIM:
            // Rim lighting effect with back light emphasis
            glEnable(GL_LIGHT0);
            glEnable(GL_LIGHT1);
            glEnable(GL_LIGHT2);
            {
                // Front fill
                GLfloat light0Pos[] = {0.0f, 5.0f, 10.0f, 1.0f};
                GLfloat light0Ambient[] = {0.1f, 0.1f, 0.1f, 1.0f};
                GLfloat light0Diffuse[] = {0.4f, 0.4f, 0.4f, 1.0f};
                GLfloat light0Specular[] = {0.2f, 0.2f, 0.2f, 1.0f};

                glLightfv(GL_LIGHT0, GL_POSITION, light0Pos);
                glLightfv(GL_LIGHT0, GL_AMBIENT, light0Ambient);
                glLightfv(GL_LIGHT0, GL_DIFFUSE, light0Diffuse);
                glLightfv(GL_LIGHT0, GL_SPECULAR, light0Specular);

                // Left rim
                GLfloat light1Pos[] = {-15.0f, 8.0f, -5.0f, 1.0f};
                GLfloat light1Ambient[] = {0.0f, 0.0f, 0.0f, 1.0f};
                GLfloat light1Diffuse[] = {0.7f, 0.7f, 0.8f, 1.0f};
                GLfloat light1Specular[] = {0.8f, 0.8f, 0.9f, 1.0f};

                glLightfv(GL_LIGHT1, GL_POSITION, light1Pos);
                glLightfv(GL_LIGHT1, GL_AMBIENT, light1Ambient);
                glLightfv(GL_LIGHT1, GL_DIFFUSE, light1Diffuse);
                glLightfv(GL_LIGHT1, GL_SPECULAR, light1Specular);

                // Right rim
                GLfloat light2Pos[] = {15.0f, 8.0f, -5.0f, 1.0f};
                GLfloat light2Ambient[] = {0.0f, 0.0f, 0.0f, 1.0f};
                GLfloat light2Diffuse[] = {0.7f, 0.7f, 0.8f, 1.0f};
                GLfloat light2Specular[] = {0.8f, 0.8f, 0.9f, 1.0f};

                glLightfv(GL_LIGHT2, GL_POSITION, light2Pos);
                glLightfv(GL_LIGHT2, GL_AMBIENT, light2Ambient);
                glLightfv(GL_LIGHT2, GL_DIFFUSE, light2Diffuse);
                glLightfv(GL_LIGHT2, GL_SPECULAR, light2Specular);

                GLfloat matAmbient[] = {0.1f, 0.1f, 0.1f, 1.0f};
                GLfloat matDiffuse[] = {0.6f, 0.6f, 0.6f, 1.0f};
                GLfloat matSpecular[] = {0.7f, 0.7f, 0.7f, 1.0f};
                GLfloat matShininess[] = {50.0f};

                glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT, matAmbient);
                glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE, matDiffuse);
                glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, matSpecular);
                glMaterialfv(GL_FRONT_AND_BACK, GL_SHININESS, matShininess);
            }
            break;
    }
#endif
}

#ifdef __EMSCRIPTEN__
void GLViewport::initializeShaders() {
    // Create shader program
    shaderProgram = new QOpenGLShaderProgram(this);

    // Vertex shader (GLSL ES)
    const char *vertexShaderSource = R"(
        attribute vec3 aPosition;
        attribute vec3 aNormal;
        attribute vec3 aColor;

        uniform mat4 uMVP;
        uniform mat4 uModelView;
        uniform mat3 uNormalMatrix;

        varying vec3 vColor;
        varying vec3 vNormal;
        varying vec3 vPosition;

        void main() {
            gl_Position = uMVP * vec4(aPosition, 1.0);
            vPosition = vec3(uModelView * vec4(aPosition, 1.0));

            // Transform normal - check input normal first
            float inputLen = length(aNormal);
            vec3 normalToTransform = aNormal;

            // If input normal is invalid, use a default
            if (inputLen < 0.0001) {
                normalToTransform = vec3(0.0, 0.0, 1.0);
            }

            // Transform and normalize
            vec3 transformedNormal = uNormalMatrix * normalToTransform;
            float len = length(transformedNormal);
            if (len > 0.0001) {
                vNormal = transformedNormal / len;
            } else {
                vNormal = vec3(0.0, 0.0, 1.0);
            }

            vColor = aColor;
        }
    )";

    // Fragment shader (GLSL ES) - Multi-mode lighting matching desktop presets
    const char *fragmentShaderSource = R"(
        precision highp float;

        varying vec3 vColor;
        varying vec3 vNormal;
        varying vec3 vPosition;

        uniform int uLightingMode;

        void main() {
            // Normalize the normal (in case it got denormalized during interpolation)
            vec3 N = normalize(vNormal);

            // Flip normal to face camera if it's pointing away (two-sided lighting)
            if (!gl_FrontFacing) {
                N = -N;
            }

            vec3 V = normalize(-vPosition); // View direction
            vec3 finalColor = vec3(0.0);

            // LIGHTING_DEFAULT (0) - Enhanced three-light setup
            if (uLightingMode == 0) {
                vec3 L0 = normalize(vec3(10.0, 10.0, 10.0));   // Main key light
                vec3 L1 = normalize(vec3(-8.0, 5.0, -8.0));    // Fill light
                vec3 L2 = normalize(vec3(0.0, 8.0, -10.0));    // Back light

                float diff0 = max(dot(N, L0), 0.0);
                float diff1 = max(dot(N, L1), 0.0);
                float diff2 = max(dot(N, L2), 0.0);

                vec3 ambient = vColor * 0.15;
                vec3 diffuse0 = vColor * 0.7 * diff0;
                vec3 diffuse1 = vColor * 0.4 * diff1;
                vec3 diffuse2 = vColor * 0.3 * diff2;

                vec3 H0 = normalize(L0 + V);
                vec3 H2 = normalize(L2 + V);
                float spec0 = pow(max(dot(N, H0), 0.0), 32.0);
                float spec2 = pow(max(dot(N, H2), 0.0), 32.0);
                vec3 specular = vec3(0.4) * spec0 + vec3(0.3) * spec2;

                finalColor = ambient + diffuse0 + diffuse1 + diffuse2 + specular;
            }
            // LIGHTING_STUDIO (1) - Three-point lighting
            else if (uLightingMode == 1) {
                vec3 L0 = normalize(vec3(12.0, 12.0, 10.0));  // Key light
                vec3 L1 = normalize(vec3(-10.0, 6.0, 8.0));   // Fill light
                vec3 L2 = normalize(vec3(0.0, 8.0, -12.0));   // Back light

                float diff0 = max(dot(N, L0), 0.0);
                float diff1 = max(dot(N, L1), 0.0);
                float diff2 = max(dot(N, L2), 0.0);

                vec3 ambient = vColor * 0.25;
                vec3 diffuse0 = vColor * 1.0 * diff0;
                vec3 diffuse1 = vColor * 0.5 * diff1;
                vec3 diffuse2 = vColor * 0.4 * diff2;

                vec3 H0 = normalize(L0 + V);
                vec3 H2 = normalize(L2 + V);
                float spec0 = pow(max(dot(N, H0), 0.0), 40.0);
                float spec2 = pow(max(dot(N, H2), 0.0), 40.0);
                vec3 specular = vec3(0.7) * spec0 + vec3(0.6) * spec2;

                finalColor = ambient + diffuse0 + diffuse1 + diffuse2 + specular;
            }
            // LIGHTING_BRIGHT (2) - Bright, evenly lit
            else if (uLightingMode == 2) {
                vec3 L0 = normalize(vec3(10.0, 10.0, 10.0));
                vec3 L1 = normalize(vec3(-10.0, 10.0, -10.0));

                float diff0 = max(dot(N, L0), 0.0);
                float diff1 = max(dot(N, L1), 0.0);

                vec3 ambient = vColor * 0.3;
                vec3 diffuse0 = vColor * 1.0 * diff0;
                vec3 diffuse1 = vColor * 0.8 * diff1;

                vec3 H0 = normalize(L0 + V);
                float spec0 = pow(max(dot(N, H0), 0.0), 20.0);
                vec3 specular = vec3(0.3) * spec0;

                finalColor = ambient + diffuse0 + diffuse1 + specular;
            }
            // LIGHTING_DRAMATIC (3) - Single strong directional light
            else if (uLightingMode == 3) {
                vec3 L0 = normalize(vec3(15.0, 10.0, 5.0));

                float diff0 = max(dot(N, L0), 0.0);

                vec3 ambient = vColor * 0.05;
                vec3 diffuse0 = vColor * 0.95 * diff0;

                vec3 H0 = normalize(L0 + V);
                float spec0 = pow(max(dot(N, H0), 0.0), 64.0);
                vec3 specular = vec3(0.9) * spec0;

                finalColor = ambient + diffuse0 + specular;
            }
            // LIGHTING_SOFT (4) - Soft, diffuse lighting
            else if (uLightingMode == 4) {
                vec3 L0 = normalize(vec3(8.0, 10.0, 8.0));
                vec3 L1 = normalize(vec3(-8.0, 10.0, -8.0));

                float diff0 = max(dot(N, L0), 0.0);
                float diff1 = max(dot(N, L1), 0.0);

                vec3 ambient = vColor * 0.25;
                vec3 diffuse0 = vColor * 0.6 * diff0;
                vec3 diffuse1 = vColor * 0.5 * diff1;

                vec3 H0 = normalize(L0 + V);
                float spec0 = pow(max(dot(N, H0), 0.0), 10.0);
                vec3 specular = vec3(0.1) * spec0;

                finalColor = ambient + diffuse0 + diffuse1 + specular;
            }
            // LIGHTING_RIM (5) - Rim lighting effect
            else if (uLightingMode == 5) {
                vec3 L0 = normalize(vec3(0.0, 5.0, 10.0));    // Front fill
                vec3 L1 = normalize(vec3(-15.0, 8.0, -5.0));  // Left rim
                vec3 L2 = normalize(vec3(15.0, 8.0, -5.0));   // Right rim

                float diff0 = max(dot(N, L0), 0.0);
                float diff1 = max(dot(N, L1), 0.0);
                float diff2 = max(dot(N, L2), 0.0);

                vec3 ambient = vColor * 0.1;
                vec3 diffuse0 = vColor * 0.4 * diff0;
                vec3 diffuse1 = vColor * 0.7 * diff1;
                vec3 diffuse2 = vColor * 0.7 * diff2;

                vec3 H1 = normalize(L1 + V);
                vec3 H2 = normalize(L2 + V);
                float spec1 = pow(max(dot(N, H1), 0.0), 50.0);
                float spec2 = pow(max(dot(N, H2), 0.0), 50.0);
                vec3 specular = vec3(0.8) * spec1 + vec3(0.8) * spec2;

                finalColor = ambient + diffuse0 + diffuse1 + diffuse2 + specular;
            }

            gl_FragColor = vec4(finalColor, 1.0);
        }
    )";

    // Compile and link shaders
    shaderProgram->addShaderFromSourceCode(QOpenGLShader::Vertex, vertexShaderSource);
    shaderProgram->addShaderFromSourceCode(QOpenGLShader::Fragment, fragmentShaderSource);
    shaderProgram->link();

    // Create simple line shader for grid and axis
    lineShaderProgram = new QOpenGLShaderProgram(this);

    const char *lineVertexShader = R"(
        attribute vec3 aPosition;
        attribute vec3 aColor;
        uniform mat4 uMVP;
        varying vec3 vColor;
        void main() {
            gl_Position = uMVP * vec4(aPosition, 1.0);
            vColor = aColor;
        }
    )";

    const char *lineFragmentShader = R"(
        precision highp float;
        varying vec3 vColor;
        void main() {
            gl_FragColor = vec4(vColor, 1.0);
        }
    )";

    lineShaderProgram->addShaderFromSourceCode(QOpenGLShader::Vertex, lineVertexShader);
    lineShaderProgram->addShaderFromSourceCode(QOpenGLShader::Fragment, lineFragmentShader);
    lineShaderProgram->link();

    // Create VBO
    vbo = new QOpenGLBuffer(QOpenGLBuffer::VertexBuffer);
    vbo->create();
}

void GLViewport::renderMeshModern() {
    if (currentMesh.getTriangleCount() == 0 || !shaderProgram) return;

    // Always update mesh data for now to ensure it's fresh
    updateMeshData();

    if (vertexData.empty()) return;

    // Upload data to VBO
    vbo->bind();
    vbo->allocate(vertexData.data(), vertexData.size() * sizeof(float));
    vbo->release();

    // Calculate actual vertex count from data (may be less than mesh triangle count if we skipped degenerate triangles)
    vertexCount = vertexData.size() / 9;  // 9 floats per vertex (pos + normal + color)
    meshNeedsUpdate = false;

    // Use shader program
    shaderProgram->bind();

    // Calculate MVP matrix
    QMatrix4x4 projection, view, model;

    // Projection with dynamic near/far based on zoom for better depth precision
    // Use a tighter near/far ratio to reduce z-fighting
    float nearPlane = zoom * 0.1f;  // Near plane scales with zoom
    float farPlane = zoom * 10.0f;   // Far plane scales with zoom
    projection.perspective(45.0f, (float)width() / (float)height(), nearPlane, farPlane);

    // View
    view.translate(0.0f, 0.0f, -zoom);
    view.rotate(rotationX, 1.0f, 0.0f, 0.0f);
    view.rotate(rotationY, 0.0f, 1.0f, 0.0f);

    // MVP
    QMatrix4x4 mvp = projection * view * model;
    QMatrix4x4 modelView = view * model;
    QMatrix3x3 normalMatrix = modelView.normalMatrix();

    // Set uniforms
    shaderProgram->setUniformValue("uMVP", mvp);
    shaderProgram->setUniformValue("uModelView", modelView);
    shaderProgram->setUniformValue("uNormalMatrix", normalMatrix);
    shaderProgram->setUniformValue("uLightingMode", (int)currentLightingMode);

    // Bind VBO and set attributes
    vbo->bind();

    int stride = 9 * sizeof(float);
    int posLoc = shaderProgram->attributeLocation("aPosition");
    int normLoc = shaderProgram->attributeLocation("aNormal");
    int colorLoc = shaderProgram->attributeLocation("aColor");

    shaderProgram->enableAttributeArray(posLoc);
    shaderProgram->setAttributeBuffer(posLoc, GL_FLOAT, 0, 3, stride);

    shaderProgram->enableAttributeArray(normLoc);
    shaderProgram->setAttributeBuffer(normLoc, GL_FLOAT, 3 * sizeof(float), 3, stride);

    shaderProgram->enableAttributeArray(colorLoc);
    shaderProgram->setAttributeBuffer(colorLoc, GL_FLOAT, 6 * sizeof(float), 3, stride);

    // Draw based on render mode
    switch (renderMode) {
        case POINTS:
            glDrawArrays(GL_POINTS, 0, vertexCount);
            break;

        case WIREFRAME:
            // Draw as wireframe - render each triangle as a line loop
            for (int i = 0; i < vertexCount; i += 3) {
                glDrawArrays(GL_LINE_LOOP, i, 3);
            }
            break;

        case FLAT:
        case SOLID:
        case SMOOTH_SHADED:
            // Draw as solid triangles
            glDrawArrays(GL_TRIANGLES, 0, vertexCount);
            break;

        case SOLID_WIREFRAME:
        case FLAT_LINES:
        {
            // Draw solid first
            glDrawArrays(GL_TRIANGLES, 0, vertexCount);

            // Draw wireframe on top using line shader
            shaderProgram->disableAttributeArray(posLoc);
            shaderProgram->disableAttributeArray(normLoc);
            shaderProgram->disableAttributeArray(colorLoc);
            shaderProgram->release();

            // Switch to line shader for wireframe overlay
            lineShaderProgram->bind();
            lineShaderProgram->setUniformValue("uMVP", mvp);

            // Create wireframe data with dark gray color
            int linePosLoc = lineShaderProgram->attributeLocation("aPosition");
            int lineColorLoc = lineShaderProgram->attributeLocation("aColor");

            lineShaderProgram->enableAttributeArray(linePosLoc);
            lineShaderProgram->setAttributeBuffer(linePosLoc, GL_FLOAT, 0, 3, stride);

            // For wireframe overlay, we need to skip the normal and use color offset
            // But we want uniform dark color, so we'll draw lines in a simpler way
            for (int i = 0; i < vertexCount; i += 3) {
                glDrawArrays(GL_LINE_LOOP, i, 3);
            }

            lineShaderProgram->disableAttributeArray(linePosLoc);
            lineShaderProgram->release();

            // Re-bind main shader for cleanup
            shaderProgram->bind();
            break;
        }

        case HIDDEN_LINE:
        {
            // Draw triangles first for hidden line removal
            glDrawArrays(GL_TRIANGLES, 0, vertexCount);

            // Draw black wireframe on top
            shaderProgram->disableAttributeArray(posLoc);
            shaderProgram->disableAttributeArray(normLoc);
            shaderProgram->disableAttributeArray(colorLoc);
            shaderProgram->release();

            // Use line shader
            lineShaderProgram->bind();
            lineShaderProgram->setUniformValue("uMVP", mvp);

            int hiddenPosLoc = lineShaderProgram->attributeLocation("aPosition");
            lineShaderProgram->enableAttributeArray(hiddenPosLoc);
            lineShaderProgram->setAttributeBuffer(hiddenPosLoc, GL_FLOAT, 0, 3, stride);

            for (int i = 0; i < vertexCount; i += 3) {
                glDrawArrays(GL_LINE_LOOP, i, 3);
            }

            lineShaderProgram->disableAttributeArray(hiddenPosLoc);
            lineShaderProgram->release();

            // Re-bind main shader for cleanup
            shaderProgram->bind();
            break;
        }

        case VERTICES:
            // Draw mesh solid first (faint)
            glEnable(GL_BLEND);
            glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
            glDrawArrays(GL_TRIANGLES, 0, vertexCount);
            glDisable(GL_BLEND);

            // Draw vertices as points
            glDrawArrays(GL_POINTS, 0, vertexCount);
            break;

        case X_RAY:
            // Enable transparency for x-ray mode
            glEnable(GL_BLEND);
            glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
            glDepthMask(GL_FALSE);
            glDrawArrays(GL_TRIANGLES, 0, vertexCount);
            glDepthMask(GL_TRUE);
            glDisable(GL_BLEND);
            break;
    }

    // Check if wireframe checkbox is enabled (independent of render mode)
    if (wireframe && renderMode != WIREFRAME && renderMode != SOLID_WIREFRAME &&
        renderMode != FLAT_LINES && renderMode != HIDDEN_LINE) {
        // Draw wireframe overlay using line loop
        for (int i = 0; i < vertexCount; i += 3) {
            glDrawArrays(GL_LINE_LOOP, i, 3);
        }
    }

    // Cleanup
    shaderProgram->disableAttributeArray(posLoc);
    shaderProgram->disableAttributeArray(normLoc);
    shaderProgram->disableAttributeArray(colorLoc);
    vbo->release();
    shaderProgram->release();
}
#endif

void GLViewport::frameAll() {
    // If mesh is empty, return
    if (currentMesh.getVertexCount() == 0) {
        return;
    }

    // Calculate bounding box of the mesh
    float minX = 1e10f, minY = 1e10f, minZ = 1e10f;
    float maxX = -1e10f, maxY = -1e10f, maxZ = -1e10f;

    for (size_t i = 0; i < currentMesh.getVertexCount(); i++) {
        const Vertex& v = currentMesh.getVertex(i);
        minX = std::min(minX, v.position.x);
        minY = std::min(minY, v.position.y);
        minZ = std::min(minZ, v.position.z);
        maxX = std::max(maxX, v.position.x);
        maxY = std::max(maxY, v.position.y);
        maxZ = std::max(maxZ, v.position.z);
    }

    // Calculate center and size
    float centerX = (minX + maxX) * 0.5f;
    float centerY = (minY + maxY) * 0.5f;
    float centerZ = (minZ + maxZ) * 0.5f;

    float sizeX = maxX - minX;
    float sizeY = maxY - minY;
    float sizeZ = maxZ - minZ;

    // Calculate diagonal size (maximum dimension)
    float size = std::sqrt(sizeX * sizeX + sizeY * sizeY + sizeZ * sizeZ);

    // If size is very small, use a default zoom
    if (size < 0.001f) {
        zoom = 10.0f;
        return;
    }

    // Calculate zoom to fit the mesh in view
    // Formula: zoom distance should be proportional to mesh size
    // Field of view is 45 degrees, so we need to account for that
    float fov = 45.0f * M_PI / 180.0f;
    zoom = size / (2.0f * std::tan(fov / 2.0f)) * 1.2f; // 1.2 adds a 20% margin

    // Clamp zoom to reasonable values
    zoom = std::max(1.0f, std::min(zoom, 50.0f));
}
