/**
 * @file GLViewport.h
 * @brief OpenGL viewport widget for 3D mesh rendering
 *
 * This header defines the GLViewport class which provides an OpenGL-based
 * 3D visualization widget for displaying parametric meshes. It supports multiple
 * render modes, lighting configurations, camera controls, and real-time mesh updates.
 *
 * Key features:
 * - Multiple render modes (points, wireframe, solid, x-ray, etc.)
 * - Configurable lighting presets (studio, dramatic, soft, rim lighting)
 * - Interactive camera controls (rotation, zoom, pan)
 * - Grid and axis display for spatial reference
 * - Animation time support for vertex animations
 * - Auto-framing to fit mesh in view
 *
 * Copyright © 2025 Linus Suter
 * Released under the GNU/GPL License
 */

#pragma once

// Qt OpenGL Includes
#include <QOpenGLWidget>
#include <QOpenGLFunctions>
#include <QOpenGLShaderProgram>
#include <QOpenGLBuffer>
#include <QOpenGLVertexArrayObject>
#include <QMatrix4x4>
#include <QMouseEvent>
#include <QTimer>
#include <QVector3D>
#include <vector>

// Application Includes
#include "mesh/Mesh.h"

/**
 * @brief OpenGL viewport widget for 3D mesh rendering
 *
 * Provides real-time 3D visualization of parametric meshes with support for
 * multiple render modes, lighting configurations, and interactive camera controls.
 */
class GLViewport : public QOpenGLWidget, protected QOpenGLFunctions {
    Q_OBJECT

public:
    /**
     * @brief Constructor - Creates the OpenGL viewport widget
     * @param parent Parent widget (nullptr for top-level window)
     */
    explicit GLViewport(QWidget *parent = nullptr);

    /**
     * @brief Destructor - Cleans up OpenGL resources
     */
    ~GLViewport();

    /**
     * @brief Enumeration of available render modes
     */
    enum RenderMode {
        POINTS,           ///< Render as point cloud
        WIREFRAME,        ///< Render as wireframe edges only
        FLAT,             ///< Flat shaded polygons
        SOLID,            ///< Solid shaded with lighting
        SOLID_WIREFRAME,  ///< Solid with wireframe overlay
        HIDDEN_LINE,      ///< Wireframe with hidden line removal
        VERTICES,         ///< Display vertices as points
        SMOOTH_SHADED,    ///< Smooth shaded with interpolated normals
        FLAT_LINES,       ///< Flat shaded with edge outlines
        X_RAY             ///< Translucent x-ray mode
    };

    /**
     * @brief Enumeration of lighting presets
     */
    enum LightingMode {
        LIGHTING_DEFAULT,   ///< Standard 3-point lighting
        LIGHTING_STUDIO,    ///< Studio lighting setup
        LIGHTING_BRIGHT,    ///< High intensity lighting
        LIGHTING_DRAMATIC,  ///< High contrast dramatic lighting
        LIGHTING_SOFT,      ///< Soft ambient lighting
        LIGHTING_RIM        ///< Rim/edge lighting
    };

    void clearMesh();
    
    /**
     * @brief Sets the mesh to display in the viewport
     * @param mesh The mesh to render
     */
    void setMesh(const Mesh& mesh);

    /**
     * @brief Enables or disables wireframe rendering mode
     * @param enabled True to enable wireframe, false otherwise
     */
    void setWireframeMode(bool enabled) { wireframe = enabled; update(); }

    /**
     * @brief Sets the current render mode
     * @param mode The render mode to use
     */
    void setRenderMode(RenderMode mode) { renderMode = mode; update(); }

    /**
     * @brief Sets the lighting mode/preset
     * @param mode The lighting mode to use
     */
    void setLightingMode(LightingMode mode);

    /**
     * @brief Sets the animation time for vertex animations
     * @param time Current animation time in seconds
     */
    void setAnimationTime(float time) { animTime = time; update(); }

    /**
     * @brief Sets the camera zoom level
     * @param z Zoom factor (higher = closer)
     */
    void setZoom(float z) { zoom = z; update(); }

    /**
     * @brief Resets camera to default view position and orientation
     */
    void resetView();
    void resetAll();
    /**
     * @brief Sets camera to front orthographic view
     */
    void setFrontView();

    /**
     * @brief Sets camera to top orthographic view
     */
    void setTopView();

    /**
     * @brief Sets camera to side orthographic view
     */
    void setSideView();

    /**
     * @brief Toggles display of reference grid
     * @param show True to show grid, false to hide
     */
    void setShowGrid(bool show) { showGrid = show; update(); }

    /**
     * @brief Toggles display of coordinate axes
     * @param show True to show axes, false to hide
     */
    void setShowAxis(bool show) { showAxis = show; update(); }

    /**
     * @brief Enables or disables back-face culling
     * @param enabled True to cull back faces, false to render both sides
     */
    void setBackFaceCulling(bool enabled) { backFaceCulling = enabled; update(); }

    /**
     * @brief Enables automatic framing of mesh on load
     * @param enabled True to auto-frame, false otherwise
     */
    void setAutoFrameAll(bool enabled) { autoFrameAll = enabled; }

    /**
     * @brief Frames the camera to fit the entire mesh in view
     */
    void frameAll();

protected:
    // ===== Qt OpenGL Virtual Functions =====

    /**
     * @brief Initializes OpenGL state and resources
     *
     * Called once when the OpenGL context is first made current.
     * Sets up OpenGL capabilities, clears color, depth testing, etc.
     */
    void initializeGL() override;

    /**
     * @brief Renders the current frame
     *
     * Called whenever the widget needs to be repainted.
     * Handles all mesh rendering, grid, axes, and lighting.
     */
    void paintGL() override;

    /**
     * @brief Handles viewport resize events
     * @param w New width in pixels
     * @param h New height in pixels
     *
     * Updates the projection matrix to maintain correct aspect ratio.
     */
    void resizeGL(int w, int h) override;

    // ===== Mouse Event Handlers =====

    /**
     * @brief Handles mouse button press events
     * @param event Mouse event containing button and position data
     *
     * Initiates camera rotation or pan operations.
     */
    void mousePressEvent(QMouseEvent *event) override;

    /**
     * @brief Handles mouse move events for camera control
     * @param event Mouse event containing position data
     *
     * Updates camera rotation/pan based on mouse drag.
     */
    void mouseMoveEvent(QMouseEvent *event) override;

    /**
     * @brief Handles mouse wheel events for zoom control
     * @param event Wheel event containing scroll delta
     *
     * Adjusts camera zoom level based on wheel movement.
     */
    void wheelEvent(QWheelEvent *event) override;

private:
    // ===== Mesh Data =====

    Mesh currentMesh;                  ///< Currently displayed mesh
    bool meshNeedsUpdate;              ///< Flag indicating mesh data needs GPU update
    std::vector<float> vertexData;     ///< Packed vertex data for OpenGL rendering

    // ===== Camera Transformation Matrices =====

    QMatrix4x4 projection;             ///< Projection matrix (perspective/orthographic)
    QMatrix4x4 view;                   ///< View matrix (camera position and orientation)
    QMatrix4x4 model;                  ///< Model matrix (mesh transformation)

    // ===== Camera State =====

    QPoint lastMousePos;               ///< Last recorded mouse position for drag operations
    float rotationX;                   ///< Camera rotation around X-axis (in degrees)
    float rotationY;                   ///< Camera rotation around Y-axis (in degrees)
    float zoom;                        ///< Camera zoom/distance from origin

    // ===== Rendering State =====

    bool wireframe;                    ///< Wireframe mode enabled (deprecated, use renderMode)
    RenderMode renderMode;             ///< Current render mode (points, wireframe, solid, etc.)
    LightingMode currentLightingMode;  ///< Current lighting preset
    float animTime;                    ///< Current animation time for vertex animations

    // ===== Display Options =====

    bool showGrid;                     ///< Display reference grid on ground plane
    bool showAxis;                     ///< Display coordinate axes (X=red, Y=green, Z=blue)
    bool backFaceCulling;              ///< Enable back-face culling for performance
    bool autoFrameAll;                 ///< Automatically frame mesh when loaded

#ifdef __EMSCRIPTEN__
    // ===== WebGL/Shader-based Rendering =====
    QOpenGLShaderProgram *shaderProgram;     ///< Shader program for mesh rendering
    QOpenGLShaderProgram *lineShaderProgram; ///< Shader program for line rendering (grid/axis)
    QOpenGLBuffer *vbo;                      ///< Vertex buffer object
    int vertexCount;                         ///< Number of vertices to render

    /** @brief Initializes shaders for WebGL rendering */
    void initializeShaders();

    /** @brief Renders mesh using shader-based approach */
    void renderMeshModern();
#endif

    // ===== Private Helper Methods =====

    /**
     * @brief Renders the current mesh with active render mode
     *
     * Applies the selected render mode and draws the mesh geometry.
     */
    void renderMesh();

    /**
     * @brief Updates mesh vertex data for GPU rendering
     *
     * Packs vertex positions, normals, and colors into OpenGL buffer format.
     */
    void updateMeshData();

    /**
     * @brief Renders the reference grid on the ground plane
     */
    void renderGrid();

    /**
     * @brief Renders the coordinate axes (X=red, Y=green, Z=blue)
     */
    void renderAxis();

    /**
     * @brief Applies the current lighting configuration
     *
     * Sets up OpenGL lighting parameters based on selected lighting mode.
     */
    void applyLighting();
};
