#include "MainWindow.h"
#include "export/FBXExporter.h"
#include "export/MeshExporter.h"
#include "algorithms/AlgorithmDescriptions.h"
#include <QApplication>
#include <QMenuBar>
#include <QFileDialog>
#include <QMessageBox>
#include <QSplitter>
#include <QDockWidget>
#include <QTabWidget>
#include <QDialog>
#include <QImage>
#include <QFileInfo>
#include <QInputDialog>
#include <algorithm>
#include <random>

#ifdef __EMSCRIPTEN__
#include <emscripten.h>
#endif

MainWindow::MainWindow(QWidget *parent)
    : QMainWindow(parent), currentAlgorithm(AlgorithmLibrary::PARAM_SPIRAL_SPHERE),
      animationTime(0.0f) {

    setWindowTitle("Mesher • Parametric Modeler");
    resize(1280, 800);

    createUI();
    createMenuBar();

    animationTimer = new QTimer(this);
    connect(animationTimer, &QTimer::timeout, this, &MainWindow::onAnimationTimerUpdate);

    onAlgorithmChanged(0);
}

MainWindow::~MainWindow() {
}

void MainWindow::keyPressEvent(QKeyEvent *event) {
    // Exit fullscreen when Escape is pressed
    if (event->key() == Qt::Key_Escape && isFullScreen()) {
        showNormal();
    }
    // Add to stack when + key is pressed and an algorithm is selected
    else if ((event->key() == Qt::Key_Plus || event->key() == Qt::Key_Equal) &&
             algorithmList->currentItem() != nullptr) {
        onAddToStack();
    }
    else {
        QMainWindow::keyPressEvent(event);
    }
}

bool MainWindow::eventFilter(QObject *obj, QEvent *event) {
    if (obj == algorithmList && event->type() == QEvent::KeyPress) {
        QKeyEvent *keyEvent = static_cast<QKeyEvent*>(event);
        if ((keyEvent->key() == Qt::Key_Plus || keyEvent->key() == Qt::Key_Equal) &&
            algorithmList->currentItem() != nullptr) {
            onAddToStack();
            return true;
        }
    }
    else if (obj == stackList && event->type() == QEvent::KeyPress) {
        QKeyEvent *keyEvent = static_cast<QKeyEvent*>(event);
        if ((keyEvent->key() == Qt::Key_Minus || keyEvent->key() == Qt::Key_Underscore || keyEvent->key() == Qt::Key_Delete) &&
            stackList->currentItem() != nullptr) {
            onRemoveFromStack();
            return true;
        }
        else if (keyEvent->key() == Qt::Key_F2 && stackList->currentItem() != nullptr) {
            onEditStackItem(stackList->currentItem());
            return true;
        }
    }
    return QMainWindow::eventFilter(obj, event);
}

void MainWindow::createUI() {
    // Central widget with three-column splitter
    QSplitter *mainSplitter = new QSplitter(Qt::Horizontal, this);

    // LEFT COLUMN - Algorithm List
    QWidget *algorithmPanel = new QWidget();
    QVBoxLayout *algorithmLayout = new QVBoxLayout(algorithmPanel);
    algorithmLayout->setContentsMargins(4, 4, 4, 4);
    algorithmLayout->setSpacing(4);

    // Algorithm list title
    QLabel *algoTitleLabel = new QLabel("Algorithms");
    algoTitleLabel->setStyleSheet("QLabel { font-size: 14px; padding: 4px; }");
    algorithmLayout->addWidget(algoTitleLabel);

    // Search box with auto-completer
    algorithmSearchBox = new QLineEdit();
    algorithmSearchBox->setPlaceholderText("🔍 Search algorithms...");
    algorithmSearchBox->setClearButtonEnabled(true);
    algorithmSearchBox->setStyleSheet(
        "QLineEdit { "
        "    background-color: #2a2a2a; "
        "    color: #fff; "
        "    border: 1px solid #444; "
        "    border-radius: 4px; "
        "    padding: 6px; "
        "    font-size: 12px; "
        "} "
        "QLineEdit:focus { "
        "    border: 1px solid #666666; "
        "}"
    );
    algorithmLayout->addWidget(algorithmSearchBox);

    // Algorithm list widget
    algorithmList = new QListWidget();
    algorithmList->setMinimumWidth(240);
    algorithmList->setStyleSheet(
        "QListWidget { background-color: #2a2a2a; color: #fff; border: 1px solid #444; }"
        "QListWidget::item { padding: 3px; border-bottom: 1px solid #333; }"
        "QListWidget::item:selected { background-color: #666666; color: white; }"
        "QListWidget::item:hover { background-color: #3a3a3a; }"
    );
    algorithmList->installEventFilter(this);

    // Populate list with categorized algorithms
    auto algorithms = AlgorithmRegistry::getAllAlgorithms();

    // Sorting disabled - algorithms appear in their original definition order
    // Categories are grouped by the order defined in AlgorithmRegistry::getCategories()
    QString lastCategory = "";

    // Clear the mapping
    indexToAlgorithmMap.clear();

    for (const auto& algo : algorithms) {
        if (algo.category != lastCategory) {
            // Add category separator
            QListWidgetItem *categoryItem = new QListWidgetItem("━━━ " + algo.category + " ━━━");
            categoryItem->setFlags(Qt::NoItemFlags); // Make it non-selectable
            categoryItem->setForeground(QColor("#888"));
            categoryItem->setBackground(QColor("#1a1a1a"));
            QFont categoryFont = categoryItem->font();
            categoryFont.setBold(true);
            categoryFont.setPointSize(10);
            categoryItem->setFont(categoryFont);
            algorithmList->addItem(categoryItem);
            lastCategory = algo.category;
        }

        // Add algorithm with icon based on category
        QString icon = "▸ ";
        if (algo.category == "PRIMITIVES") icon = "◼ ";
        else if (algo.category == "PLATONIC SOLIDS") icon = "🔷 ";
        else if (algo.category == "PARAMETRIC SURFACES") icon = "◆ ";
        else if (algo.category == "TOPOLOGY") icon = "∞ ";
        else if (algo.category == "MINIMAL SURFACES") icon = "◈ ";
        else if (algo.category == "ORGANIC") icon = "🌿 ";
        else if (algo.category == "FRACTALS") icon = "❄ ";
        else if (algo.category == "ARCHITECTURAL") icon = "🏛 ";
        else if (algo.category == "MECHANICAL") icon = "⚙ ";
        else if (algo.category == "STRUCTURAL") icon = "⬡ ";
        else if (algo.category == "TESSELLATIONS") icon = "◇ ";
        else if (algo.category == "COMPUTATIONAL") icon = "⚡ ";
        else if (algo.category == "ATTRACTORS") icon = "🌀 ";
        else if (algo.category == "MODIFIERS") icon = "🔧 ";
        else if (algo.category == "POINT CLOUD") icon = "⚫ ";

        QListWidgetItem *item = new QListWidgetItem(icon + algo.name);

        // Set tooltip with detailed description
        QString detailedDesc = AlgorithmDescriptions::getDetailedDescription(algo.name);
        item->setToolTip("<b>" + algo.name + "</b><br>" +
                        "<i>[" + algo.category + "]</i><br><br>" +
                        "<b>Quick Info:</b> " + algo.description + "<br><br>" +
                        "<b>Detailed Description:</b><br>" + detailedDesc);

        algorithmList->addItem(item);

        // Store the mapping from UI index to actual enum value
        indexToAlgorithmMap.push_back(static_cast<AlgorithmLibrary::AlgorithmType>(algo.algorithmIndex));
    }

    // Connect signals
    connect(algorithmList, &QListWidget::currentRowChanged, this, &MainWindow::onAlgorithmChanged);
    connect(algorithmSearchBox, &QLineEdit::textChanged, this, &MainWindow::onAlgorithmSearchChanged);
    connect(algorithmSearchBox, &QLineEdit::returnPressed, this, &MainWindow::onAlgorithmSearchSubmitted);

    algorithmLayout->addWidget(algorithmList);

    // Description label at bottom of algorithm list
    algoDescLabel = new QLabel();
    algoDescLabel->setWordWrap(true);
    algoDescLabel->setStyleSheet("QLabel { padding: 6px; background-color: #2a2a2a; color: #ccc; border-radius: 4px; font-size: 11px; }");
    algoDescLabel->setMinimumHeight(60);
    algoDescLabel->setMaximumHeight(80);
    algorithmLayout->addWidget(algoDescLabel);

    mainSplitter->addWidget(algorithmPanel);

    // MIDDLE COLUMN - Parameters and Controls
    QWidget *leftPanel = new QWidget();
    QVBoxLayout *leftLayout = new QVBoxLayout(leftPanel);
    leftLayout->setContentsMargins(4, 4, 4, 4);
    leftLayout->setSpacing(4);

    // Parameters group
    QGroupBox *paramGroup = new QGroupBox("Parameters");
    paramGroup->setStyleSheet("QGroupBox::title { padding: 4px; }");
    parameterLayout = new QVBoxLayout();
    parameterLayout->setContentsMargins(4, 4, 4, 4);
    parameterLayout->setSpacing(2);

    QScrollArea *scrollArea = new QScrollArea();
    parameterPanel = new QWidget();
    parameterPanel->setLayout(parameterLayout);
    scrollArea->setWidget(parameterPanel);
    scrollArea->setWidgetResizable(true);
    scrollArea->setMinimumWidth(350);
    scrollArea->setFixedHeight(155);

    QVBoxLayout *paramGroupLayout = new QVBoxLayout();
    paramGroupLayout->setContentsMargins(4, 4, 4, 4);
    paramGroupLayout->setSpacing(4);
    paramGroupLayout->addWidget(scrollArea);
    paramGroup->setLayout(paramGroupLayout);
    paramGroup->setFixedHeight(195);
    leftLayout->addWidget(paramGroup, 0); // Stretch factor of 0 since height is fixed

    // Generate button
    generateButton = new QPushButton("Generate Mesh");
    generateButton->setFlat(true);
    generateButton->setEnabled(false); // Disabled until algorithm is selected
    generateButton->setStyleSheet("QPushButton { background-color: #9E9E9E; color: white; font-weight: bold; padding: 6px; }");
    connect(generateButton, &QPushButton::clicked, this, &MainWindow::onGenerateClicked);
    leftLayout->addWidget(generateButton);

    // Algorithm Stack
    stackGroupBox = new QGroupBox("Algorithm Stack");
    QVBoxLayout *stackLayout = new QVBoxLayout();
    stackLayout->setContentsMargins(4, 4, 4, 4);
    stackLayout->setSpacing(4);

    // Stack list
    stackList = new QListWidget();
    stackList->setStyleSheet("QListWidget { background-color: #2a2a2a; color: #fff; border: 1px solid #444; }");
    connect(stackList, &QListWidget::itemDoubleClicked, this, &MainWindow::onEditStackItem);
    stackList->installEventFilter(this);
    stackLayout->addWidget(stackList, 1); // Stretch factor of 1 to expand vertically

    // Stack control buttons - Row 1
    QHBoxLayout *stackButtons1 = new QHBoxLayout();

    addToStackButton = new QPushButton("+ Add to Stack");
    addToStackButton->setFlat(true);
    addToStackButton->setStyleSheet("QPushButton { color: #4CAF50; padding: 4px; }");
    addToStackButton->setEnabled(false); // Disabled until algorithm is selected
    connect(addToStackButton, &QPushButton::clicked, this, &MainWindow::onAddToStack);
    stackButtons1->addWidget(addToStackButton);

    removeStackButton = new QPushButton("- Remove");
    removeStackButton->setFlat(true);
    removeStackButton->setStyleSheet("QPushButton { color: #f44336; padding: 4px; }");
    connect(removeStackButton, &QPushButton::clicked, this, &MainWindow::onRemoveFromStack);
    stackButtons1->addWidget(removeStackButton);

    clearStackButton = new QPushButton("🗑 Clear");
    clearStackButton->setFlat(true);
    clearStackButton->setStyleSheet("QPushButton { color: #9E9E9E; padding: 4px; }");
    clearStackButton->setFixedHeight(removeStackButton->sizeHint().height());
    connect(clearStackButton, &QPushButton::clicked, this, &MainWindow::onClearStack);
    stackButtons1->addWidget(clearStackButton);

    stackLayout->addLayout(stackButtons1);

    // Stack control buttons - Row 2
    QHBoxLayout *stackButtons2 = new QHBoxLayout();

    moveUpButton = new QPushButton("⬆ Up");
    moveUpButton->setFlat(true);
    moveUpButton->setStyleSheet("QPushButton { color: #B3B3B3; }");
    connect(moveUpButton, &QPushButton::clicked, this, &MainWindow::onMoveUp);
    stackButtons2->addWidget(moveUpButton);

    moveDownButton = new QPushButton("⬇ Down");
    moveDownButton->setFlat(true);
    moveDownButton->setStyleSheet("QPushButton { color: #B3B3B3; }");
    connect(moveDownButton, &QPushButton::clicked, this, &MainWindow::onMoveDown);
    stackButtons2->addWidget(moveDownButton);

    shuffleButton = new QPushButton("🔀 Shuffle");
    shuffleButton->setFlat(true);
    shuffleButton->setStyleSheet("QPushButton { color: #B3B3B3; }");
    connect(shuffleButton, &QPushButton::clicked, this, &MainWindow::onShuffleStack);
    stackButtons2->addWidget(shuffleButton);

    stackLayout->addLayout(stackButtons2);

    // Morph percentage slider
    QHBoxLayout *morphLayout = new QHBoxLayout();
    QLabel *morphLabel = new QLabel("Morph%:");
    morphLabel->setMinimumWidth(60);
    morphLabel->setStyleSheet("padding-left: 1px;");
    morphLayout->addWidget(morphLabel);

    morphSlider = new QSlider(Qt::Horizontal);
    morphSlider->setRange(0, 100);
    morphSlider->setValue(100);
    morphLayout->addWidget(morphSlider);

    QLabel *morphValueLabel = new QLabel("100");
    morphValueLabel->setMinimumWidth(30);
    morphValueLabel->setAlignment(Qt::AlignRight | Qt::AlignVCenter);
    connect(morphSlider, &QSlider::valueChanged, [morphValueLabel](int value) {
        morphValueLabel->setText(QString::number(value));
    });
    morphLayout->addWidget(morphValueLabel);

    // Auto-generate from stack when morph slider is released (mouse-up event)
    connect(morphSlider, &QSlider::sliderReleased, this, &MainWindow::onGenerateFromStack);

    stackLayout->addLayout(morphLayout);

    // Art Pattern Selector
    QLabel *artPatternLabel = new QLabel("Generative Art Pattern:");
    artPatternLabel->setStyleSheet("QLabel { font-weight: bold; margin-top: 4px; }");
    artPatternLabel->setVisible(false); // Hide the label
    stackLayout->addWidget(artPatternLabel);

    artPatternCombo = new QComboBox();
    artPatternCombo->setStyleSheet("QComboBox { background-color: #2a2a2a; color: white; padding: 4px; border: none; }");
    artPatternCombo->addItem("Generative Art");
    artPatternCombo->addItem("1. Organic Fractals + Deformations");
    artPatternCombo->addItem("2. Minimal Surfaces + Tessellations");
    artPatternCombo->addItem("3. Complex Topology + Fractals");
    artPatternCombo->addItem("4. Computational Geometry Art");
    artPatternCombo->addItem("5. Architectural Patterns");
    artPatternCombo->addItem("6. Organic Coral Growth");
    artPatternCombo->addItem("7. Lorenz Attractor Art");
    artPatternCombo->addItem("8. Structural Lattices");
    artPatternCombo->addItem("9. Reaction Diffusion Patterns");
    artPatternCombo->addItem("10. Gyroid Combinations");
    artPatternCombo->addItem("11. Koch Snowflake Variations");
    artPatternCombo->addItem("12. L-System Organic");
    artPatternCombo->addItem("13. Hyperbolic Art");
    artPatternCombo->addItem("14. Knot Variations");
    artPatternCombo->addItem("15. Platonic Transformations");
    stackLayout->addWidget(artPatternCombo);

    // Button to load selected art pattern into stack
    QPushButton *loadArtPatternButton = new QPushButton("🎨 Load Art Pattern");
    loadArtPatternButton->setStyleSheet("QPushButton { background-color: #673AB7; color: white; font-weight: bold; padding: 6px; font-size: 13px; }");
    loadArtPatternButton->setVisible(false); // Hide the button

    // Lambda function for loading and executing art pattern (shared by button and dropdown)
    auto loadAndExecutePattern = [this]() {
        // Clear existing stack
        algorithmStack.clear();

        // Random number generator
        std::random_device rd;
        std::mt19937 gen(rd());

        // Define aesthetically pleasing algorithm combinations for generative art
        std::vector<std::vector<AlgorithmLibrary::AlgorithmType>> artPatterns = {
            // Pattern 1: Organic fractals with deformations
            {
                AlgorithmLibrary::FRAC_TREE,
                AlgorithmLibrary::MOD_TWIST_DEFORM,
                AlgorithmLibrary::MOD_NOISE,
                AlgorithmLibrary::MOD_SPHERIFY
            },
            // Pattern 2: Minimal surfaces with tessellations
            {
                AlgorithmLibrary::MIN_GYROID,
                AlgorithmLibrary::TESS_PENROSE,
                AlgorithmLibrary::MOD_ARRAY,
                AlgorithmLibrary::MOD_SMOOTH
            },
            // Pattern 3: Complex topology with fractals
            {
                AlgorithmLibrary::TOPO_KLEIN_BOTTLE,
                AlgorithmLibrary::FRAC_MANDELBROT_3D,
                AlgorithmLibrary::MOD_WAVE_DEFORM,
                AlgorithmLibrary::MOD_INFLATE
            },
            // Pattern 4: Computational geometry art
            {
                AlgorithmLibrary::COMP_VORONOI_SURFACE,
                AlgorithmLibrary::MOD_PERLIN_NOISE,
                AlgorithmLibrary::MOD_EXTRUDE,
                AlgorithmLibrary::MOD_MIRROR
            },
            // Pattern 5: Architectural patterns
            {
                AlgorithmLibrary::ARCH_VORONOI_LATTICE,
                AlgorithmLibrary::TESS_ISLAMIC,
                AlgorithmLibrary::MOD_TWIST_DEFORM,
                AlgorithmLibrary::MOD_SHELL
            },
            // Pattern 6: Organic coral growth
            {
                AlgorithmLibrary::ORG_CORAL,
                AlgorithmLibrary::MOD_SUBDIVIDE,
                AlgorithmLibrary::MOD_NOISE,
                AlgorithmLibrary::MOD_SMOOTH
            },
            // Pattern 7: Lorenz attractor art
            {
                AlgorithmLibrary::ATTR_LORENZ,
                AlgorithmLibrary::MOD_WAVE_DEFORM,
                AlgorithmLibrary::MOD_INFLATE,
                AlgorithmLibrary::MOD_ARRAY
            },
            // Pattern 8: Structural lattices
            {
                AlgorithmLibrary::STRUCT_DIAMOND_LATTICE,
                AlgorithmLibrary::MOD_BEND_DEFORM,
                AlgorithmLibrary::MOD_PERLIN_NOISE,
                AlgorithmLibrary::MOD_SPHERIFY
            },
            // Pattern 9: Reaction diffusion patterns
            {
                AlgorithmLibrary::COMP_REACTION_DIFFUSION,
                AlgorithmLibrary::MOD_EXTRUDE,
                AlgorithmLibrary::MOD_SMOOTH,
                AlgorithmLibrary::MOD_TWIST_DEFORM
            },
            // Pattern 10: Gyroid combinations
            {
                AlgorithmLibrary::MIN_GYROID_MC,
                AlgorithmLibrary::MOD_LATTICE_DEFORM,
                AlgorithmLibrary::MOD_ARRAY,
                AlgorithmLibrary::MOD_INFLATE
            },
            // Pattern 11: Koch snowflake variations
            {
                AlgorithmLibrary::FRAC_KOCH_SNOWFLAKE,
                AlgorithmLibrary::MOD_EXTRUDE,
                AlgorithmLibrary::MOD_TWIST_DEFORM,
                AlgorithmLibrary::MOD_MIRROR
            },
            // Pattern 12: L-system organic
            {
                AlgorithmLibrary::FRAC_L_SYSTEM,
                AlgorithmLibrary::MOD_WAVE_DEFORM,
                AlgorithmLibrary::MOD_SMOOTH,
                AlgorithmLibrary::MOD_SPHERIFY
            },
            // Pattern 13: Hyperbolic art
            {
                AlgorithmLibrary::MIN_HYPERBOLIC_PARABOLOID,
                AlgorithmLibrary::TESS_TRUCHET,
                AlgorithmLibrary::MOD_NOISE,
                AlgorithmLibrary::MOD_ARRAY
            },
            // Pattern 14: Knot variations
            {
                AlgorithmLibrary::TOPO_TREFOIL_KNOT,
                AlgorithmLibrary::MOD_INFLATE,
                AlgorithmLibrary::MOD_TWIST_DEFORM,
                AlgorithmLibrary::MOD_PERLIN_NOISE
            },
            // Pattern 15: Platonic transformations
            {
                AlgorithmLibrary::PLATO_GEODESIC_SPHERE,
                AlgorithmLibrary::MOD_LATTICE_DEFORM,
                AlgorithmLibrary::MOD_NOISE,
                AlgorithmLibrary::MOD_ARRAY
            }
        };

        // Get the selected pattern from dropdown (index 0 is the default "Generative Art" text)
        int selectedPattern = artPatternCombo->currentIndex() - 1; // Subtract 1 because first item is placeholder
        if (selectedPattern < 0 || selectedPattern >= (int)artPatterns.size()) {
            return; // Don't execute if "Generative Art" placeholder is selected
        }

        // Build the stack from the selected pattern
        for (AlgorithmLibrary::AlgorithmType algoType : artPatterns[selectedPattern]) {
            const auto& algo = library.getAlgorithm(algoType);

            // Create stack entry with randomized parameters
            StackEntry entry;
            entry.algorithmType = algoType;
            entry.name = QString::fromStdString(algo.name);
            entry.parameters = algo.parameters;

            // Randomize each parameter within its range
            for (auto& param : entry.parameters) {
                std::uniform_real_distribution<float> paramDist(param.minValue, param.maxValue);
                param.value = paramDist(gen);
            }

            algorithmStack.push_back(entry);
        }

        // Update UI
        updateStackUI();

        // Automatically generate the mesh
        onGenerateFromStack();
    };

    // Connect button to the lambda
    connect(loadArtPatternButton, &QPushButton::clicked, loadAndExecutePattern);

    // Connect dropdown selection change to automatically load and execute pattern
    connect(artPatternCombo, QOverload<int>::of(&QComboBox::currentIndexChanged), [loadAndExecutePattern](int index) {
        if (index > 0) { // Skip index 0 which is the placeholder "Generative Art"
            loadAndExecutePattern();
        }
    });

    stackLayout->addWidget(loadArtPatternButton);

    // Large action buttons
    generateStackButton = new QPushButton("⚡ Generate from Stack");
    generateStackButton->setStyleSheet("QPushButton { background-color: #2E2E2E; color: white; font-weight: bold; padding: 6px; font-size: 13px; }");
    generateStackButton->setEnabled(false); // Disabled until stack has items
    connect(generateStackButton, &QPushButton::clicked, this, &MainWindow::onGenerateFromStack);
    stackLayout->addWidget(generateStackButton);

    randomizeButton = new QPushButton("🎲 Randomize All Settings");
    randomizeButton->setStyleSheet("QPushButton { background-color: #242424; color: white; font-weight: normal; padding: 6px; font-size: 13px; margin-bottom: 2px; }");
    connect(randomizeButton, &QPushButton::clicked, this, &MainWindow::onRandomizeSettings);
    stackLayout->addWidget(randomizeButton);

    stackGroupBox->setLayout(stackLayout);
    leftLayout->addWidget(stackGroupBox);

    mainSplitter->addWidget(leftPanel);

    // RIGHT COLUMN - 3D viewport with controls beneath
    QWidget *rightPanel = new QWidget();
    QVBoxLayout *rightLayout = new QVBoxLayout(rightPanel);
    rightLayout->setContentsMargins(0, 0, 0, 0);
    rightLayout->setSpacing(4);

    viewport = new GLViewport();
    viewport->setMinimumSize(400, 400);
    rightLayout->addWidget(viewport);

    // 3D View Controls beneath viewport
    QGroupBox *controlsGroup = new QGroupBox("");
    controlsGroup->setFixedHeight(97);
    QVBoxLayout *controlsLayout = new QVBoxLayout();
    controlsLayout->setContentsMargins(1, 1, 1, 1);
    controlsLayout->setSpacing(1);

    // Top row: Render/Lighting controls on left, View buttons on right
    QHBoxLayout *topRowLayout = new QHBoxLayout();
    topRowLayout->setSpacing(1);
    topRowLayout->setContentsMargins(0, 0, 0, 0);

    // Left quad: Render Mode and Lighting dropdowns (2x2 grid)
    QVBoxLayout *renderLightingLayout = new QVBoxLayout();
    renderLightingLayout->setSpacing(1);
    renderLightingLayout->setContentsMargins(0, 0, 0, 0);

    // Render mode selection
    QHBoxLayout *renderModeLayout = new QHBoxLayout();
    renderModeLayout->setSpacing(1);
    renderModeLayout->setContentsMargins(0, 0, 0, 0);
    QLabel *renderModeLabel = new QLabel("Render Mode:");
    renderModeCombo = new QComboBox();
    renderModeCombo->setStyleSheet("QComboBox { border: none; background: transparent; }");
    renderModeCombo->addItem("Points");
    renderModeCombo->addItem("Wireframe");
    renderModeCombo->addItem("Flat");
    renderModeCombo->addItem("Solid");
    renderModeCombo->addItem("Solid+Wireframe");
    renderModeCombo->addItem("Hidden Line");
    renderModeCombo->addItem("Vertices");
    renderModeCombo->addItem("Smooth Shaded");
    renderModeCombo->addItem("Flat+Lines");
    renderModeCombo->addItem("X-Ray");
    renderModeCombo->setCurrentIndex(3); // Default to Solid
    connect(renderModeCombo, QOverload<int>::of(&QComboBox::currentIndexChanged),
            this, &MainWindow::onRenderModeChanged);
    renderModeLayout->addWidget(renderModeLabel);
    renderModeLayout->addWidget(renderModeCombo);
    renderLightingLayout->addLayout(renderModeLayout);

    // Lighting mode selection
    QHBoxLayout *lightingModeLayout = new QHBoxLayout();
    lightingModeLayout->setSpacing(1);
    lightingModeLayout->setContentsMargins(0, 0, 0, 0);
    QLabel *lightingModeLabel = new QLabel("Lighting:");
    QComboBox *lightingModeCombo = new QComboBox();
    lightingModeCombo->setStyleSheet("QComboBox { border: none; background: transparent; }");
    lightingModeCombo->addItem("Default");
    lightingModeCombo->addItem("Studio");
    lightingModeCombo->addItem("Bright");
    lightingModeCombo->addItem("Dramatic");
    lightingModeCombo->addItem("Soft");
    lightingModeCombo->addItem("Rim");
    lightingModeCombo->setCurrentIndex(1); // Default to Studio
    connect(lightingModeCombo, QOverload<int>::of(&QComboBox::currentIndexChanged),
            [this](int index) {
                viewport->setLightingMode(static_cast<GLViewport::LightingMode>(index));
            });
    lightingModeLayout->addWidget(lightingModeLabel);
    lightingModeLayout->addWidget(lightingModeCombo);
    renderLightingLayout->addLayout(lightingModeLayout);

    topRowLayout->addLayout(renderLightingLayout);

    // Right quad: View buttons (2x2 grid)
    QGridLayout *viewButtonsGrid = new QGridLayout();
    viewButtonsGrid->setSpacing(1);
    viewButtonsGrid->setContentsMargins(0, 0, 0, 0);

    QPushButton *frontViewBtn = new QPushButton("Front");
    frontViewBtn->setFlat(true);
    connect(frontViewBtn, &QPushButton::clicked, viewport, &GLViewport::setFrontView);
    viewButtonsGrid->addWidget(frontViewBtn, 0, 0);

    QPushButton *topViewBtn = new QPushButton("Top");
    topViewBtn->setFlat(true);
    connect(topViewBtn, &QPushButton::clicked, viewport, &GLViewport::setTopView);
    viewButtonsGrid->addWidget(topViewBtn, 0, 1);

    QPushButton *sideViewBtn = new QPushButton("Side");
    sideViewBtn->setFlat(true);
    connect(sideViewBtn, &QPushButton::clicked, viewport, &GLViewport::setSideView);
    viewButtonsGrid->addWidget(sideViewBtn, 1, 0);

    QPushButton *resetViewBtn = new QPushButton("Reset View");
    resetViewBtn->setFlat(true);
    resetViewBtn->setStyleSheet("QPushButton { background-color: #607D8B; color: white; padding: 1px; }");
    connect(resetViewBtn, &QPushButton::clicked, viewport, &GLViewport::resetView);
    viewButtonsGrid->addWidget(resetViewBtn, 1, 1);

    topRowLayout->addLayout(viewButtonsGrid);
    controlsLayout->addLayout(topRowLayout);

    // Camera controls
    QHBoxLayout *cameraLayout = new QHBoxLayout();
    cameraLayout->setSpacing(1);
    cameraLayout->setContentsMargins(0, 0, 0, 0);
    QLabel *zoomLabel = new QLabel("Zoom:");
    QSlider *zoomSlider = new QSlider(Qt::Horizontal);
    zoomSlider->setRange(10, 500); // 1.0 to 50.0 (x10)
    zoomSlider->setValue(100); // 10.0
    QLabel *zoomValueLabel = new QLabel("10.0");
    connect(zoomSlider, &QSlider::valueChanged, [this, zoomValueLabel](int value) {
        float zoom = value / 10.0f;
        zoomValueLabel->setText(QString::number(zoom, 'f', 1));
        viewport->setZoom(zoom);
    });
    cameraLayout->addWidget(zoomLabel);
    cameraLayout->addWidget(zoomSlider);
    cameraLayout->addWidget(zoomValueLabel);
    controlsLayout->addLayout(cameraLayout);

    // Checkboxes on same line
    QHBoxLayout *checkboxLayout = new QHBoxLayout();
    checkboxLayout->setSpacing(1);
    checkboxLayout->setContentsMargins(0, 0, 0, 0);
    wireframeCheck = new QCheckBox("Wireframe");
    wireframeCheck->setStyleSheet("QCheckBox { color: #999999; }");
    connect(wireframeCheck, &QCheckBox::toggled, this, &MainWindow::onWireframeToggled);
    checkboxLayout->addWidget(wireframeCheck);

    frameAllCheck = new QCheckBox("FrameAll View");
    frameAllCheck->setStyleSheet("QCheckBox { color: #999999; }");
    frameAllCheck->setChecked(true); // Checked by default
    connect(frameAllCheck, &QCheckBox::toggled, viewport, &GLViewport::setAutoFrameAll);
    connect(frameAllCheck, &QCheckBox::toggled, this, &MainWindow::onFrameAllToggled);
    checkboxLayout->addWidget(frameAllCheck);

    QCheckBox *gridCheck = new QCheckBox("Grid");
    gridCheck->setStyleSheet("QCheckBox { color: #999999; }");
    connect(gridCheck, &QCheckBox::toggled, viewport, &GLViewport::setShowGrid);
    checkboxLayout->addWidget(gridCheck);

    QCheckBox *axisCheck = new QCheckBox("Axis");
    axisCheck->setStyleSheet("QCheckBox { color: #999999; }");
    connect(axisCheck, &QCheckBox::toggled, viewport, &GLViewport::setShowAxis);
    checkboxLayout->addWidget(axisCheck);

    QCheckBox *cullingCheck = new QCheckBox("Back Face Culling");
    cullingCheck->setStyleSheet("QCheckBox { color: #999999; }");
    connect(cullingCheck, &QCheckBox::toggled, viewport, &GLViewport::setBackFaceCulling);
    checkboxLayout->addWidget(cullingCheck);
    controlsLayout->addLayout(checkboxLayout);

    controlsGroup->setLayout(controlsLayout);
    rightLayout->addWidget(controlsGroup);

    // Action Buttons Group - 2x3 grid
    QGroupBox *actionsGroup = new QGroupBox("");
    actionsGroup->setFixedHeight(89);
    QGridLayout *actionsGridLayout = new QGridLayout();
    actionsGridLayout->setContentsMargins(4, 4, 4, 4);
    actionsGridLayout->setSpacing(4);

    // Reset All Settings button
    QPushButton *resetSettingsButton = new QPushButton("Reset All Settings");
    resetSettingsButton->setStyleSheet("QPushButton { background-color: #2E2E2E; color: white; font-weight: bold; padding: 6px; font-size: 13px; }");
    resetSettingsButton->setMinimumHeight(generateStackButton->sizeHint().height());
    connect(resetSettingsButton, &QPushButton::clicked, [this, zoomSlider, gridCheck, axisCheck, cullingCheck, lightingModeCombo]() {
        // 1. Clear all parameter controls (empty Parameters group)
        for (auto slider : parameterSliders) {
            delete slider;
        }
        for (auto label : parameterLabels) {
            delete label;
        }
        parameterSliders.clear();
        parameterLabels.clear();

        QLayoutItem *item;
        while ((item = parameterLayout->takeAt(0)) != nullptr) {
            delete item->widget();
            delete item;
        }

        // 2. Reset morph slider to 100%
        morphSlider->setValue(100);

        // 3. Clear algorithm stack
        algorithmStack.clear();
        updateStackUI();

        // 4. Reset render mode to Solid (index 3)
        renderModeCombo->setCurrentIndex(3);

        // 5. Reset lighting mode to Studio (index 1)
        lightingModeCombo->setCurrentIndex(1);

        // 6. Reset zoom to 10.0 (value 100)
        zoomSlider->setValue(100);

        // 7. Reset all checkboxes to default state
        wireframeCheck->setChecked(false);
        frameAllCheck->setChecked(true); // FrameAll is checked by default
        gridCheck->setChecked(false);
        axisCheck->setChecked(false);
        cullingCheck->setChecked(false);

        // 8. Reset Generative Art dropdown to default "Generative Art" text
        artPatternCombo->setCurrentIndex(0);

        // 9. CRITICAL: Completely clear and reset the current mesh
        currentMesh.clear();  // Clear all vertices and triangles
        currentMesh = Mesh(); // Create a fresh empty mesh
        viewport->resetAll();
        
        // 13. Clear any SDF meshes
        sdfMeshA.clear();
        sdfMeshA = Mesh();
        sdfMeshB.clear();
        sdfMeshB = Mesh();

        // 14. Reset animation timer if running
        if (animationTimer && animationTimer->isActive()) {
            animationTimer->stop();
        }
        animationTime = 0.0f;

        // 15. Reset current algorithm to default
        currentAlgorithm = AlgorithmLibrary::PARAM_SPIRAL_SPHERE;

        // 16. Disable Generate Mesh and Save buttons
        generateButton->setEnabled(false);
        generateButton->setStyleSheet("QPushButton { background-color: #9E9E9E; color: white; font-weight: bold; padding: 6px; }");
        saveButton->setEnabled(false);
        addToStackButton->setEnabled(false);
        addToStackButton->setStyleSheet("QPushButton { color: #4CAF50; padding: 4px; }");
        generateStackButton->setEnabled(false);
        generateStackButton->setStyleSheet("QPushButton { background-color: #2E2E2E; color: white; font-weight: bold; padding: 6px; font-size: 13px; }");

        // 17. Reset stack control buttons
        removeStackButton->setEnabled(true);
        clearStackButton->setEnabled(true);
        moveUpButton->setEnabled(true);
        moveDownButton->setEnabled(true);
        shuffleButton->setEnabled(true);

        // 18. Scroll algorithm list to top and deselect all
        algorithmList->clearSelection();
        algorithmList->setCurrentRow(-1);
        algorithmList->scrollToTop();

        // 19. Clear the "Quick Info" description label at bottom left
        algoDescLabel->clear();
        algoDescLabel->setText("<i>Select an algorithm from the list above</i>");

        // 20. Clear search box
        algorithmSearchBox->clear();

        // 21. Show all algorithms (in case some were hidden by search)
        for (int i = 0; i < algorithmList->count(); i++) {
            algorithmList->item(i)->setHidden(false);
        }

        // 22. Update window title to default
        setWindowTitle("Mesher • Parametric Modeler");

        // 23. Focus on the search box
        algorithmSearchBox->setFocus();
        algorithmSearchBox->selectAll();

        // 24. Process all pending events to ensure clean state
        QApplication::processEvents();
        
        // 25. Final viewport refresh to ensure clean render
        viewport->update();
    });
    actionsGridLayout->addWidget(resetSettingsButton, 0, 0);

    QPushButton *toggleFullscreenButton = new QPushButton("Toggle Fullscreen");
    toggleFullscreenButton->setMinimumHeight(generateStackButton->sizeHint().height());
    
#ifdef __EMSCRIPTEN__
    toggleFullscreenButton->setStyleSheet("QPushButton { background-color: #2E2E2E; color: #808080; padding: 6px; font-size: 13px; }");
    toggleFullscreenButton->setEnabled(false);  // Disable fullscreen toggle button
#else
    toggleFullscreenButton->setStyleSheet("QPushButton { background-color: #2E2E2E; color: #FFFFFF; padding: 6px; font-size: 13px; }");
    toggleFullscreenButton->setEnabled(true);
#endif
    connect(toggleFullscreenButton, &QPushButton::clicked, [this]() {
        if (isFullScreen()) {
            showNormal();
        } else {
            showFullScreen();
        }
    });
    actionsGridLayout->addWidget(toggleFullscreenButton, 0, 1);

    QPushButton *meshFromImageButton = new QPushButton("Mesh from Image");
    meshFromImageButton->setStyleSheet("QPushButton { background-color: #242424; color: white; font-weight: normal; padding: 6px; font-size: 13px; }");
    meshFromImageButton->setMinimumHeight(generateStackButton->sizeHint().height());
    connect(meshFromImageButton, &QPushButton::clicked, [this]() {
#ifdef __EMSCRIPTEN__
        // WebAssembly: Use native browser file picker
        QFileDialog::getOpenFileContent(
            "image/png,image/jpeg,image/bmp,.png,.jpg,.jpeg,.bmp",
            [this](const QString &fileName, const QByteArray &fileContent) {
                if (fileContent.isEmpty()) {
                    return;
                }

                // Load the image from byte array
                QImage image;
                if (!image.loadFromData(fileContent)) {
                    QMessageBox::warning(this, "Error", "Failed to load image: " + fileName);
                    return;
                }

                // Convert to a manageable format
                image = image.convertToFormat(QImage::Format_RGB888);
#else
        // Desktop: Use traditional file dialog
        const QStringList filters({
            "Image Files (*.jpg *.jpeg *.png *.bmp)",
            "JPEG Image • Joint Photographic Experts Group (*.jpg *.jpeg)",
            "PNG Image • Portable Network Graphics (*.png)",
            "BMP Image • Windows Bitmap (*.bmp)",
            "All Files (*)"
        });

        QString filename = QFileDialog::getOpenFileName(
            this,
            "Select Image for Mesh Generation",
            "",
            filters.join(";;"),
            nullptr,
            QFileDialog::DontUseNativeDialog
        );

        if (!filename.isEmpty()) {
            // Load the image
            QImage image(filename);
            if (image.isNull()) {
                QMessageBox::warning(this, "Error", "Failed to load image: " + filename);
                return;
            }

            // Convert to a manageable format
            image = image.convertToFormat(QImage::Format_RGB888);
#endif

            // Determine mesh resolution (limit to reasonable size for performance)
            int width = image.width();
            int height = image.height();
            int maxDim = 200; // Maximum dimension for mesh

            if (width > maxDim || height > maxDim) {
                float scale = std::min(float(maxDim) / width, float(maxDim) / height);
                image = image.scaled(width * scale, height * scale, Qt::KeepAspectRatio, Qt::SmoothTransformation);
                width = image.width();
                height = image.height();
            }

            // Generate mesh from image
            Mesh newMesh;
            newMesh.clear();

            // Create vertices based on image brightness (heightmap) and colors
            for (int y = 0; y < height; ++y) {
                for (int x = 0; x < width; ++x) {
                    QRgb pixel = image.pixel(x, y);

                    // Calculate height from brightness (grayscale)
                    float r = qRed(pixel) / 255.0f;
                    float g = qGreen(pixel) / 255.0f;
                    float b = qBlue(pixel) / 255.0f;
                    float brightness = (r + g + b) / 3.0f;

                    // Position: center the mesh and scale appropriately
                    float px = (x - width * 0.5f) / (width * 0.5f);
                    float py = brightness * 0.5f; // Height from brightness
                    float pz = (y - height * 0.5f) / (height * 0.5f);

                    Vertex v;
                    v.position = Vec3(px, py, pz);
                    v.color = Vec3(r, g, b);
                    v.u = float(x) / (width - 1);
                    v.v = float(y) / (height - 1);

                    newMesh.addVertex(v);
                }
            }

            // Create triangles (two triangles per quad)
            for (int y = 0; y < height - 1; ++y) {
                for (int x = 0; x < width - 1; ++x) {
                    int idx0 = y * width + x;
                    int idx1 = y * width + (x + 1);
                    int idx2 = (y + 1) * width + x;
                    int idx3 = (y + 1) * width + (x + 1);

                    // First triangle
                    newMesh.addTriangle(idx0, idx2, idx1);
                    // Second triangle
                    newMesh.addTriangle(idx1, idx2, idx3);
                }
            }

            // Compute normals for proper lighting
            newMesh.computeNormals();

            // Set the mesh name
#ifdef __EMSCRIPTEN__
            newMesh.setName("Mesh from " + fileName.toStdString());
#else
            QFileInfo fileInfo(filename);
            newMesh.setName("Mesh from " + fileInfo.fileName().toStdString());
#endif

            // Update current mesh and viewport
            currentMesh = newMesh;
            viewport->setMesh(currentMesh);
            saveButton->setEnabled(true);
            updateWindowTitle();
#ifdef __EMSCRIPTEN__
            }
        );
#else
        }
#endif
    });
    actionsGridLayout->addWidget(meshFromImageButton, 0, 2);

    // Bottom row
    saveButton = new QPushButton("Save Mesh to File");
    saveButton->setStyleSheet("QPushButton { background-color: #2E2E2E; color: white; font-weight: bold; padding: 6px; font-size: 13px; }");
    saveButton->setMinimumHeight(generateStackButton->sizeHint().height());
    saveButton->setEnabled(false); // Disabled until mesh is generated
    connect(saveButton, &QPushButton::clicked, this, &MainWindow::onExportClicked);
    actionsGridLayout->addWidget(saveButton, 1, 0);

    QPushButton *aboutButton = new QPushButton("About Mesher");
    aboutButton->setStyleSheet("QPushButton { background-color: #242424; color: white; font-weight: normal; padding: 6px; font-size: 13px; }");
    aboutButton->setMinimumHeight(generateStackButton->sizeHint().height());
    connect(aboutButton, &QPushButton::clicked, [this]() {
        QMessageBox::about(this, "About Mesher",
            "<span style=\"font-size:12pt;\"><b>Mesher</b> • <i>Parametric Modeler</i><br/>"
            "Copyright © 2025 Linus Suter<br/>"
            "<br/>Released under the GNU/GPL License:<br/>"
            "<a href=\"http://www.gnu.org/licenses/gpl.html\">http://www.gnu.org/licenses/gpl.html</a><br/>"
            "<br/>For more information visit:<br/>"
            "<a href=\"https://codewelt.com/mesher\">https://codewelt.com/mesher</a><br/><br/>"
            "<br/>Send me a few bucks if you like this app:"
            "<br/><a href=\"https://www.paypal.com/paypalme/codewelt/11\">https://www.paypal.com/paypalme/codewelt/11</a>"
        );
    });
    actionsGridLayout->addWidget(aboutButton, 1, 1);

    QPushButton *quitButton = new QPushButton("Quit Application");
    
    quitButton->setMinimumHeight(generateStackButton->sizeHint().height());
    
#ifdef __EMSCRIPTEN__
    quitButton->setStyleSheet("QPushButton { background-color: #242424; color: #808080; font-weight: normal; padding: 6px; font-size: 13px; }");
    quitButton->setEnabled(false);  // Disable quit button
#else
    quitButton->setStyleSheet("QPushButton { background-color: #242424; color: #FFFFFF; font-weight: normal; padding: 6px; font-size: 13px; }");
    quitButton->setEnabled(true);
#endif
    
    connect(quitButton, &QPushButton::clicked, this, &QApplication::quit);
    actionsGridLayout->addWidget(quitButton, 1, 2);

    actionsGroup->setLayout(actionsGridLayout);
    rightLayout->addWidget(actionsGroup);

    // Assemble three-column layout
    mainSplitter->addWidget(algorithmPanel);  // LEFT COLUMN
    mainSplitter->addWidget(leftPanel);       // MIDDLE COLUMN (parameters)
    mainSplitter->addWidget(rightPanel);      // RIGHT COLUMN (viewport)

    // Set stretch factors: algorithm list fixed, parameters fixed, viewport stretches
    mainSplitter->setStretchFactor(0, 0);
    mainSplitter->setStretchFactor(1, 0);
    mainSplitter->setStretchFactor(2, 1);

    // Set initial sizes: 250px (list) + 350px (params) + remaining (viewport)
    mainSplitter->setSizes(QList<int>() << 250 << 350 << 800);

    setCentralWidget(mainSplitter);
}

void MainWindow::createMenuBar() {
    // MenuBar removed - all functionality moved to buttons in GUI
}

void MainWindow::setupParameterControls() {
    // Clear existing controls
    for (auto slider : parameterSliders) {
        delete slider;
    }
    for (auto label : parameterLabels) {
        delete label;
    }
    parameterSliders.clear();
    parameterLabels.clear();

    QLayoutItem *item;
    while ((item = parameterLayout->takeAt(0)) != nullptr) {
        delete item->widget();
        delete item;
    }

    // Get algorithm parameters
    const auto& algo = library.getAlgorithm(currentAlgorithm);

    for (size_t i = 0; i < algo.parameters.size(); i++) {
        const auto& param = algo.parameters[i];

        // Create container widget for parameter row
        QWidget *paramRowWidget = new QWidget();
        QHBoxLayout *paramRowLayout = new QHBoxLayout(paramRowWidget);
        paramRowLayout->setContentsMargins(0, 0, 0, 0);
        paramRowLayout->setSpacing(8);

        QLabel *nameLabel = new QLabel(QString::fromStdString(param.name) + ":");
        nameLabel->setStyleSheet("font-weight: bold;");
        nameLabel->setMinimumWidth(95);
        paramRowLayout->addWidget(nameLabel);

        QSlider *slider = new QSlider(Qt::Horizontal);
        slider->setRange(0, 100);
        float normalizedValue = (param.value - param.minValue) / (param.maxValue - param.minValue);
        slider->setValue((int)(normalizedValue * 100));
        paramRowLayout->addWidget(slider);

        QLabel *valueLabel = new QLabel(QString::number(param.value, 'f', 2));
        valueLabel->setMinimumWidth(50);
        valueLabel->setMaximumWidth(50);
        valueLabel->setAlignment(Qt::AlignRight | Qt::AlignVCenter);

        connect(slider, &QSlider::valueChanged, this, [this, i, &algo, valueLabel, slider]() {
            float normalized = slider->value() / 100.0f;
            float value = algo.parameters[i].minValue +
                         normalized * (algo.parameters[i].maxValue - algo.parameters[i].minValue);
            valueLabel->setText(QString::number(value, 'f', 2));
            onParameterChanged();
        });

        paramRowLayout->addWidget(valueLabel);

        parameterLayout->addWidget(paramRowWidget);

        parameterSliders.push_back(slider);
        parameterLabels.push_back(valueLabel);
    }

    parameterLayout->addStretch();
}

void MainWindow::onAlgorithmChanged(int index) {
    // Get algorithm metadata
    auto algorithms = AlgorithmRegistry::getAllAlgorithms();

    // Get current item text
    QListWidgetItem* currentItem = algorithmList->currentItem();
    if (!currentItem) return;

    QString itemText = currentItem->text();

    // Separators are non-selectable, so this check is for safety only
    if (itemText.startsWith("━━━")) {
        return;
    }

    // Find actual algorithm index (accounting for separators)
    int algoCount = 0;
    for (int i = 0; i <= index; i++) {
        QListWidgetItem* item = algorithmList->item(i);
        if (item && !item->text().startsWith("━━━")) {
            if (i == index) {
                // Use the mapping to get the correct enum value
                if (algoCount < (int)indexToAlgorithmMap.size()) {
                    currentAlgorithm = indexToAlgorithmMap[algoCount];

                    // Update description
                    for (const auto& algo : algorithms) {
                        if (algo.algorithmIndex == (int)currentAlgorithm) {
                            algoDescLabel->setText(
                                "<b>[" + algo.category + "]</b><br>" +
                                algo.description
                            );
                            break;
                        }
                    }

                    // Enable Generate Mesh button now that an algorithm is selected
                    generateButton->setEnabled(true);
                    // Enable Add to Stack button now that an algorithm is selected
                    addToStackButton->setEnabled(true);
                }
                break;
            }
            algoCount++;
        }
    }

    setupParameterControls();
    regenerateMesh();
}

void MainWindow::onParameterChanged() {
    // Auto-regenerate on parameter change
    regenerateMesh();
}

void MainWindow::onAlgorithmSearchChanged(const QString& text) {
    // Filter algorithm list based on search text
    for (int i = 0; i < algorithmList->count(); i++) {
        QListWidgetItem* item = algorithmList->item(i);
        QString itemText = item->text();

        // Skip category separators
        if (itemText.startsWith("━━━")) {
            item->setHidden(false);
            continue;
        }

        // Remove icon prefix for matching
        QString algorithmName = itemText;
        if (algorithmName.length() > 2) {
            algorithmName = algorithmName.mid(2); // Remove icon (2 characters including space)
        }

        // Show/hide based on search text (case insensitive, contains match)
        if (text.isEmpty()) {
            item->setHidden(false);
        } else {
            bool matches = algorithmName.contains(text, Qt::CaseInsensitive);
            item->setHidden(!matches);
        }
    }

    // Hide empty categories (categories with no visible algorithms)
    QString currentCategory = "";
    bool categoryHasVisibleItems = false;

    for (int i = 0; i < algorithmList->count(); i++) {
        QListWidgetItem* item = algorithmList->item(i);
        QString itemText = item->text();

        if (itemText.startsWith("━━━")) {
            // This is a category header
            if (!currentCategory.isEmpty()) {
                // Hide previous category if it has no visible items
                QListWidgetItem* prevCategoryItem = algorithmList->item(i - 1);
                for (int j = i - 1; j >= 0; j--) {
                    QListWidgetItem* checkItem = algorithmList->item(j);
                    if (checkItem->text().startsWith("━━━")) {
                        checkItem->setHidden(!categoryHasVisibleItems);
                        break;
                    }
                }
            }
            currentCategory = itemText;
            categoryHasVisibleItems = false;
        } else {
            // This is an algorithm item
            if (!item->isHidden()) {
                categoryHasVisibleItems = true;
            }
        }
    }

    // Handle last category
    if (!currentCategory.isEmpty()) {
        for (int i = algorithmList->count() - 1; i >= 0; i--) {
            QListWidgetItem* item = algorithmList->item(i);
            if (item->text().startsWith("━━━")) {
                item->setHidden(!categoryHasVisibleItems);
                break;
            }
        }
    }
}

void MainWindow::onAlgorithmSearchSubmitted() {
    QString searchText = algorithmSearchBox->text().trimmed();

    if (searchText.isEmpty()) {
        return;
    }

    // Find the algorithm that matches the search text
    for (int i = 0; i < algorithmList->count(); i++) {
        QListWidgetItem* item = algorithmList->item(i);
        QString itemText = item->text();

        // Skip category separators
        if (itemText.startsWith("━━━")) {
            continue;
        }

        // Remove icon prefix for comparison
        QString algorithmName = itemText;
        if (algorithmName.length() > 2) {
            algorithmName = algorithmName.mid(2);
        }

        // Check for exact match (case insensitive) - this handles completer selection
        // or contains match for partial searches
        bool exactMatch = algorithmName.compare(searchText, Qt::CaseInsensitive) == 0;
        bool partialMatch = !item->isHidden() && algorithmName.contains(searchText, Qt::CaseInsensitive);

        if (exactMatch || partialMatch) {
            // Select this item
            algorithmList->setCurrentItem(item);
            algorithmList->scrollToItem(item);

            // Clear the search box to show all algorithms again
            algorithmSearchBox->clear();

            // If exact match (from completer), break immediately
            // If partial match, break on first visible match
            break;
        }
    }
}

void MainWindow::onGenerateClicked() {
    regenerateMesh();
}

void MainWindow::regenerateMesh() {
    const auto& algo = library.getAlgorithm(currentAlgorithm);

    std::vector<AlgorithmParameter> params;
    for (size_t i = 0; i < algo.parameters.size(); i++) {
        float normalized = parameterSliders[i]->value() / 100.0f;
        float value = algo.parameters[i].minValue +
                     normalized * (algo.parameters[i].maxValue - algo.parameters[i].minValue);

        params.push_back(AlgorithmParameter(
            algo.parameters[i].name,
            value,
            algo.parameters[i].minValue,
            algo.parameters[i].maxValue,
            algo.parameters[i].step
        ));
    }

    // Clear current mesh before generating new one
    currentMesh.clear();
    currentMesh = Mesh();
    
    // Generate new mesh
    currentMesh = library.generate(currentAlgorithm, params);
    
    // Set to viewport
    viewport->setMesh(currentMesh);
    updateWindowTitle();

    // Enable Save button now that mesh is generated
    saveButton->setEnabled(true);
}

void MainWindow::updateWindowTitle() {
    setWindowTitle(QString("Mesher • Parametric Modeler - Vertices: %1, Triangles: %2")
        .arg(currentMesh.getVertexCount())
        .arg(currentMesh.getTriangleCount()));
}

void MainWindow::onExportClicked() {
#ifdef __EMSCRIPTEN__
    // For WebAssembly: Use a custom dialog with filename input and format selection
    QDialog* dialog = new QDialog(this);
    dialog->setWindowTitle("Save Mesh to File");
    dialog->setMinimumWidth(400);
    dialog->setAttribute(Qt::WA_DeleteOnClose);
    
    QVBoxLayout* layout = new QVBoxLayout(dialog);
    layout->setContentsMargins(12, 12, 12, 12);
    layout->setSpacing(12);
    
    // Filename input
    QLabel* nameLabel = new QLabel("Filename:");
    layout->addWidget(nameLabel);
    
    QLineEdit* filenameEdit = new QLineEdit("mesh");
    filenameEdit->setPlaceholderText("Enter filename (without extension)");
    layout->addWidget(filenameEdit);
    
    // File format dropdown
    QLabel* formatLabel = new QLabel("File Format:");
    layout->addWidget(formatLabel);
    
    QComboBox* formatCombo = new QComboBox();
    formatCombo->addItem("Alias Wavefront Object (.obj)", "obj");
    formatCombo->addItem("STL File Format (.stl)", "stl");
    formatCombo->addItem("Stanford Polygon File Format (.ply)", "ply");
    formatCombo->addItem("FBX (.fbx)", "fbx");
    formatCombo->addItem("X3D File Format (.x3d)", "x3d");
    formatCombo->addItem("VRML File Format (.wrl)", "wrl");
    formatCombo->addItem("DXF File Format (.dxf)", "dxf");
    formatCombo->addItem("3D-Studio File Format (.3ds)", "3ds");
    formatCombo->addItem("JavaScript JSON (.json)", "json");
    layout->addWidget(formatCombo);
    
    // Buttons
    QHBoxLayout* buttonLayout = new QHBoxLayout();
    QPushButton* saveButton = new QPushButton("Save");
    QPushButton* cancelButton = new QPushButton("Cancel");
    saveButton->setStyleSheet("QPushButton { background-color: #2E2E2E; color: white; padding: 8px 20px; }");
    cancelButton->setStyleSheet("QPushButton { background-color: #242424; color: white; padding: 8px 20px; }");
    
    buttonLayout->addStretch();
    buttonLayout->addWidget(saveButton);
    buttonLayout->addWidget(cancelButton);
    layout->addLayout(buttonLayout);
    
    // Connect buttons
    connect(cancelButton, &QPushButton::clicked, dialog, &QDialog::reject);
    connect(saveButton, &QPushButton::clicked, [this, dialog, filenameEdit, formatCombo]() {
        QString baseName = filenameEdit->text().trimmed();
        if (baseName.isEmpty()) {
            QMessageBox::warning(dialog, "Invalid Filename", "Please enter a filename.");
            return;
        }
        
        // Get selected extension
        QString ext = formatCombo->currentData().toString();
        QString filename = baseName + "." + ext;
        
        bool success = false;
        std::string tempPath = "/tmp/" + filename.toStdString();
        std::string filepath = filename.toStdString();
        std::string extStr = ext.toStdString();
        
        // Use FBX exporter for FBX format (supports animation)
        if (extStr == "fbx") {
            if (animator.getKeyFrameCount() > 0) {
                success = FBXExporter::exportAnimatedMesh(tempPath, currentMesh, animator);
            } else {
                success = FBXExporter::exportMesh(tempPath, currentMesh);
            }
        } else {
            // Use MeshExporter for all other formats
            success = MeshExporter::exportMesh(tempPath, currentMesh);
        }
        
        if (success) {
            // Trigger browser download
            std::string jsFilename = filename.toStdString();
            EM_ASM({
                var filename = UTF8ToString($0);
                var filepath = UTF8ToString($1);
                
                try {
                    // Read file from virtual filesystem
                    var content = FS.readFile(filepath);
                    
                    // Create blob and trigger download
                    var blob = new Blob([content], {type: 'application/octet-stream'});
                    var url = URL.createObjectURL(blob);
                    var a = document.createElement('a');
                    a.href = url;
                    a.download = filename;
                    a.style.display = 'none';
                    document.body.appendChild(a);
                    a.click();
                    
                    // Cleanup after a short delay
                    setTimeout(function() {
                        document.body.removeChild(a);
                        URL.revokeObjectURL(url);
                    }, 100);
                    
                    // Clean up virtual filesystem
                    try {
                        FS.unlink(filepath);
                    } catch(e) {}
                } catch(e) {
                    console.error('Download failed:', e);
                }
            }, jsFilename.c_str(), tempPath.c_str());
            
            dialog->accept();
        } else {
            QMessageBox::warning(dialog, "Export Failed",
                "Failed to export mesh to " + filename);
        }
    });
    
    // Show dialog
    dialog->show();
    
#else
    // Desktop version: Use traditional file dialog
    const QStringList filters({
        "Save Mesh (*.3ds *.obj *.dxf *.fbx *.json *.ply *.stl *.wrl *.x3d)",
        "3D-Studio File Format (.3ds)",
        "Alias Wavefront Object (.obj)",
        "DXF File Format (.dxf)",
        "FBX (.fbx)",
        "JavaScript JSON (.json)",
        "Stanford Polygon File Format (.ply)",
        "STL File Format(.stl)",
        "VRML File Format (.wrl)",
        "X3D File Format (.x3d)",
        "All Files (*)"
    });

    QString filename = QFileDialog::getSaveFileName(
        this,
        "Save Mesh to File",
        "mesh.fbx",
        filters.join(";;"),
        nullptr,
        QFileDialog::DontUseNativeDialog
    );

    if (!filename.isEmpty()) {
        bool success = false;
        std::string filepath = filename.toStdString();
        
        // Determine file extension
        std::string ext = filepath.substr(filepath.find_last_of('.') + 1);
        std::transform(ext.begin(), ext.end(), ext.begin(), ::tolower);

        // Use FBX exporter for FBX format (supports animation)
        if (ext == "fbx") {
            if (animator.getKeyFrameCount() > 0) {
                success = FBXExporter::exportAnimatedMesh(filepath, currentMesh, animator);
            } else {
                success = FBXExporter::exportMesh(filepath, currentMesh);
            }
        } else {
            // Use MeshExporter for all other formats
            success = MeshExporter::exportMesh(filepath, currentMesh);
        }

        if (!success) {
            QMessageBox::warning(this, "Export Failed",
                "Failed to export mesh to " + filename);
        }
    }
#endif
}

void MainWindow::onWireframeToggled(bool checked) {
    viewport->setWireframeMode(checked);
}

void MainWindow::onRenderModeChanged(int index) {
    GLViewport::RenderMode mode;
    switch (index) {
        case 0: mode = GLViewport::POINTS; break;
        case 1: mode = GLViewport::WIREFRAME; break;
        case 2: mode = GLViewport::FLAT; break;
        case 3: mode = GLViewport::SOLID; break;
        case 4: mode = GLViewport::SOLID_WIREFRAME; break;
        case 5: mode = GLViewport::HIDDEN_LINE; break;
        case 6: mode = GLViewport::VERTICES; break;
        case 7: mode = GLViewport::SMOOTH_SHADED; break;
        case 8: mode = GLViewport::FLAT_LINES; break;
        case 9: mode = GLViewport::X_RAY; break;
        default: mode = GLViewport::SOLID; break;
    }
    viewport->setRenderMode(mode);
}

void MainWindow::onFrameAllToggled(bool checked) {
    // When FrameAll is toggled on, immediately frame the current mesh
    if (checked && currentMesh.getVertexCount() > 0) {
        viewport->frameAll();
    }
}

void MainWindow::onAnimationTimerUpdate() {
    animationTime += 0.016f;

    Mesh animatedMesh = currentMesh;
    animator.animateWave(animatedMesh, animationTime, 0.3f, 2.0f);

    viewport->setMesh(animatedMesh);
    viewport->setAnimationTime(animationTime);
}

std::vector<AlgorithmParameter> MainWindow::getCurrentParameters() {
    const auto& algo = library.getAlgorithm(currentAlgorithm);
    std::vector<AlgorithmParameter> params;

    for (size_t i = 0; i < algo.parameters.size(); i++) {
        float normalized = parameterSliders[i]->value() / 100.0f;
        float value = algo.parameters[i].minValue +
                     normalized * (algo.parameters[i].maxValue - algo.parameters[i].minValue);

        params.push_back(AlgorithmParameter(
            algo.parameters[i].name,
            value,
            algo.parameters[i].minValue,
            algo.parameters[i].maxValue,
            algo.parameters[i].step
        ));
    }

    return params;
}

void MainWindow::onAddToStack() {
    const auto& algo = library.getAlgorithm(currentAlgorithm);
    auto algorithms = AlgorithmRegistry::getAllAlgorithms();

    QListWidgetItem* currentItem = algorithmList->currentItem();
    QString algoName = currentItem ? currentItem->text() : "Unknown";

    StackEntry entry;
    entry.algorithmType = currentAlgorithm;
    entry.parameters = getCurrentParameters();
    entry.name = algoName;

    algorithmStack.push_back(entry);
    updateStackUI();

    // Generate from stack with new item
    onGenerateFromStack();
}

void MainWindow::updateStackUI() {
    stackList->clear();

    // Update group box title with item count (only show count if > 0)
    int count = algorithmStack.size();
    QString title;
    if (count > 0) {
        title = QString("Algorithm Stack (%1 %2)").arg(count).arg(count == 1 ? "item" : "items");
    } else {
        title = "Algorithm Stack";
    }
    stackGroupBox->setTitle(title);

    // Enable/disable Generate from Stack button based on stack size
    generateStackButton->setEnabled(count > 0);

    for (size_t i = 0; i < algorithmStack.size(); i++) {
        QString itemText = QString::number(i + 1) + ". " + algorithmStack[i].name;

        // Add parameter values
        if (!algorithmStack[i].parameters.empty()) {
            itemText += " (";
            for (size_t j = 0; j < algorithmStack[i].parameters.size(); j++) {
                const auto& param = algorithmStack[i].parameters[j];
                itemText += QString::fromStdString(param.name) + "=" + QString::number(param.value, 'f', 2);
                if (j < algorithmStack[i].parameters.size() - 1) {
                    itemText += ", ";
                }
            }
            itemText += ")";
        }

        stackList->addItem(itemText);
    }
}

void MainWindow::onRemoveFromStack() {
    int currentRow = stackList->currentRow();

    if (currentRow >= 0 && currentRow < (int)algorithmStack.size()) {
        algorithmStack.erase(algorithmStack.begin() + currentRow);
        updateStackUI();

        // Generate from stack after removal
        if (!algorithmStack.empty()) {
            onGenerateFromStack();
        }
    }
}

void MainWindow::onClearStack() {
    algorithmStack.clear();
    updateStackUI();
}

void MainWindow::onMoveUp() {
    int currentRow = stackList->currentRow();

    if (currentRow > 0 && currentRow < (int)algorithmStack.size()) {
        std::swap(algorithmStack[currentRow], algorithmStack[currentRow - 1]);
        updateStackUI();
        stackList->setCurrentRow(currentRow - 1);

        // Generate from stack with new order
        onGenerateFromStack();
    }
}

void MainWindow::onMoveDown() {
    int currentRow = stackList->currentRow();

    if (currentRow >= 0 && currentRow < (int)algorithmStack.size() - 1) {
        std::swap(algorithmStack[currentRow], algorithmStack[currentRow + 1]);
        updateStackUI();
        stackList->setCurrentRow(currentRow + 1);

        // Generate from stack with new order
        onGenerateFromStack();
    }
}

void MainWindow::onShuffleStack() {
    if (algorithmStack.empty()) return;

    std::random_device rd;
    std::mt19937 gen(rd());
    std::shuffle(algorithmStack.begin(), algorithmStack.end(), gen);

    updateStackUI();

    // Generate from stack with shuffled order
    onGenerateFromStack();
}

// In MainWindow.cpp, within onGenerateFromStack() function
// Replace the existing modifier/generator logic section with this expanded version:
void MainWindow::onGenerateFromStack() {
    if (algorithmStack.empty()) {
        return;
    }

    // Get morph percentage from slider
    float morphPercent = morphSlider->value();

    // Start with first algorithm
    currentMesh = library.generate(algorithmStack[0].algorithmType, algorithmStack[0].parameters);
    QApplication::processEvents();

    // Apply remaining algorithms one by one
    for (size_t i = 1; i < algorithmStack.size(); i++) {
        AlgorithmLibrary::AlgorithmType type = algorithmStack[i].algorithmType;
        const auto& params = algorithmStack[i].parameters;
        QApplication::processEvents();

        // Check if this is a modifier (MOD_* enum range is 76-101)
        bool isModifier = (type >= AlgorithmLibrary::MOD_SCALE && type <= AlgorithmLibrary::MOD_RBF_OPTIMIZE);
        
        // Check if this is a stackable overlay pattern that should merge with base mesh
        bool isStackableOverlay = false;
        
        // Define which algorithms can be used as stackable overlays
        if (!isModifier) {  // Only check for overlays if not a modifier
            switch(type) {
                // TESSELLATIONS - Apply as surface decoration
                case AlgorithmLibrary::TESS_PENROSE:
                case AlgorithmLibrary::TESS_ISLAMIC:
                case AlgorithmLibrary::TESS_TRUCHET:
                case AlgorithmLibrary::TESS_MODULAR_CIRCLES:
                    isStackableOverlay = true;
                    break;
                    
                // ARCHITECTURAL - Apply as structural overlay
                case AlgorithmLibrary::ARCH_VORONOI_LATTICE:
                case AlgorithmLibrary::ARCH_HEXAGONAL_GRID:
                case AlgorithmLibrary::ARCH_FACADE_PATTERN:
                case AlgorithmLibrary::ARCH_STRUCTURAL_FRAME:
                    isStackableOverlay = true;
                    break;
                    
                // ORGANIC - Apply as surface decoration/growth
                case AlgorithmLibrary::ORG_CORAL:
                case AlgorithmLibrary::ORG_FLOWER:
                case AlgorithmLibrary::ORG_LEAF_VENATION:
                case AlgorithmLibrary::ORG_CRYSTAL:
                case AlgorithmLibrary::ORG_SHELL_NAUTILUS:
                case AlgorithmLibrary::ORG_SHELL_CONCH:
                case AlgorithmLibrary::ORG_TENTACLE:
                    isStackableOverlay = true;
                    break;
                    
                // FRACTALS - Apply as surface detail/decoration
                case AlgorithmLibrary::FRAC_TREE:
                case AlgorithmLibrary::FRAC_L_SYSTEM:
                case AlgorithmLibrary::FRAC_KOCH_SNOWFLAKE:
                case AlgorithmLibrary::FRAC_DRAGON_CURVE:
                case AlgorithmLibrary::FRAC_SIERPINSKI:
                case AlgorithmLibrary::FRAC_FERN:
                case AlgorithmLibrary::FRAC_MENGER_SPONGE:
                    isStackableOverlay = true;
                    break;
                    
                // STRUCTURAL - Apply as lattice/framework overlay
                case AlgorithmLibrary::STRUCT_DIAMOND_LATTICE:
                case AlgorithmLibrary::STRUCT_PIPE_NETWORK:
                case AlgorithmLibrary::STRUCT_LATTICE:
                    isStackableOverlay = true;
                    break;
                    
                // COMPUTATIONAL - Apply as pattern overlay
                case AlgorithmLibrary::COMP_VORONOI_SURFACE:
                case AlgorithmLibrary::COMP_DELAUNAY_SURFACE:
                case AlgorithmLibrary::COMP_VORONOI_DIAGRAM:
                case AlgorithmLibrary::COMP_DELAUNAY_TRIANGULATION:
                case AlgorithmLibrary::COMP_PERLIN_NOISE_SURFACE:
                    isStackableOverlay = true;
                    break;
                    
                // ATTRACTORS - Apply as curve decoration
                case AlgorithmLibrary::ATTR_LORENZ:
                    isStackableOverlay = true;
                    break;
                    
                // PARAMETRIC - Apply as surface detail
                case AlgorithmLibrary::PARAM_LISSAJOUS:
                case AlgorithmLibrary::PARAM_ROSE:
                case AlgorithmLibrary::PARAM_HELIX:
                    isStackableOverlay = true;
                    break;
                    
                // TOPOLOGY - Apply as decorative knots/curves
                case AlgorithmLibrary::TOPO_TREFOIL_KNOT:
                case AlgorithmLibrary::TOPO_FIGURE_EIGHT_KNOT:
                case AlgorithmLibrary::TOPO_HEART:
                    isStackableOverlay = true;
                    break;
                    
                // MECHANICAL - Apply as functional components
                case AlgorithmLibrary::MECH_GEAR:
                case AlgorithmLibrary::PARAM_SPRING:
                case AlgorithmLibrary::MECH_THREADED_BOLT:
                case AlgorithmLibrary::MECH_BEARING:
                    isStackableOverlay = true;
                    break;
                    
                default:
                    isStackableOverlay = false;
                    break;
            }
        }
        
        if (isModifier) {
            // Apply modifier to current mesh
            if (type == AlgorithmLibrary::MOD_SCALE && params.size() >= 3) {
                currentMesh.transform(Vec3(0,0,0), Vec3(params[0].value, params[1].value, params[2].value), Vec3(0,0,0));
            } else if (type == AlgorithmLibrary::MOD_TRANSLATE && params.size() >= 3) {
                currentMesh.transform(Vec3(params[0].value, params[1].value, params[2].value), Vec3(1,1,1), Vec3(0,0,0));
            } else if (type == AlgorithmLibrary::MOD_ROTATE && params.size() >= 3) {
                currentMesh.transform(Vec3(0,0,0), Vec3(1,1,1), Vec3(params[0].value, params[1].value, params[2].value));
            } else if (type == AlgorithmLibrary::MOD_SUBDIVIDE && params.size() >= 1) {
                MeshGenerator::subdivide(currentMesh, (int)params[0].value);
            } else if (type == AlgorithmLibrary::MOD_EXTRUDE && params.size() >= 1) {
                float distance = params[0].value;
                currentMesh.computeNormals();
                Mesh extruded;
                for (size_t j = 0; j < currentMesh.getVertexCount(); j++) {
                    const Vertex& v = currentMesh.getVertex(j);
                    Vertex extrudedV = v;
                    extrudedV.position = v.position + v.normal * distance;
                    extruded.addVertex(extrudedV);
                }
                size_t originalVertCount = currentMesh.getVertexCount();
                for (size_t j = 0; j < originalVertCount; j++) {
                    extruded.addVertex(currentMesh.getVertex(j));
                }
                for (size_t j = 0; j < currentMesh.getTriangleCount(); j++) {
                    Triangle t = currentMesh.getTriangle(j);
                    extruded.addTriangle(t.v0, t.v1, t.v2);
                    extruded.addTriangle(originalVertCount + t.v2, originalVertCount + t.v1, originalVertCount + t.v0);
                }
                currentMesh = extruded;
                currentMesh.computeNormals();
            } else if (type == AlgorithmLibrary::MOD_TWIST_DEFORM && params.size() >= 1) {
                float amount = params[0].value;
                float minY = 1e10f, maxY = -1e10f;
                for (size_t j = 0; j < currentMesh.getVertexCount(); j++) {
                    const Vertex& v = currentMesh.getVertex(j);
                    minY = std::min(minY, v.position.y);
                    maxY = std::max(maxY, v.position.y);
                }
                float yRange = maxY - minY;
                if (yRange < 0.001f) yRange = 1.0f;
                for (size_t j = 0; j < currentMesh.getVertexCount(); j++) {
                    Vertex& v = currentMesh.getVertex(j);
                    float normalizedY = (v.position.y - minY) / yRange;
                    float angle = (normalizedY - 0.5f) * 2.0f * amount;
                    float c = std::cos(angle);
                    float s = std::sin(angle);
                    Vec3 twisted;
                    twisted.x = v.position.x * c - v.position.z * s;
                    twisted.y = v.position.y;
                    twisted.z = v.position.x * s + v.position.z * c;
                    v.position = twisted;
                }
                currentMesh.computeNormals();
            } else if (type == AlgorithmLibrary::MOD_WAVE_DEFORM && params.size() >= 2) {
                float amplitude = params[0].value;
                float frequency = params[1].value;
                for (size_t j = 0; j < currentMesh.getVertexCount(); j++) {
                    Vertex& v = currentMesh.getVertex(j);
                    v.position.y += amplitude * std::sin(v.position.x * frequency) * std::cos(v.position.z * frequency);
                }
                currentMesh.computeNormals();
            } else if (type == AlgorithmLibrary::MOD_SPHERIFY && params.size() >= 1) {
                float amount = params[0].value;
                for (size_t j = 0; j < currentMesh.getVertexCount(); j++) {
                    Vertex& v = currentMesh.getVertex(j);
                    Vec3 normalized = v.position;
                    normalized.normalize();
                    v.position = v.position * (1.0f - amount) + normalized * amount;
                }
                currentMesh.computeNormals();
            } else if (type == AlgorithmLibrary::MOD_BEND_DEFORM && params.size() >= 1) {
                float angle = params[0].value * M_PI / 180.0f;
                for (size_t j = 0; j < currentMesh.getVertexCount(); j++) {
                    Vertex& v = currentMesh.getVertex(j);
                    float bendAngle = v.position.y * angle / 3.0f;
                    Vec3 bent;
                    bent.x = v.position.x * std::cos(bendAngle) - v.position.y * std::sin(bendAngle);
                    bent.y = v.position.x * std::sin(bendAngle) + v.position.y * std::cos(bendAngle);
                    bent.z = v.position.z;
                    v.position = bent;
                }
                currentMesh.computeNormals();
            } else if (type == AlgorithmLibrary::MOD_TAPER_DEFORM && params.size() >= 1) {
                float amount = params[0].value;
                for (size_t j = 0; j < currentMesh.getVertexCount(); j++) {
                    Vertex& v = currentMesh.getVertex(j);
                    float scale = 1.0f - (v.position.y + 1.0f) * amount * 0.5f;
                    v.position.x *= scale;
                    v.position.z *= scale;
                }
                currentMesh.computeNormals();
            } else if (type == AlgorithmLibrary::MOD_NOISE && params.size() >= 2) {
                float amplitude = params[0].value;
                int seed = (int)params[1].value;
                currentMesh.computeNormals();
                std::mt19937 gen(seed);
                std::uniform_real_distribution<float> dist(-1.0f, 1.0f);
                for (size_t j = 0; j < currentMesh.getVertexCount(); j++) {
                    Vertex& v = currentMesh.getVertex(j);
                    float noise = dist(gen) * amplitude;
                    v.position = v.position + v.normal * noise;
                }
                currentMesh.computeNormals();
            } else if (type == AlgorithmLibrary::MOD_PERLIN_NOISE && params.size() >= 2) {
                float amplitude = params[0].value;
                float frequency = params[1].value;
                std::random_device rd;
                std::mt19937 gen(rd());
                std::uniform_real_distribution<float> dist(-1.0f, 1.0f);
                for (size_t j = 0; j < currentMesh.getVertexCount(); j++) {
                    Vertex& v = currentMesh.getVertex(j);
                    float noise = dist(gen) * amplitude * std::sin(v.position.x * frequency) * std::cos(v.position.z * frequency);
                    v.position.y += noise;
                }
                currentMesh.computeNormals();
            } else if (type == AlgorithmLibrary::MOD_PULSE && params.size() >= 2) {
                float amplitude = params[0].value;
                float frequency = params[1].value;
                currentMesh.computeNormals();
                for (size_t j = 0; j < currentMesh.getVertexCount(); j++) {
                    Vertex& v = currentMesh.getVertex(j);
                    float distance = v.position.length();
                    float pulse = amplitude * std::sin(distance * frequency * M_PI);
                    v.position = v.position + v.normal * pulse;
                }
                currentMesh.computeNormals();
            } else if (type == AlgorithmLibrary::MOD_REPEAT_DOMAIN && params.size() >= 4) {
                int repeatX = (int)params[0].value;
                int repeatY = (int)params[1].value;
                int repeatZ = (int)params[2].value;
                float spacing = params[3].value;

                Mesh baseMesh = currentMesh;
                Mesh repeatedMesh;

                for (int x = 0; x < repeatX; x++) {
                    for (int y = 0; y < repeatY; y++) {
                        for (int z = 0; z < repeatZ; z++) {
                            Vec3 offset(x * spacing, y * spacing, z * spacing);

                            size_t baseVertexCount = repeatedMesh.getVertexCount();
                            for (size_t j = 0; j < baseMesh.getVertexCount(); j++) {
                                Vertex v = baseMesh.getVertex(j);
                                v.position = v.position + offset;
                                repeatedMesh.addVertex(v);
                            }

                            for (size_t j = 0; j < baseMesh.getTriangleCount(); j++) {
                                Triangle t = baseMesh.getTriangle(j);
                                repeatedMesh.addTriangle(t.v0 + baseVertexCount, t.v1 + baseVertexCount, t.v2 + baseVertexCount);
                            }
                        }
                    }
                }
                currentMesh = repeatedMesh;
                currentMesh.computeNormals();
            } else if (type == AlgorithmLibrary::MOD_SMOOTH && params.size() >= 2) {
                int iterations = (int)params[0].value;
                float factor = params[1].value;
                for (int iter = 0; iter < iterations; iter++) {
                    std::vector<Vec3> newPositions(currentMesh.getVertexCount());
                    for (size_t j = 0; j < currentMesh.getVertexCount(); j++) {
                        const Vertex& v = currentMesh.getVertex(j);
                        Vec3 avgPos = v.position;
                        int neighborCount = 1;
                        if (j > 0) {
                            avgPos = avgPos + currentMesh.getVertex(j - 1).position;
                            neighborCount++;
                        }
                        if (j < currentMesh.getVertexCount() - 1) {
                            avgPos = avgPos + currentMesh.getVertex(j + 1).position;
                            neighborCount++;
                        }
                        avgPos = avgPos * (1.0f / neighborCount);
                        newPositions[j] = v.position * (1.0f - factor) + avgPos * factor;
                    }
                    for (size_t j = 0; j < currentMesh.getVertexCount(); j++) {
                        currentMesh.getVertex(j).position = newPositions[j];
                    }
                }
                currentMesh.computeNormals();
            } else if (type == AlgorithmLibrary::MOD_INFLATE && params.size() >= 1) {
                float amount = params[0].value;
                currentMesh.computeNormals();
                for (size_t j = 0; j < currentMesh.getVertexCount(); j++) {
                    Vertex& v = currentMesh.getVertex(j);
                    v.position = v.position + v.normal * amount;
                }
                currentMesh.computeNormals();
            } else if (type == AlgorithmLibrary::MOD_SOLIDIFY && params.size() >= 1) {
                float thickness = params[0].value;
                currentMesh.computeNormals();
                Mesh inner;
                for (size_t j = 0; j < currentMesh.getVertexCount(); j++) {
                    const Vertex& v = currentMesh.getVertex(j);
                    Vertex innerV = v;
                    innerV.position = v.position - v.normal * thickness;
                    inner.addVertex(innerV);
                }
                for (size_t j = 0; j < currentMesh.getTriangleCount(); j++) {
                    Triangle t = currentMesh.getTriangle(j);
                    inner.addTriangle(t.v2, t.v1, t.v0);
                }
                currentMesh.merge(inner);
                currentMesh.computeNormals();
            } else if (type == AlgorithmLibrary::MOD_DECIMATE && params.size() >= 1) {
                float ratio = params[0].value;
                if (ratio < 1.0f && ratio > 0.0f) {
                    Mesh decimated;
                    int skip = (int)(1.0f / ratio);
                    if (skip < 1) skip = 1;
                    for (size_t j = 0; j < currentMesh.getVertexCount(); j += skip) {
                        decimated.addVertex(currentMesh.getVertex(j));
                    }
                    size_t newVertCount = decimated.getVertexCount();
                    for (size_t j = 0; j < currentMesh.getTriangleCount(); j++) {
                        Triangle t = currentMesh.getTriangle(j);
                        int newV0 = t.v0 / skip;
                        int newV1 = t.v1 / skip;
                        int newV2 = t.v2 / skip;
                        if (newV0 < (int)newVertCount && newV1 < (int)newVertCount && newV2 < (int)newVertCount) {
                            decimated.addTriangle(newV0, newV1, newV2);
                        }
                    }
                    currentMesh = decimated;
                    currentMesh.computeNormals();
                }
            } else if (type == AlgorithmLibrary::MOD_LATTICE_DEFORM && params.size() >= 2) {
                float strength = params[0].value;
                int resolution = (int)params[1].value;
                if (resolution < 2) resolution = 2;
                for (size_t j = 0; j < currentMesh.getVertexCount(); j++) {
                    Vertex& v = currentMesh.getVertex(j);
                    float gridX = v.position.x * resolution;
                    float gridY = v.position.y * resolution;
                    float gridZ = v.position.z * resolution;
                    v.position.x += std::sin(gridY) * strength;
                    v.position.y += std::sin(gridZ) * strength;
                    v.position.z += std::sin(gridX) * strength;
                }
                currentMesh.computeNormals();
            } else if (type == AlgorithmLibrary::MOD_MIRROR && params.size() >= 1) {
                int axis = (int)params[0].value;
                if (axis < 0) axis = 0;
                if (axis > 2) axis = 2;
                Mesh mirrored;
                for (size_t j = 0; j < currentMesh.getVertexCount(); j++) {
                    Vertex v = currentMesh.getVertex(j);
                    if (axis == 0) v.position.x = -v.position.x;
                    else if (axis == 1) v.position.y = -v.position.y;
                    else v.position.z = -v.position.z;
                    mirrored.addVertex(v);
                }
                for (size_t j = 0; j < currentMesh.getTriangleCount(); j++) {
                    Triangle t = currentMesh.getTriangle(j);
                    mirrored.addTriangle(t.v2, t.v1, t.v0);
                }
                currentMesh.merge(mirrored);
                currentMesh.computeNormals();
            } else if (type == AlgorithmLibrary::MOD_ARRAY && params.size() >= 2) {
                int count = (int)params[0].value;
                float spacing = params[1].value;
                if (count < 1) count = 1;
                if (count > 20) count = 20;
                Mesh original = currentMesh;
                for (int k = 1; k < count; k++) {
                    Mesh copy = original;
                    Vec3 offset(spacing * k, 0, 0);
                    for (size_t j = 0; j < copy.getVertexCount(); j++) {
                        copy.getVertex(j).position = copy.getVertex(j).position + offset;
                    }
                    currentMesh.merge(copy);
                }
            } else if (type == AlgorithmLibrary::MOD_SHELL && params.size() >= 1) {
                float thickness = params[0].value;
                currentMesh.computeNormals();
                for (size_t j = 0; j < currentMesh.getVertexCount(); j++) {
                    Vertex& v = currentMesh.getVertex(j);
                    v.position = v.position + v.normal * thickness;
                }
                currentMesh.computeNormals();
            } else if (type == AlgorithmLibrary::MOD_DISPLACE && params.size() >= 2) {
                float strength = params[0].value;
                float scale = params[1].value;
                currentMesh.computeNormals();
                for (size_t j = 0; j < currentMesh.getVertexCount(); j++) {
                    Vertex& v = currentMesh.getVertex(j);
                    float displacement = std::sin(v.position.x * scale) * std::cos(v.position.z * scale) * strength;
                    v.position = v.position + v.normal * displacement;
                }
                currentMesh.computeNormals();
            } else if (type == AlgorithmLibrary::MOD_REMESH && params.size() >= 1) {
                currentMesh.computeNormals();
            } else if (type == AlgorithmLibrary::MOD_VOXELIZE && params.size() >= 2) {
                int gridSize = (int)params[0].value;
                float voxelSize = params[1].value;

                VoxelGenerator voxGen(gridSize, gridSize, gridSize);
                voxGen.fillFromMesh(currentMesh, voxelSize);
                currentMesh = voxGen.generateMesh(voxelSize);
            } else if (type == AlgorithmLibrary::MOD_LAPLACIAN_RELAX && params.size() >= 2) {
                int iterations = (int)params[0].value;
                float factor = params[1].value;

                VoxelGenerator voxGen(1, 1, 1);
                voxGen.relaxMesh(currentMesh, iterations, factor);
            } else if (type == AlgorithmLibrary::MOD_CMAES_OPTIMIZE && params.size() >= 3) {
                int generations = (int)params[0].value;
                int population = (int)params[1].value;
                float target = params[2].value;

                int numParams = 8;
                CMAESOptimizer optimizer(numParams, 0.3);
                optimizer.setPopulationSize(population);

                auto objective = [target](const std::vector<double>& params) -> double {
                    double fitness = 0.0;
                    for (size_t i = 0; i < params.size(); i++) {
                        double deviation = std::abs(params[i] - target);
                        fitness += deviation * deviation;
                    }
                    return fitness;
                };

                std::vector<double> optimized = optimizer.optimize(objective, generations);

                currentMesh.computeNormals();
                for (size_t j = 0; j < currentMesh.getVertexCount(); j++) {
                    Vertex& v = currentMesh.getVertex(j);
                    Vec3 pos = v.position;

                    float theta = std::atan2(pos.z, pos.x);
                    float phi = std::acos(pos.y / (pos.length() + 0.001f));

                    float deformation = 0.0f;
                    for (int p = 0; p < numParams && p < (int)optimized.size(); p++) {
                        deformation += optimized[p] * std::sin((p + 1) * theta) * std::cos((p + 1) * phi);
                    }

                    v.position = pos + v.normal * deformation * 0.3f;
                }
                currentMesh.computeNormals();
            } else if (type == AlgorithmLibrary::MOD_NSGAII_OPTIMIZE && params.size() >= 3) {
                int generations = (int)params[0].value;
                int population = (int)params[1].value;
                float complexity = params[2].value;

                int numParams = 6;
                NSGAIIOptimizer optimizer(numParams, 2);
                optimizer.setPopulationSize(population);

                auto objectives = [complexity](const std::vector<double>& params) -> std::vector<double> {
                    double obj1 = 0.0;
                    for (size_t i = 0; i < params.size(); i++) {
                        obj1 += std::abs(params[i] - complexity * 0.2);
                    }
                    double obj2 = 0.0;
                    for (size_t i = 0; i < params.size(); i++) {
                        obj2 -= params[i] * params[i];
                    }
                    return {obj1, obj2};
                };

                std::vector<NSGAIIOptimizer::Individual> paretoFront = optimizer.optimize(objectives, generations);
                std::vector<double> optimized = paretoFront.empty() ?
                    std::vector<double>(numParams, 1.0) : paretoFront[0].variables;

                currentMesh.computeNormals();
                for (size_t j = 0; j < currentMesh.getVertexCount(); j++) {
                    Vertex& v = currentMesh.getVertex(j);
                    Vec3 pos = v.position;

                    float theta = std::atan2(pos.z, pos.x);
                    float phi = std::acos(pos.y / (pos.length() + 0.001f));

                    float deformation = 0.0f;
                    for (int p = 0; p < numParams && p < (int)optimized.size(); p++) {
                        deformation += optimized[p] * std::cos((p + 1) * theta) * std::sin((p + 1) * phi);
                    }

                    v.position = pos + v.normal * deformation * 0.4f;
                }
                currentMesh.computeNormals();
            } else if (type == AlgorithmLibrary::MOD_RBF_OPTIMIZE && params.size() >= 2) {
                int evaluations = (int)params[0].value;
                float smoothness = params[1].value;

                int numParams = 6;
                RBFOptimizer optimizer(numParams);

                std::vector<double> lower(numParams, -2.0);
                std::vector<double> upper(numParams, 2.0);
                optimizer.setBounds(lower, upper);

                auto objective = [smoothness](const std::vector<double>& params) -> double {
                    double fitness = 0.0;
                    for (size_t i = 0; i < params.size(); i++) {
                        fitness += params[i] * params[i] / smoothness;
                    }
                    for (size_t i = 0; i < params.size() - 1; i++) {
                        fitness += std::abs(params[i] * params[i+1]) * 0.5;
                    }
                    return fitness;
                };

                std::vector<double> optimized = optimizer.optimize(objective, evaluations);

                currentMesh.computeNormals();
                for (size_t j = 0; j < currentMesh.getVertexCount(); j++) {
                    Vertex& v = currentMesh.getVertex(j);
                    Vec3 pos = v.position;

                    float theta = std::atan2(pos.z, pos.x);
                    float phi = std::acos(pos.y / (pos.length() + 0.001f));

                    float deformation = 0.0f;
                    for (int p = 0; p < numParams && p < (int)optimized.size(); p++) {
                        float r = pos.length();
                        deformation += optimized[p] * std::exp(-r * r * 0.5f) * std::cos((p + 1) * theta);
                    }

                    v.position = pos + v.normal * deformation * smoothness * 0.5f;
                }
                currentMesh.computeNormals();
            } else {
                currentMesh.computeNormals();
            }
            
        } else if (isStackableOverlay) {
            // Generate the overlay pattern
            Mesh overlay = library.generate(type, params);
            
            // Optional: Scale or transform overlay based on base mesh bounds
            // This ensures the overlay fits nicely with the base mesh
            Vec3 baseMin, baseMax;
            currentMesh.getBounds(baseMin, baseMax);
            Vec3 baseCenter = (baseMin + baseMax) * 0.5f;
            Vec3 baseSize = baseMax - baseMin;
            float baseScale = std::max(std::max(baseSize.x, baseSize.y), baseSize.z);
            
            // For certain patterns, scale them to match the base mesh size
            bool shouldScaleToBase = false;
            switch(type) {
                case AlgorithmLibrary::TESS_PENROSE:
                case AlgorithmLibrary::TESS_ISLAMIC:
                case AlgorithmLibrary::TESS_TRUCHET:
                case AlgorithmLibrary::ARCH_HEXAGONAL_GRID:
                case AlgorithmLibrary::ARCH_FACADE_PATTERN:
                    shouldScaleToBase = true;
                    break;
                default:
                    break;
            }
            
            if (shouldScaleToBase && baseScale > 0.1f) {
                Vec3 overlayMin, overlayMax;
                overlay.getBounds(overlayMin, overlayMax);
                Vec3 overlaySize = overlayMax - overlayMin;
                float overlayScale = std::max(std::max(overlaySize.x, overlaySize.y), overlaySize.z);
                
                if (overlayScale > 0.1f) {
                    float scaleFactor = baseScale / overlayScale;
                    overlay.transform(Vec3(0,0,0), Vec3(scaleFactor, scaleFactor, scaleFactor), Vec3(0,0,0));
                }
                
                // Center the overlay on the base mesh
                Vec3 overlayCenter;
                overlay.getBounds(overlayMin, overlayMax);
                overlayCenter = (overlayMin + overlayMax) * 0.5f;
                Vec3 offset = baseCenter - overlayCenter;
                overlay.transform(offset, Vec3(1,1,1), Vec3(0,0,0));
            }
            
            // Merge overlay with current mesh
            currentMesh.merge(overlay);
            currentMesh.computeNormals();
            
        } else {
            // This is a generator algorithm - use morphing (existing code)
            Mesh newMesh = library.generate(type, params);

            if (morphPercent >= 100.0f) {
                currentMesh = newMesh;
            } else {
                Mesh meshACopy = currentMesh;
                Mesh meshBCopy = newMesh;
                matchVertexCounts(meshACopy, meshBCopy);
                currentMesh = morphMeshes(meshACopy, meshBCopy, morphPercent);
            }
        }
    }

    viewport->setMesh(currentMesh);
    updateWindowTitle();
    saveButton->setEnabled(true);
}

void MainWindow::onRandomizeSettings() {
    if (algorithmStack.empty()) {
        return;
    }

    std::random_device rd;
    std::mt19937 gen(rd());

    // Randomize parameters for each algorithm in the stack
    for (auto& entry : algorithmStack) {
        QApplication::processEvents();
        for (auto& param : entry.parameters) {
            // Generate random value within parameter's min/max range
            std::uniform_real_distribution<float> dist(param.minValue, param.maxValue);
            param.value = dist(gen);
        }
    }

    // Update the stack UI to show new parameter values
    updateStackUI();

    // Directly generate from stack
    onGenerateFromStack();
}

void MainWindow::onEditStackItem(QListWidgetItem* item) {
    int row = stackList->row(item);
    if (row < 0 || row >= (int)algorithmStack.size()) {
        return;
    }

    StackEntry& entry = algorithmStack[row];

    // Create edit dialog
    QDialog* dialog = new QDialog(this);
    dialog->setWindowTitle("Edit Item - " + entry.name);
    dialog->setMinimumWidth(400);
    dialog->setAttribute(Qt::WA_DeleteOnClose);

    QVBoxLayout* layout = new QVBoxLayout(dialog);
    layout->setContentsMargins(8, 8, 8, 8);
    layout->setSpacing(8);

    // Info label
    QLabel* infoLabel = new QLabel("<b>" + entry.name + "</b>");
    layout->addWidget(infoLabel);

    // Create sliders for each parameter
    QVector<QSlider*> sliders;
    QVector<QLabel*> valueLabels;

    for (size_t i = 0; i < entry.parameters.size(); i++) {
        auto& param = entry.parameters[i];

        QHBoxLayout* paramLayout = new QHBoxLayout();

        QLabel* nameLabel = new QLabel(QString::fromStdString(param.name) + ":");
        nameLabel->setStyleSheet("font-weight: bold;");
        nameLabel->setMinimumWidth(95);
        paramLayout->addWidget(nameLabel);

        QSlider* slider = new QSlider(Qt::Horizontal);
        slider->setRange(0, 100);
        float normalizedValue = (param.value - param.minValue) / (param.maxValue - param.minValue);
        slider->setValue((int)(normalizedValue * 100));
        paramLayout->addWidget(slider);

        QLabel* valueLabel = new QLabel(QString::number(param.value, 'f', 2));
        valueLabel->setMinimumWidth(60);

        // Capture min/max by value to avoid dangling references
        float minVal = param.minValue;
        float maxVal = param.maxValue;
        connect(slider, &QSlider::valueChanged, [minVal, maxVal, valueLabel, slider]() {
            float normalized = slider->value() / 100.0f;
            float value = minVal + normalized * (maxVal - minVal);
            valueLabel->setText(QString::number(value, 'f', 2));
        });

        paramLayout->addWidget(valueLabel);
        layout->addLayout(paramLayout);

        sliders.push_back(slider);
        valueLabels.push_back(valueLabel);
    }

    // Buttons
    QHBoxLayout* buttonLayout = new QHBoxLayout();
    QPushButton* okButton = new QPushButton("Ok");
    QPushButton* cancelButton = new QPushButton("Cancel");
    okButton->setStyleSheet("QPushButton { background-color: #2E2E2E; color: white; padding: 6px; }");
    cancelButton->setStyleSheet("QPushButton { background-color: #242424; color: white; padding: 6px; }");

    // Capture row and sliders by value for the lambda
    connect(okButton, &QPushButton::clicked, [this, row, sliders, dialog]() {
        // Update parameter values
        if (row >= 0 && row < (int)algorithmStack.size()) {
            StackEntry& entry = algorithmStack[row];
            for (size_t i = 0; i < entry.parameters.size() && i < (size_t)sliders.size(); i++) {
                float normalized = sliders[i]->value() / 100.0f;
                entry.parameters[i].value = entry.parameters[i].minValue +
                    normalized * (entry.parameters[i].maxValue - entry.parameters[i].minValue);
            }

            // Update UI to show new values
            updateStackUI();

            // Generate from stack with updated parameters
            onGenerateFromStack();
        }
        dialog->accept();
    });

    connect(cancelButton, &QPushButton::clicked, dialog, &QDialog::reject);

    buttonLayout->addStretch();
    buttonLayout->addWidget(okButton);
    buttonLayout->addWidget(cancelButton);
    layout->addLayout(buttonLayout);

    // Show dialog (non-modal in WebAssembly, modal on desktop)
#ifdef __EMSCRIPTEN__
    dialog->show();
#else
    dialog->exec();
#endif
}

void MainWindow::onMarchingCubes() {
    sdfEngine.setSDF([](const Vec3& p) -> float {
        return p.length() - 2.0f;  // Sphere with radius 2
    });

    QApplication::processEvents();
    currentMesh = MeshGenerator::marchingCubes(sdfEngine, 32);
    QApplication::processEvents();
    viewport->setMesh(currentMesh);
    updateWindowTitle();
}

void MainWindow::onSDFUnion() {
    if (currentMesh.getVertexCount() == 0) {
        QMessageBox::information(this, "No Mesh", "Generate a mesh first!");
        return;
    }

    sdfMeshA = currentMesh;
    Mesh sphere = MeshGenerator::createSphere(1.5f, 16, 16);
    sphere.transform(Vec3(1, 0, 0), Vec3(1, 1, 1), Vec3(0, 0, 0));

    currentMesh.merge(sphere);
    viewport->setMesh(currentMesh);
    updateWindowTitle();
}

void MainWindow::onSDFSubtract() {
    QMessageBox::information(this, "SDF Subtract",
        "Subtraction requires boolean mesh operations.\nUse Marching Cubes with SDF subtract function.");
}

void MainWindow::onSDFIntersect() {
    QMessageBox::information(this, "SDF Intersect",
        "Intersection requires boolean mesh operations.\nUse Marching Cubes with SDF intersect function.");
}

void MainWindow::onSDFSmoothUnion() {
    if (currentMesh.getVertexCount() == 0) {
        QMessageBox::information(this, "No Mesh", "Generate a mesh first!");
        return;
    }

    Mesh sphere = MeshGenerator::createSphere(1.0f, 16, 16);
    sphere.transform(Vec3(0.8f, 0.8f, 0), Vec3(1, 1, 1), Vec3(0, 0, 0));

    currentMesh.merge(sphere);
    viewport->setMesh(currentMesh);
    updateWindowTitle();
}

void MainWindow::onSDFTwist() {
    if (currentMesh.getVertexCount() == 0) {
        QMessageBox::information(this, "No Mesh", "Generate a mesh first!");
        return;
    }

    QApplication::processEvents();
    // Apply twist deformation using vertices
    for (size_t i = 0; i < currentMesh.getVertexCount(); i++) {
        Vertex& v = currentMesh.getVertex(i);
        float angle = v.position.y * 0.5f;  // Twist amount
        float c = std::cos(angle);
        float s = std::sin(angle);

        Vec3 twisted;
        twisted.x = v.position.x * c - v.position.z * s;
        twisted.y = v.position.y;
        twisted.z = v.position.x * s + v.position.z * c;

        v.position = twisted;
    }

    QApplication::processEvents(); // Keep UI responsive
    currentMesh.computeNormals();
    viewport->setMesh(currentMesh);
    updateWindowTitle();
}

void MainWindow::onSDFBend() {
    if (currentMesh.getVertexCount() == 0) {
        QMessageBox::information(this, "No Mesh", "Generate a mesh first!");
        return;
    }

    QApplication::processEvents();
    // Apply bend deformation
    for (size_t i = 0; i < currentMesh.getVertexCount(); i++) {
        Vertex& v = currentMesh.getVertex(i);
        float bendAmount = 0.3f;
        float angle = v.position.y * bendAmount;

        Vec3 bent;
        bent.x = v.position.x * std::cos(angle) - v.position.y * std::sin(angle);
        bent.y = v.position.x * std::sin(angle) + v.position.y * std::cos(angle);
        bent.z = v.position.z;

        v.position = bent;
    }

    QApplication::processEvents();
    currentMesh.computeNormals();
    viewport->setMesh(currentMesh);
    updateWindowTitle();
}

void MainWindow::onSDFRepeat() {
    if (currentMesh.getVertexCount() == 0) {
        QMessageBox::information(this, "No Mesh", "Generate a mesh first!");
        return;
    }

    Mesh original = currentMesh;

    for (int x = -1; x <= 1; x++) {
        for (int z = -1; z <= 1; z++) {
            if (x == 0 && z == 0) continue;

            QApplication::processEvents();
            Mesh copy = original;
            copy.transform(Vec3(x * 3.0f, 0, z * 3.0f), Vec3(1, 1, 1), Vec3(0, 0, 0));
            currentMesh.merge(copy);
        }
    }

    viewport->setMesh(currentMesh);
    updateWindowTitle();
}

// ========== Mesh Operations ==========

void MainWindow::onSubdivide() {
    if (currentMesh.getVertexCount() == 0) {
        QMessageBox::information(this, "No Mesh", "Generate a mesh first!");
        return;
    }

    QApplication::processEvents();
    MeshGenerator::subdivide(currentMesh, 1);
    QApplication::processEvents();
    viewport->setMesh(currentMesh);
    updateWindowTitle();
}

void MainWindow::onExtrude() {
    if (currentMesh.getVertexCount() == 0) {
        QMessageBox::information(this, "No Mesh", "Generate a mesh first!");
        return;
    }

    MeshGenerator::extrude(currentMesh, Vec3(0, 1, 0), 2.0f);
    viewport->setMesh(currentMesh);
    updateWindowTitle();
}

void MainWindow::onLathe() {
    std::vector<Vec3> profile = {
        Vec3(0.5f, 0, 0),
        Vec3(1.0f, 0.3f, 0),
        Vec3(0.8f, 0.6f, 0),
        Vec3(0.5f, 1.0f, 0)
    };

    currentMesh = MeshGenerator::lathe(profile, 32);
    viewport->setMesh(currentMesh);
    updateWindowTitle();
}

void MainWindow::onLoft() {
    std::vector<std::vector<Vec3>> profiles;

    // Profile 1: Circle at y=0
    std::vector<Vec3> p1;
    for (int i = 0; i < 8; i++) {
        float angle = i * 2.0f * M_PI / 8.0f;
        p1.push_back(Vec3(std::cos(angle), 0, std::sin(angle)));
    }

    // Profile 2: Smaller circle at y=2
    std::vector<Vec3> p2;
    for (int i = 0; i < 8; i++) {
        float angle = i * 2.0f * M_PI / 8.0f;
        p2.push_back(Vec3(0.5f * std::cos(angle), 2, 0.5f * std::sin(angle)));
    }

    profiles.push_back(p1);
    profiles.push_back(p2);

    currentMesh = MeshGenerator::loft(profiles);
    viewport->setMesh(currentMesh);
    updateWindowTitle();
}

void MainWindow::onSmooth() {
    if (currentMesh.getVertexCount() == 0) {
        QMessageBox::information(this, "No Mesh", "Generate a mesh first!");
        return;
    }

    // Simple Laplacian smoothing
    Mesh smoothed = currentMesh;

    for (int iter = 0; iter < 3; iter++) {
        QApplication::processEvents();
        for (size_t i = 0; i < smoothed.getVertexCount(); i++) {
            Vertex& v = smoothed.getVertex(i);
            Vec3 avgPos = v.position * 0.5f;  // Keep some original position

            // Average with neighbors (simplified)
            if (i > 0) avgPos = avgPos + smoothed.getVertex(i - 1).position * 0.25f;
            if (i < smoothed.getVertexCount() - 1) avgPos = avgPos + smoothed.getVertex(i + 1).position * 0.25f;

            v.position = avgPos;
        }
    }

    QApplication::processEvents();
    smoothed.computeNormals();
    currentMesh = smoothed;
    viewport->setMesh(currentMesh);
    updateWindowTitle();
}

// ========== Procedural Animation Effects ==========

void MainWindow::onApplyWave() {
    if (currentMesh.getVertexCount() == 0) {
        QMessageBox::information(this, "No Mesh", "Generate a mesh first!");
        return;
    }

    animator.animateWave(currentMesh, animationTime, 0.5f, 2.0f);
    viewport->setMesh(currentMesh);
    updateWindowTitle();
}

void MainWindow::onApplyTwist() {
    if (currentMesh.getVertexCount() == 0) {
        QMessageBox::information(this, "No Mesh", "Generate a mesh first!");
        return;
    }

    animator.animateTwist(currentMesh, animationTime, 1.0f);
    viewport->setMesh(currentMesh);
    updateWindowTitle();
}

void MainWindow::onApplyBend() {
    if (currentMesh.getVertexCount() == 0) {
        QMessageBox::information(this, "No Mesh", "Generate a mesh first!");
        return;
    }

    animator.animateBend(currentMesh, animationTime, 0.5f);
    viewport->setMesh(currentMesh);
    updateWindowTitle();
}

void MainWindow::onApplyPulse() {
    if (currentMesh.getVertexCount() == 0) {
        QMessageBox::information(this, "No Mesh", "Generate a mesh first!");
        return;
    }

    animator.animatePulse(currentMesh, animationTime, 0.3f);
    viewport->setMesh(currentMesh);
    updateWindowTitle();
}

void MainWindow::onApplyNoise() {
    if (currentMesh.getVertexCount() == 0) {
        QMessageBox::information(this, "No Mesh", "Generate a mesh first!");
        return;
    }

    animator.animateNoise(currentMesh, animationTime, 0.2f);
    viewport->setMesh(currentMesh);
    updateWindowTitle();
}

// ========== Helper Methods for Mesh Morphing ==========

// Match vertex counts between two meshes by subdividing the mesh with fewer vertices
void MainWindow::matchVertexCounts(Mesh& meshA, Mesh& meshB) {
    size_t countA = meshA.getVertexCount();
    size_t countB = meshB.getVertexCount();

    if (countA == countB) {
        return; // Already matched
    }

    // Determine which mesh needs more vertices
    Mesh* smallerMesh = (countA < countB) ? &meshA : &meshB;
    size_t targetCount = std::max(countA, countB);
    size_t currentCount = std::min(countA, countB);

    // Subdivide until we get close to the target count
    while (currentCount < targetCount) {
        size_t prevCount = smallerMesh->getVertexCount();
        MeshGenerator::subdivide(*smallerMesh, 1);
        currentCount = smallerMesh->getVertexCount();

        // If subdividing didn't change vertex count or we're close enough, stop
        if (currentCount == prevCount || currentCount >= targetCount * 0.9) {
            break;
        }
    }
}

// Morph between two meshes based on morphPercent (0-100)
Mesh MainWindow::morphMeshes(const Mesh& meshA, const Mesh& meshB, float morphPercent) {
    Mesh result;

    // Clamp morphPercent to 0-100 range
    morphPercent = std::max(0.0f, std::min(100.0f, morphPercent));
    float t = morphPercent / 100.0f; // Normalize to 0-1

    size_t vertexCount = std::min(meshA.getVertexCount(), meshB.getVertexCount());

    // Interpolate vertices
    for (size_t i = 0; i < vertexCount; i++) {
        const Vertex& vA = meshA.getVertex(i);
        const Vertex& vB = meshB.getVertex(i);

        Vertex morphedVertex;

        // Linear interpolation of position
        morphedVertex.position.x = vA.position.x * (1.0f - t) + vB.position.x * t;
        morphedVertex.position.y = vA.position.y * (1.0f - t) + vB.position.y * t;
        morphedVertex.position.z = vA.position.z * (1.0f - t) + vB.position.z * t;

        // Linear interpolation of normal
        morphedVertex.normal.x = vA.normal.x * (1.0f - t) + vB.normal.x * t;
        morphedVertex.normal.y = vA.normal.y * (1.0f - t) + vB.normal.y * t;
        morphedVertex.normal.z = vA.normal.z * (1.0f - t) + vB.normal.z * t;
        morphedVertex.normal.normalize();

        // Interpolate color if present
        morphedVertex.color.x = vA.color.x * (1.0f - t) + vB.color.x * t;
        morphedVertex.color.y = vA.color.y * (1.0f - t) + vB.color.y * t;
        morphedVertex.color.z = vA.color.z * (1.0f - t) + vB.color.z * t;

        result.addVertex(morphedVertex);
    }

    // Copy triangles from meshA (topology remains from first mesh)
    size_t triangleCount = meshA.getTriangleCount();
    for (size_t i = 0; i < triangleCount; i++) {
        Triangle t = meshA.getTriangle(i);
        // Only add triangle if all vertices exist in the morphed mesh
        if (t.v0 < vertexCount && t.v1 < vertexCount && t.v2 < vertexCount) {
            result.addTriangle(t.v0, t.v1, t.v2);
        }
    }

    result.computeNormals();
    return result;
}
