/**
 * @file MainWindow.h
 * @brief Main application window for Mesher - Parametric Modeler
 *
 * This header defines the MainWindow class which serves as the primary user interface
 * for the parametric modeling application. It manages the algorithm selection, parameter
 * controls, 3D viewport, algorithm stack, and various mesh operations.
 *
 * Key features:
 * - 127+ parametric mesh generation algorithms
 * - Real-time parameter adjustment with sliders
 * - Algorithm stack for sequential operations
 * - SDF (Signed Distance Field) boolean operations
 * - Vertex animation system
 * - Multiple export formats
 *
 * Copyright © 2025 Linus Suter
 * Released under the GNU/GPL License
 */

#pragma once

// Qt Includes
#include <QMainWindow>
#include <QComboBox>
#include <QSlider>
#include <QLabel>
#include <QPushButton>
#include <QVBoxLayout>
#include <QHBoxLayout>
#include <QGroupBox>
#include <QCheckBox>
#include <QScrollArea>
#include <QTimer>
#include <QListWidget>
#include <QLineEdit>
#include <QKeyEvent>
#include <vector>

// Application Includes
#include "GLViewport.h"
#include "algorithms/AlgorithmLibrary.h"
#include "algorithms/AlgorithmRegistry.h"
#include "algorithms/FractalGenerators.h"
#include "algorithms/PatternGenerator.h"
#include "animation/VertexAnimator.h"
#include "sdf/SDFEngine.h"
#include "sdf/SDFPrimitives.h"
#include "mesh/MeshGenerator.h"
#include "voxel/VoxelGenerator.h"
#include "optimization/CMAESOptimizer.h"
#include "optimization/NSGAIIOptimizer.h"
#include "optimization/RBFOptimizer.h"
#include "neuralnet/NeuralNetwork.h"
#include "neuralnet/CPPNHyperNEAT.h"

#include "mesh/Mesh.h"
#include "animation/VertexAnimator.h"

/**
 * @brief Structure to hold an algorithm stack entry
 *
 * Each entry in the algorithm stack contains the algorithm type,
 * its parameters, and a display name for the UI.
 */
struct StackEntry {
    AlgorithmLibrary::AlgorithmType algorithmType;  ///< The algorithm to execute
    std::vector<AlgorithmParameter> parameters;      ///< Algorithm parameters and values
    QString name;                                     ///< Display name for the stack list
};

/**
 * @brief Main application window class
 *
 * Manages the entire user interface including algorithm selection,
 * parameter controls, 3D viewport, and various mesh operations.
 */
class MainWindow : public QMainWindow {
    Q_OBJECT

public:
    /**
     * @brief Constructor - Creates the main window
     * @param parent Parent widget (nullptr for top-level window)
     */
    explicit MainWindow(QWidget *parent = nullptr);

    /**
     * @brief Destructor - Cleans up resources
     */
    ~MainWindow();

private slots:
    // ===== UI Event Handlers =====

    /** @brief Handles algorithm selection changes in the list */
    void onAlgorithmChanged(int index);

    /** @brief Handles search box text changes for algorithm filtering */
    void onAlgorithmSearchChanged(const QString& text);

    /** @brief Handles search box submission (Enter key) */
    void onAlgorithmSearchSubmitted();

    /** @brief Handles parameter slider value changes */
    void onParameterChanged();

    /** @brief Handles Generate button click */
    void onGenerateClicked();

    /** @brief Handles mesh export button click */
    void onExportClicked();

    /** @brief Handles wireframe toggle checkbox */
    void onWireframeToggled(bool checked);

    /** @brief Handles render mode combo box changes */
    void onRenderModeChanged(int index);

    /** @brief Handles Frame All checkbox toggle */
    void onFrameAllToggled(bool checked);

    /** @brief Animation timer callback for vertex animations */
    void onAnimationTimerUpdate();

    // ===== Algorithm Stack Operations =====

    /** @brief Adds current algorithm to the stack */
    void onAddToStack();

    /** @brief Removes selected item from stack */
    void onRemoveFromStack();

    /** @brief Clears all items from stack */
    void onClearStack();

    /** @brief Moves selected stack item up */
    void onMoveUp();

    /** @brief Moves selected stack item down */
    void onMoveDown();

    /** @brief Randomly shuffles stack order */
    void onShuffleStack();

    /** @brief Generates mesh by applying all stack algorithms */
    void onGenerateFromStack();

    /** @brief Randomizes all parameter values */
    void onRandomizeSettings();

    /** @brief Opens edit dialog for stack item */
    void onEditStackItem(QListWidgetItem* item);

    // ===== SDF/Boolean Operations =====

    /** @brief SDF Union operation */
    void onSDFUnion();

    /** @brief SDF Subtraction operation */
    void onSDFSubtract();

    /** @brief SDF Intersection operation */
    void onSDFIntersect();

    /** @brief SDF Smooth Union operation */
    void onSDFSmoothUnion();

    /** @brief SDF Twist deformation */
    void onSDFTwist();

    /** @brief SDF Bend deformation */
    void onSDFBend();

    /** @brief SDF Repeat/Array operation */
    void onSDFRepeat();

    /** @brief Marching cubes voxelization */
    void onMarchingCubes();

    // ===== Mesh Operations =====

    /** @brief Subdivides mesh (adds detail) */
    void onSubdivide();

    /** @brief Extrudes mesh faces */
    void onExtrude();

    /** @brief Creates mesh by revolution (lathe) */
    void onLathe();

    /** @brief Creates mesh by lofting between profiles */
    void onLoft();

    /** @brief Smooths mesh using Laplacian smoothing */
    void onSmooth();

    // ===== Vertex Animation Operations =====

    /** @brief Applies wave animation to vertices */
    void onApplyWave();

    /** @brief Applies twist animation to vertices */
    void onApplyTwist();

    /** @brief Applies bend animation to vertices */
    void onApplyBend();

    /** @brief Applies pulse animation to vertices */
    void onApplyPulse();

    /** @brief Applies noise animation to vertices */
    void onApplyNoise();

protected:
    /** @brief Handles keyboard shortcuts (Space, Plus, Minus keys) */
    void keyPressEvent(QKeyEvent *event) override;

    /** @brief Event filter for handling list widget keyboard events */
    bool eventFilter(QObject *obj, QEvent *event) override;

private:
    // ===== UI Setup Methods =====
    /** @brief Creates and initializes all UI components */
    void createUI();

    /** @brief Creates the application menu bar */
    void createMenuBar();

    /** @brief Sets up parameter sliders for selected algorithm */
    void setupParameterControls();

    /** @brief Regenerates mesh with current algorithm and parameters */
    void regenerateMesh();

    /** @brief Updates window title with mesh statistics */
    void updateWindowTitle();

    // ===== UI Components =====

    GLViewport *viewport;              ///< 3D OpenGL rendering viewport
    QListWidget *algorithmList;        ///< List of available algorithms
    QLineEdit *algorithmSearchBox;     ///< Search/filter box for algorithms
    QComboBox *renderModeCombo;        ///< Render mode selector (wireframe, solid, etc.)
    QWidget *parameterPanel;           ///< Container for parameter controls
    QVBoxLayout *parameterLayout;      ///< Layout for parameter sliders
    QPushButton *generateButton;       ///< Generate mesh button
    QCheckBox *wireframeCheck;         ///< Wireframe toggle (deprecated)
    QCheckBox *frameAllCheck;          ///< Auto-frame mesh toggle
    QLabel *algoDescLabel;             ///< Algorithm description label

    // Algorithm Stack UI Components
    QGroupBox *stackGroupBox;          ///< Stack group box container
    QListWidget *stackList;            ///< List showing algorithm stack
    QComboBox *artPatternCombo;        ///< Preset art pattern selector
    QPushButton *addToStackButton;     ///< Add algorithm to stack button
    QPushButton *removeStackButton;    ///< Remove from stack button
    QPushButton *clearStackButton;     ///< Clear entire stack button
    QPushButton *moveUpButton;         ///< Move item up in stack
    QPushButton *moveDownButton;       ///< Move item down in stack
    QPushButton *shuffleButton;        ///< Shuffle stack order button
    QPushButton *generateStackButton;  ///< Generate from stack button
    QPushButton *randomizeButton;      ///< Randomize parameters button
    QPushButton *saveButton;           ///< Save/export mesh button
    QSlider *morphSlider;              ///< Morph between meshes slider

    // ===== Algorithm Management =====

    AlgorithmLibrary library;                                    ///< Algorithm library instance
    AlgorithmLibrary::AlgorithmType currentAlgorithm;           ///< Currently selected algorithm
    std::vector<QSlider*> parameterSliders;                     ///< Parameter slider widgets
    std::vector<QLabel*> parameterLabels;                       ///< Parameter value labels
    std::vector<AlgorithmLibrary::AlgorithmType> indexToAlgorithmMap;  ///< Maps UI index to algorithm type
    std::vector<StackEntry> algorithmStack;                     ///< Stack of algorithms to execute

    // ===== Mesh Data =====

    Mesh currentMesh;      ///< Current active mesh
    Mesh sdfMeshA;         ///< First mesh for SDF boolean operations
    Mesh sdfMeshB;         ///< Second mesh for SDF boolean operations

    // ===== Animation System =====

    VertexAnimator animator;     ///< Vertex animation controller
    QTimer *animationTimer;      ///< Timer for animation updates
    float animationTime;         ///< Current animation time in seconds

    // ===== Advanced Systems =====

    SDFEngine sdfEngine;         ///< Signed Distance Field engine for boolean ops

    // ===== Helper Methods =====

    /** @brief Updates the algorithm stack UI display */
    void updateStackUI();

    /** @brief Gets current parameter values from UI sliders */
    std::vector<AlgorithmParameter> getCurrentParameters();

    /** @brief Matches vertex counts between two meshes for morphing */
    void matchVertexCounts(Mesh& meshA, Mesh& meshB);

    /** @brief Interpolates between two meshes based on morph percentage */
    Mesh morphMeshes(const Mesh& meshA, const Mesh& meshB, float morphPercent);
};
