#pragma once
#include <QString>
#include <QMap>

class AlgorithmDescriptions {
public:
    static QString getDetailedDescription(const QString& algorithmName) {
        static QMap<QString, QString> descriptions;

        if (descriptions.isEmpty()) {
            // PRIMITIVES
            descriptions["Sphere"] =
                "A UV sphere is created by tessellating along meridians (lines from pole to pole) and parallels (lines parallel to the equator). "
                "This method produces quad faces with triangle fans at the poles. While not the most geometrically accurate method for a given triangle count, "
                "it provides excellent UV mapping properties for texture application. The sphere is parametrically defined using spherical coordinates "
                "(θ, φ) where vertices are positioned at (r·sin(φ)·cos(θ), r·sin(φ)·sin(θ), r·cos(φ)). This makes it ideal for rendering planets, "
                "spherical objects, and any application requiring straightforward texture mapping.";

            descriptions["Cube"] =
                "The cube (hexahedron) is one of the five Platonic solids and the most fundamental primitive in 3D graphics. It consists of 6 square faces, "
                "12 edges, and 8 vertices, with each vertex being equidistant from the center. The cube exhibits perfect cubic symmetry and serves as the "
                "building block for voxel-based representations, minecraft-style geometry, and architectural modeling. Its simple topology and regular "
                "structure make it computationally efficient for collision detection, spatial partitioning (octrees), and boolean operations.";

            descriptions["Torus"] =
                "A torus (donut shape) is a surface of revolution generated by rotating a circle around an axis coplanar with the circle. Mathematically "
                "defined by two radii: the major radius R (distance from torus center to tube center) and minor radius r (tube radius). Parametric equations: "
                "x = (R + r·cos(v))·cos(u), y = (R + r·cos(v))·sin(u), z = r·sin(v), where u and v are angular parameters. Torii exhibit interesting "
                "topological properties with genus 1 (one hole) and are used in computer graphics for modeling rings, pipes, and architectural elements.";

            descriptions["Cylinder"] =
                "A right circular cylinder consists of two parallel circular bases connected by a curved lateral surface. The cylinder is defined by its "
                "radius r and height h, with the lateral surface parametrized as x = r·cos(θ), y = r·sin(θ), z = t where θ ∈ [0, 2π] and t ∈ [0, h]. "
                "Cylinders are fundamental primitives in CAD modeling, used for shafts, pipes, columns, and mechanical components. The cylindrical coordinate "
                "system provides natural UV mapping for the lateral surface, making texture application straightforward.";

            descriptions["Cone"] =
                "A right circular cone is formed by connecting a circular base to an apex point directly above the center. Defined by base radius r and "
                "height h, with the lateral surface tapering linearly from base to apex. The cone's slant height is √(r² + h²). Parametric form: "
                "x = r(1-t)·cos(θ), y = r(1-t)·sin(θ), z = h·t where t ∈ [0,1]. Cones are used in 3D graphics for traffic cones, funnels, arrows, "
                "and spotlight volumes. The apex creates a singularity requiring special handling in normal calculations.";

            descriptions["Plane"] =
                "A subdivided plane is a flat, two-dimensional surface embedded in 3D space, typically aligned with the XY, XZ, or YZ coordinate planes. "
                "Unlike a simple quad, a subdivided plane contains multiple segments in both dimensions, creating a regular grid of smaller quads or triangles. "
                "This subdivision is essential for applications requiring vertex-level deformation, displacement mapping, cloth simulation, or terrain generation. "
                "The plane serves as the foundation for height field terrains, water surfaces, and ground planes in scenes.";

            descriptions["Superellipsoid"] =
                "A superellipsoid (or superquadric) is a generalization of the ellipsoid where the implicit equation |x/a|^n + |y/b|^n + |z/c|^n = 1 "
                "includes exponents n and e that control the 'squareness' of the shape. When n=e=2, it's a standard ellipsoid; as n,e→∞, it approaches "
                "a box; as n,e→0, it becomes more star-like. These shapes were introduced by Barr (1981) and are valuable in computer graphics for modeling "
                "objects that transition between rounded and cubic forms, such as cushions, boulders, and organic shapes with controlled roundness.";

            descriptions["Supertoroid"] =
                "A supertoroid extends the torus concept by applying superquadric equations to both the main body and tube cross-section. It's defined by "
                "four parameters: major radius R, minor radius r, and two exponents that control squareness in the poloidal and toroidal directions. "
                "This allows creation of shapes ranging from round donuts to square/rectangular tori. Supertoroids are useful in industrial design for "
                "modeling gaskets, O-rings, and architectural elements that require controlled corner sharpness while maintaining toroidal topology.";

            descriptions["Teapot"] =
                "The Utah Teapot, created by Martin Newell in 1975, is one of the most iconic models in computer graphics history. This mathematical "
                "representation of a teapot consists of Bézier surface patches describing the body, spout, handle, and lid. It has become the standard "
                "reference model for testing rendering algorithms, ray tracing, and shading techniques. The teapot's complex curved surfaces with varying "
                "curvature make it ideal for evaluating highlight distribution, reflection mapping, and tessellation quality. It's essentially the 'Hello World' "
                "of 3D graphics.";

            descriptions["Oil Tank"] =
                "An industrial oil storage tank is a cylindrical vessel with hemispherical or ellipsoidal end caps (heads), supported by legs or a foundation. "
                "The design combines a long cylindrical section for volume efficiency with domed ends that better distribute internal pressure. Typical features "
                "include support legs at cardinal points, access ladders, and connection flanges. This geometry is common in petroleum storage, chemical "
                "processing, and industrial facilities. The cylindrical body provides optimal volume-to-surface-area ratio, while the curved ends eliminate "
                "stress concentrations found in flat end caps.";

            descriptions["GeoSphere"] =
                "Geodesic sphere with icosahedral symmetry - created by subdividing an icosahedron and projecting vertices onto a sphere surface. Provides nearly uniform triangle distribution superior to UV spheres, ideal for low-poly modeling, game development, and scientific visualization. Used in geodesic domes, climate modeling, and virus capsid structures";

            descriptions["Tube"] =
                "Hollow cylindrical tube with adjustable wall thickness - essential primitive for architectural modeling, piping systems, and structural design. Objects created include pipes, drinking glasses, rings, and architectural elements. Features inner and outer cylindrical surfaces with controllable thickness for realistic hollow geometry";

            descriptions["Pyramid"] =
                "Classic square-based pyramid primitive with apex point - fundamental 3D shape used extensively in architectural design, game level prototyping, and geometric modeling. Architects use pyramid primitives for initial concept visualization and spatial planning. Features a square base with four triangular faces converging at a single apex";

            descriptions["Hedra"] =
                "General polyhedron generator supporting multiple Platonic solids based on face count: 4 faces (tetrahedron), 8 faces (octahedron), 12 faces (dodecahedron), 20 faces (icosahedron). 'Hedra' means face in Greek. Features subdivision control for creating more complex geometric forms from basic polyhedra, useful for scientific visualization and sacred geometry";

            descriptions["Torus Knot"] =
                "Parametric (p,q)-torus knot where p defines longitude windings and q defines meridian windings - creates beautiful mathematical knot topology following parametric equations. P winds around the torus tube, Q winds around the ring. Ideal for jewelry design including rings and pendants, decorative elements, and mathematical visualization of knot theory";

            descriptions["Capsule"] =
                "Pill-shaped capsule combining cylindrical body with hemispherical end caps - widely used in game development for character collision detection due to smooth, round circumference that prevents snagging on sharp corners. Better choice than spheres or boxes for representing game entities. Features seamless blend between cylinder and sphere geometry";

            descriptions["Spindle"] =
                "Double cone shape with apex points at top and bottom joined at their bases - resembles shapes created on a lathe through turning operations. Useful for rotation bodies, decorative turned elements, and symmetric objects with bilateral tapering. Common in architectural balusters, furniture legs, and ornamental designs";

            descriptions["L-Ext"] =
                "L-shaped aluminum/steel extrusion profile forming 90-degree angle - versatile structural element used across all industrial sectors. Applications include structural beams offering support and stability, architectural molding and trim, construction framework reinforcements, and corner supports. Available in equal and unequal leg configurations";

            descriptions["C-Ext"] =
                "C-shaped channel extrusion with tapered flanges - manufactured through hot rolling for structural applications. Two C-channels bolted back-to-back form an I-beam equivalent. Used extensively in construction framing, structural steel supports, tracks and sliders for machinery, and architectural profiles. Features high strength-to-weight ratio";

            descriptions["Gengon"] =
                "General polygon extrusion with filleted side edges - creates regular-sided polygonal prism ideal for mechanical modeling. Used extensively for creating hexagonal nuts, bolt heads, polygonal columns, and architectural elements. Fillet segments control edge rounding: low values create chamfers, high values create smooth fillets";

            descriptions["RingWave"] =
                "Torus with sinusoidal wave deformation applied radially - combines mathematical wave functions with toroidal geometry. Creates organic wave patterns flowing around the ring circumference, ideal for decorative jewelry including wave-themed wedding bands, bracelets with rhythm patterns, and mathematical art incorporating wave phenomena";

            descriptions["Hose"] =
                "Flexible bent tube with controllable bend angle - simulates cables, wires, and flexible piping in mechanical assemblies. Features swept tube cross-section following curved path with adjustable curvature. Essential for routing designs including hydraulic hoses, electrical cables, flexible connections, and any bent tubular geometry requiring realistic flexibility";

            descriptions["Prism"] =
                "Polygonal prism with customizable base polygon and height - fundamental 3D shape comprising parallel polygonal bases connected by rectangular faces. Named by base shape: triangular prism, pentagonal prism, hexagonal prism. Used in crystal modeling, architectural columns, optical prisms, and geometric design. Features in office buildings and structural designs";

            // PLATONIC SOLIDS
            descriptions["Tetrahedron"] =
                "The tetrahedron is the simplest Platonic solid, consisting of 4 equilateral triangular faces, 6 edges, and 4 vertices. Each vertex connects "
                "to exactly 3 edges. With tetrahedral symmetry (Td), it's the 3D simplex and exhibits fascinating mathematical properties: it's self-dual "
                "(its dual is another tetrahedron), has dihedral angle ≈70.53°, and inscribes/circumscribes with radius ratio 1:3. Tetrahedra are fundamental "
                "in FEM (Finite Element Method) analysis, molecular chemistry (methane CH4 geometry), crystallography, and mesh generation for volumetric data.";

            descriptions["Octahedron"] =
                "The regular octahedron contains 8 equilateral triangular faces, 12 edges, and 6 vertices, with 4 triangles meeting at each vertex. It's the "
                "dual polyhedron of the cube—vertices of the octahedron correspond to face centers of the cube and vice versa. The octahedron has octahedral "
                "symmetry (Oh) and appears in crystal structures (diamond, fluorite), as the standard d8 die in gaming, and in subdivision surface algorithms. "
                "Its high symmetry and regular triangulation make it ideal for icosphere generation via subdivision.";

            descriptions["Icosahedron"] =
                "The regular icosahedron is composed of 20 equilateral triangular faces, 30 edges, and 12 vertices, with 5 triangles meeting at each vertex. "
                "It has the largest volume-to-surface-area ratio of any Platonic solid, making it the best polyhedral approximation of a sphere. The icosahedron "
                "exhibits icosahedral symmetry (Ih) and is fundamental in virology (many viral capsids have icosahedral symmetry), geodesic dome construction "
                "(Buckminster Fuller's designs), and as the starting point for generating icospheres through recursive subdivision. Its dihedral angle is ≈138.19°.";

            descriptions["Dodecahedron"] =
                "The regular dodecahedron consists of 12 regular pentagonal faces, 30 edges, and 20 vertices, with 3 pentagons meeting at each vertex. "
                "It exhibits icosahedral symmetry (Ih) and is the dual of the icosahedron. The dodecahedron appears in crystallography (pyrite crystals), "
                "as the standard d12 gaming die, and has historical significance in Greek philosophy where it represented the universe/aether. Its dihedral "
                "angle is ≈116.57°. The pentagonal faces create interesting tessellation challenges and make it useful for spherical approximations.";

            descriptions["Geodesic Sphere"] =
                "A geodesic sphere is created by subdividing the faces of an icosahedron and projecting the resulting vertices onto a sphere. This method, "
                "popularized by Buckminster Fuller for dome construction, produces near-uniform triangle distributions superior to UV spheres. Each subdivision "
                "level (frequency) increases triangle count by 4x while improving sphericity. Geodesic spheres exhibit better mesh quality with more uniform "
                "edge lengths and face areas, making them ideal for planet rendering, physics simulations, and any application requiring isotropic vertex "
                "distribution. The triangulation pattern creates natural structural integrity for architectural domes.";

            // PARAMETRIC SURFACES
            descriptions["Spiral Sphere"] =
                "A spiral sphere combines spherical geometry with spiral patterns, creating undulating surfaces through modulated radius or displaced vertices "
                "following helical paths. The parametric form typically modulates the standard sphere equation with sinusoidal functions: r(θ,φ) = R(1 + A·sin(nθ + mφ)) "
                "where n and m control spiral frequency and A controls amplitude. This creates organic, shell-like forms reminiscent of nautilus patterns or "
                "fractal spheres. Applications include decorative modeling, organic architecture, and biomimetic design inspired by natural growth patterns.";

            descriptions["Twisted Torus"] =
                "A twisted torus applies a twist deformation along the major circumference of a standard torus, creating Möbius-like surface characteristics "
                "without changing topology. The twist angle θ_twist varies linearly or non-linearly around the major axis. Parametrically: the tube rotation "
                "parameter v is modified as v' = v + θ_twist·u where u is the major circumference parameter. This produces interesting self-intersecting "
                "forms for specified twist values, demonstrates topological transformations, and creates visually engaging sculptural geometry useful in "
                "mathematical visualization and architectural design.";

            descriptions["Wave Surface"] =
                "A parametric wave surface is generated by applying sinusoidal displacement to a base plane or surface: z = A·sin(kx·x + φx)·sin(ky·y + φy) "
                "where A is amplitude, kx and ky are spatial frequencies, and φx, φy are phase offsets. This creates periodic undulating surfaces resembling "
                "water waves, standing waves, or interference patterns. Wave surfaces are fundamental in modeling ocean surfaces, acoustic phenomena, "
                "architectural facades with wave motifs, and mathematical visualization of harmonic functions. Multiple wave components can be superimposed "
                "for complex terrain generation.";

            descriptions["Spring"] =
                "A helical spring (coil spring) follows a 3D helix path while sweeping a circular cross-section. The parametric form combines helical motion "
                "x = R·cos(θ), y = R·sin(θ), z = p·θ (where p is pitch) with circular tube geometry. This creates the characteristic coiled shape with "
                "parameters controlling outer radius, wire diameter, pitch (spacing between coils), and number of turns. Springs are essential mechanical "
                "components used in CAD modeling, physics simulations, and demonstrate the mathematical relationship between helices, torque, and elastic "
                "deformation in mechanical engineering applications.";

            descriptions["Lissajous"] =
                "Lissajous curves (Bowditch curves) are generated by parametric equations combining perpendicular sinusoidal oscillations: x = A·sin(at + δ), "
                "y = B·sin(bt), z = C·sin(ct + φ). The ratios a:b:c and phase differences δ, φ determine the curve's shape, creating symmetric patterns "
                "ranging from simple ellipses (1:1) to complex interwoven forms (3:2, 5:4, etc.). Named after Jules Antoine Lissajous (1857), these curves "
                "visualize harmonic relationships, appear in oscilloscope displays, and model coupled oscillations in physics, audio synthesis, and "
                "mathematical art. 3D extensions create spatial knot-like structures.";

            descriptions["Rose"] =
                "Rose curves (rhodonea curves) are sinusoidal curves resembling flower petals, defined in polar form as r = a·cos(kθ) or r = a·sin(kθ). "
                "The number of petals is k (if k odd) or 2k (if k even). In 3D, the rose pattern can be swept, extruded, or modulated to create rose surfaces "
                "with petal-like structures: x = r(θ)·cos(θ)·f(z), y = r(θ)·sin(θ)·f(z) where f(z) modulates petal shape vertically. These elegant mathematical "
                "forms are used in decorative design, architectural ornamentation, botanical visualization, and mathematical art exploring the relationship "
                "between trigonometry and natural patterns.";

            descriptions["Helix"] =
                "A helix is a 3D curve spiraling around an axis at constant radius while moving parallel to that axis. The standard circular helix parametrically: "
                "x = R·cos(t), y = R·sin(t), z = p·t where R is radius, p is pitch (rise per 2π radians), and t is the parameter. The helix has constant "
                "curvature κ = R/(R² + p²) and torsion τ = p/(R² + p²). Helices appear throughout nature (DNA double helix, snail shells, climbing vines) "
                "and engineering (screw threads, springs, spiral staircases). They're fundamental in swept surface generation and demonstrate the relationship "
                "between curvature, torsion, and 3D space curves.";

            descriptions["Lathe"] =
                "Surface of revolution (lathe operation) generates 3D surfaces by rotating a 2D profile curve around an axis. The profile curve C(t) = (r(t), z(t)) "
                "in the rz-plane is revolved through angle θ to create surface S(t,θ) = (r(t)·cos(θ), r(t)·sin(θ), z(t)). This technique creates axially "
                "symmetric objects like vases, bottles, columns, and mechanical parts. The lathe is fundamental in CAD/CAM, pottery design, and woodturning "
                "visualization. Profile curves can be defined as Bézier curves, splines, or polylines. The resulting surface naturally has cylindrical UV "
                "coordinates perfect for texture mapping.";

            descriptions["Loft"] =
                "Lofting creates smooth surfaces by blending between multiple profile curves (cross-sections) positioned along a path. Unlike extrusion which "
                "maintains constant cross-section, lofting interpolates between different profiles using techniques like skinning, ruled surfaces, or NURBS. "
                "Each profile curve is parameterized C_i(u) and blending functions B_i(v) interpolate: S(u,v) = Σ B_i(v)·C_i(u). This technique is essential "
                "in aircraft fuselage design, ship hull construction, automotive body panels, and character modeling. Lofting provides designer control over "
                "shape evolution along the path while maintaining smooth, high-quality surfaces.";

            // TOPOLOGY
            descriptions["Möbius Strip"] =
                "The Möbius strip is a non-orientable surface with only one side and one boundary, created by taking a rectangular strip, giving it a half-twist "
                "(180°), and joining the ends. Discovered independently by Möbius and Listing in 1858, it has remarkable properties: traversing the centerline "
                "returns to the starting point with reversed orientation, cutting along the centerline creates a longer strip with two full twists, and it has "
                "Euler characteristic χ = 0. The Möbius strip demonstrates fundamental concepts in topology, appears in conveyor belt design (even wear), and "
                "inspired art (Escher) and architecture. It's the simplest example of a non-orientable surface embeddable in 3D space.";

            descriptions["Klein Bottle"] =
                "The Klein bottle is a non-orientable closed surface with no inside or outside, requiring 4D space for true embedding but can be immersed in "
                "3D with self-intersections. It's constructed by joining the ends of a Möbius strip or by connecting two Möbius strips along their boundaries. "
                "The Klein bottle has Euler characteristic χ = 0 and genus 1 in non-orientable terms. Parametric immersion: x = (r + cos(u/2)sin(v) - sin(u/2)sin(2v))cos(u), "
                "y = (r + cos(u/2)sin(v) - sin(u/2)sin(2v))sin(u), z = sin(u/2)sin(v) + cos(u/2)sin(2v). This topological oddity illustrates advanced concepts "
                "in differential geometry and has connections to quantum field theory.";

            descriptions["Trefoil Knot"] =
                "The trefoil knot is the simplest non-trivial knot with crossing number 3, appearing as a closed curve in 3D that cannot be untangled into a "
                "circle without cutting. It has two chiral forms (left and right-handed) that are mirror images but not topologically equivalent. Parametric "
                "representation: x = sin(t) + 2sin(2t), y = cos(t) - 2cos(2t), z = -sin(3t) for t ∈ [0,2π]. The trefoil is the (3,2)-torus knot and appears "
                "in Celtic art, protein folding, DNA structures, and quantum physics. In knot theory, it's the first entry in knot tables and fundamental to "
                "understanding knot invariants like the Jones polynomial.";

            descriptions["Figure-Eight Knot"] =
                "The figure-eight knot (4₁ knot in knot tables) has crossing number 4 and is the unique knot with that crossing number. Unlike the trefoil, "
                "it's achiral (amphichiral)—identical to its mirror image. Parametric form: x = (2 + cos(2t))cos(3t), y = (2 + cos(2t))sin(3t), z = sin(4t). "
                "The figure-eight knot has hyperbolic geometry in its complement, making it fundamental in hyperbolic knot theory. It appears in climbing and "
                "sailing as a stopper knot, demonstrates important mathematical properties including being fibered, and its complement has finite volume in "
                "hyperbolic 3-space. The knot's symmetry and relatively simple structure make it a key example in knot theory research.";

            descriptions["Heart"] =
                "Mathematical heart curves create the iconic heart shape using various equations. One popular 3D parametric form: x = 16sin³(t), "
                "y = 13cos(t) - 5cos(2t) - 2cos(3t) - cos(4t), z = 13sin(t) + A·cos(φ) for generating a 3D heart surface. Another implicit form: "
                "(x² + 9/4·y² + z² - 1)³ - x²z³ - 9/80·y²z³ = 0 creates a smooth heart-shaped surface. These mathematical representations are used in "
                "visualization, romantic graphics, medical imaging (cardiac modeling), and demonstrate how complex organic shapes can be represented "
                "by elegant equations. The curves combine circular and polynomial components to achieve the characteristic heart profile.";

            // MINIMAL SURFACES
            descriptions["Hyperbolic Paraboloid"] =
                "The hyperbolic paraboloid (hypar or saddle surface) is a doubly-ruled quadric surface with equation z = (x²/a²) - (y²/b²), exhibiting "
                "negative Gaussian curvature (saddle-shaped). It's remarkable for being a minimal surface (zero mean curvature) and doubly-ruled (through "
                "every point pass two straight lines lying entirely on the surface). This property enables construction from straight elements, making it "
                "structurally efficient in architecture (Sagrada Família, cooling towers, roofs). The hypar balances compressive and tensile forces naturally, "
                "and its anticlastic curvature provides stiffness while using minimal material. It appears in thin-shell structures, Pringles chips geometry, "
                "and demonstrates the intersection of mathematics, engineering, and aesthetics.";

            descriptions["Gyroid (Surfels)"] =
                "The gyroid is a triply periodic minimal surface (TPMS) discovered by Alan Schoen in 1970, defined implicitly by sin(x)cos(y) + sin(y)cos(z) + "
                "sin(z)cos(x) = 0. It divides space into two congruent labyrinths and has cubic symmetry (space group I4₁32). The surfels (surface elements) "
                "method represents the gyroid through discrete surface points, making it suitable for 3D printing and additive manufacturing. Gyroids appear "
                "in butterfly wing structures, block copolymer phases, and biological membranes. Their zero mean curvature provides optimal structural efficiency, "
                "making them ideal for lightweight lattice structures, heat exchangers, tissue engineering scaffolds, and acoustic metamaterials.";

            descriptions["Gyroid (Marching Cubes)"] =
                "This implementation generates gyroid surfaces using the marching cubes algorithm, which converts implicit surface definitions (isosurfaces of "
                "scalar fields) into explicit triangle meshes. Marching cubes samples the gyroid's implicit function on a 3D grid, determines where the surface "
                "crosses grid cell edges, and generates triangles based on lookup tables for 256 possible cell configurations. This method produces smooth, "
                "high-quality meshes suitable for rendering, FEA analysis, and CFD simulations. The approach offers better control over mesh resolution and "
                "topology compared to surfel methods, making it preferred for engineering analysis of gyroid-based structures.";

            descriptions["Schwarz P (Surfels)"] =
                "The Schwarz P (Primitive) surface, discovered by Hermann Schwarz in 1865, was the first known example of a triply periodic minimal surface. "
                "Its implicit equation is cos(x) + cos(y) + cos(z) = 0 with cubic symmetry (space group Pm3̄m). The P-surface consists of saddle-shaped patches "
                "meeting at tetrahedral angles, dividing space into two interpenetrating labyrinths. The surfels representation using discrete surface elements "
                "is computationally efficient for 3D printing. Schwarz P surfaces appear in biomembranes, zeolite crystals, and are used in architected materials "
                "for superior strength-to-weight ratios, catalytic supports, and bone tissue scaffolds due to high surface area and interconnectivity.";

            descriptions["Schwarz P (Marching Cubes)"] =
                "The marching cubes implementation of the Schwarz P surface converts its implicit equation cos(x) + cos(y) + cos(z) = 0 into an explicit triangle "
                "mesh through volumetric sampling. This method evaluates the function at grid vertices, identifies surface crossings, and generates triangles "
                "using the marching cubes algorithm's case tables. The result is a watertight, manifold mesh with controllable resolution perfect for CFD "
                "simulations, structural FEA, and high-quality visualization. Compared to surfels, marching cubes provides better surface continuity and "
                "more accurate geometry representation, essential for engineering applications requiring precise surface properties.";

            descriptions["Neovius (Surfels)"] =
                "The Neovius surface was discovered by Edvard Rudolf Neovius (Schwarz's student) in 1883. Its implicit equation is 3(cos(x) + cos(y) + cos(z)) + "
                "4cos(x)cos(y)cos(z) = 0 with cubic symmetry. The Neovius surface combines characteristics of both gyroid and Schwarz P surfaces, creating a "
                "different labyrinthine partition of space. The surfels representation enables efficient 3D printing of these complex structures. Neovius surfaces "
                "are increasingly used in lattice structures for aerospace applications, biomedical implants, and heat exchangers due to their optimal balance "
                "of surface area, mechanical properties, and flow characteristics. Their unique geometry provides advantages in specific loading conditions.";

            descriptions["Neovius (Marching Cubes)"] =
                "Marching cubes implementation of the Neovius surface converts its implicit equation into a high-quality triangle mesh through voxel grid sampling "
                "and edge intersection calculation. This approach generates smooth, analytically accurate surfaces essential for engineering simulations. The "
                "resulting mesh can be directly used in FEA for mechanical analysis, CFD for fluid flow optimization, or thermal analysis for heat transfer "
                "applications. Marching cubes produces watertight geometry with controllable mesh density, enabling multi-scale analysis of Neovius-based "
                "structures from microscale (material design) to macroscale (structural components). The method captures the surface's subtle curvature variations "
                "critical for performance prediction.";

            descriptions["Helicoid"] =
                "The helicoid is a ruled minimal surface, the only ruled minimal surface besides the plane. It's generated by moving a line through a central "
                "axis while rotating it uniformly: x = ρ·cos(θ), y = ρ·sin(θ), z = c·θ where ρ is the radial distance and c is a constant determining pitch. "
                "The helicoid has zero mean curvature everywhere (minimal surface property) and constant negative Gaussian curvature. It's locally isometric to "
                "the catenoid—they can be continuously deformed into each other (Bonnet transformation). The helicoid appears in screw dislocations in crystals, "
                "propeller design, and architectural staircases. Its minimal surface property makes it stable for soap film demonstrations.";

            descriptions["Catenoid"] =
                "The catenoid is the surface of revolution formed by rotating a catenary curve (y = a·cosh(x/a)) around its directrix. It's the minimal surface "
                "with the least area spanning two coaxial circles, making it the shape naturally formed by soap films between circular rings. Parametrically: "
                "x = c·cosh(v/c)·cos(u), y = c·cosh(v/c)·sin(u), z = v. The catenoid has constant negative Gaussian curvature and is the only surface of "
                "revolution that is also a minimal surface (besides the plane). It's related to the helicoid through the Bonnet transformation and demonstrates "
                "the principle of minimal surface energy. Catenoids appear in materials science, biological membranes, and theoretical physics.";

            descriptions["Enneper"] =
                "The Enneper surface, discovered by Alfred Enneper in 1864, is a self-intersecting minimal surface with beautiful symmetry. Its Weierstrass "
                "parametrization: x = u - u³/3 + uv², y = v - v³/3 + vu², z = u² - v² creates a saddle-like shape with three-fold rotational symmetry. "
                "Unlike helicoid and catenoid, the Enneper surface has self-intersections for large parameter values and conjugate minimal surface pairs. "
                "It has zero mean curvature (minimal) and demonstrates interesting behaviors in conformal geometry. The surface is used in mathematical "
                "visualization, architectural design exploration, and studying the calculus of variations. Its elegant parametric form makes it pedagogically "
                "valuable for teaching minimal surface theory.";

            // ORGANIC
            descriptions["Nautilus Shell"] =
                "The nautilus shell follows a logarithmic spiral (equiangular spiral) with equation r = a·e^(bθ) where the ratio between successive whorl radii "
                "approximates the golden ratio φ ≈ 1.618. This growth pattern maintains similar proportions at all scales (self-similarity). The 3D shell surface "
                "combines the logarithmic spiral in the xy-plane with expanding circular cross-sections, creating the characteristic chambered nautilus form. "
                "This geometry appears throughout nature (spiral galaxies, hurricanes, sunflower seeds) due to its optimization properties—efficient packing, "
                "structural strength, and minimal surface energy. The nautilus demonstrates the connection between mathematics (Fibonacci sequence, golden ratio) "
                "and biological growth patterns, making it fundamental in biomimetic design and architectural applications.";

            descriptions["Conch Shell"] =
                "Conch shells exhibit conical spiral geometry with increasing aperture size, combining logarithmic spiral growth with conical expansion: "
                "r(θ) = a·e^(bθ), z(θ) = c·θ creating a 3D helico-spiral. Unlike the nautilus's chamber structure, conch shells have continuous internal volume "
                "with gradually expanding whorl tubes. The shell's thickness varies with growth, and surface ornamentation (ridges, spines, knobs) serves structural "
                "and defensive functions. The conch's acoustic properties—acting as natural resonator—arise from its specific geometric ratios. This geometry "
                "informs acoustic design, architectural spirals, and demonstrates how helical growth combined with variable expansion rates creates diverse "
                "natural forms. The surface texture and coloration patterns follow reaction-diffusion dynamics.";

            descriptions["Flower"] =
                "Procedural flowers use mathematical models to generate petal arrangements following phyllotaxis (leaf arrangement) patterns, often based on "
                "golden angle ≈137.5° for optimal packing. Each petal can be modeled using: superellipse cross-sections for shape, Bézier curves for edges, "
                "and displacement functions for ruffling. The petal count often follows Fibonacci numbers (3, 5, 8, 13, 21), and radial symmetry follows "
                "Cn rotational symmetry. Advanced models include: petal overlap and occlusion, stamen/pistil central structures, sepals beneath petals, and "
                "stem attachment. Flower generation demonstrates L-systems, subdivision surfaces, and nature's optimization—maximizing light capture and "
                "pollinator visibility while minimizing material cost. Applications include botanical visualization, game environments, and generative art.";

            descriptions["Coral"] =
                "Procedural coral structures use branching algorithms combined with random perturbations to create organic reef-like forms. Common approaches: "
                "differential growth (surface expansion with self-collision avoidance), reaction-diffusion patterns for texture, space colonization algorithms "
                "(growth toward unoccupied space), and L-system fractals with stochastic rules. Coral geometry exhibits: fractal dimension ~2.5, surface "
                "complexity for maximum area-to-volume ratio (feeding efficiency), and structural optimization for wave resistance. Parameters control branching "
                "angle, growth rate variation, tendency to fuse branches, and surface roughness. Coral modeling is used in marine biology visualization, "
                "reef restoration simulation, architectural structures inspired by natural load distribution, and demonstrates emergent complexity from simple rules.";

            descriptions["Crystal"] =
                "Crystalline structures are defined by their space lattice symmetry and atomic arrangement patterns. 3D crystal models use: lattice unit cells "
                "(7 crystal systems: cubic, tetragonal, orthorhombic, monoclinic, triclinic, hexagonal, trigonal), point group symmetries (32 crystallographic "
                "point groups), and space group operations (230 space groups). The geometry combines: polyhedral faces following crystallographic planes (Miller "
                "indices), face development based on growth rates, and symmetric faceting determined by molecular bonds. Crystal habit (external morphology) "
                "results from surface energy minimization. Applications include: mineralogy visualization, materials science (predicting crystal structure), "
                "jewelry design, and demonstrating group theory in 3D geometry. The faceted surfaces create interesting light refraction patterns (optical properties).";

            descriptions["Tentacle"] =
                "Organic tentacle structures use tapered cylinder stacks with procedural animation curves for natural motion. The geometry typically involves: "
                "a base circular cross-section with gradual radius reduction following power law r(t) = r₀·(1-t)^α, vertebra-like segment stacking for articulation, "
                "and Catmull-Rom or Bézier splines defining the central curve. Surface features include: sucker distribution (circular or spiral patterns), "
                "muscle ridge details, and membrane webbing between segments. Procedural techniques use: inverse kinematics for physics-based motion, "
                "Perlin noise for surface texture variation, and tapering functions for natural thickness variation. Applications: character rigging for animation, "
                "soft-body robotics (biomimetic manipulators), medical device design (endoscopes), and demonstrates the mathematics of flexible appendages.";

            descriptions["Leaf Venation"] =
                "Leaf venation networks are generated using space colonization algorithms, Dijkstra-based path optimization, or reaction-diffusion systems. "
                "The vein pattern emerges from: auxin transport models (hormone distribution), attraction points distributed on leaf blade, iterative growth "
                "toward uncolonized regions, and hierarchical branching (primary, secondary, tertiary veins). Vein patterns follow: Murray's law (optimal "
                "branching for flow distribution), fractal scaling (self-similar branching hierarchy), and minimize total path length while maintaining structural "
                "support. The resulting network shows: reticulate (net-like) or parallel venation, closed loops for redundancy, and tapering vein thickness. "
                "Applications: architectural structure optimization, microfluidic device design, understanding plant hydraulics, and demonstrates biological "
                "optimization through distributed algorithms.";

            // FRACTALS
            descriptions["Fractal Tree"] =
                "Fractal trees use recursive branching with self-similar structure at multiple scales. Basic algorithm: from trunk endpoint, generate N child "
                "branches at angles θᵢ with length scale factors sᵢ, recursively repeat for each branch. Parameters control: branching factor (binary trees: 2, "
                "ternary: 3), branching angles (often golden angle ~137.5°), length ratio between parent and child (~0.7-0.8), branch thickness taper, and "
                "recursion depth (tree 'age'). Advanced models add: stochastic variation (natural irregularity), tropism (growth direction bias toward light/gravity), "
                "seasonal changes, and leaf/flower placement. Fractal trees exhibit: power-law scaling, fractal dimension ~1.5-2.0, and optimal light capture vs "
                "structural cost. Used in: forest visualization, procedural generation, studying botanical growth rules, and demonstrating L-systems.";

            descriptions["L-System Plant"] =
                "Lindenmayer Systems (L-systems) are parallel rewriting systems that model plant development. A grammar consists of: alphabet of symbols (F=forward, "
                "+/- = turn, [/] = push/pop state), axiom (initial string), and production rules (symbol replacement). Each iteration applies all rules simultaneously, "
                "creating exponential complexity growth. Example: Axiom: F, Rule: F→F[+F]F[-F]F generates bushy plant structure. The string is interpreted "
                "geometrically: 'F' draws line segment, '+' rotates left, '[' saves position/angle, ']' restores state. Parameters: angle δ, length reduction factor, "
                "and stochastic rules for variation. L-systems elegantly model: phyllotaxis patterns, inflorescence architecture, colony growth, and demonstrate how "
                "simple rules generate complex morphology. Extensions include parametric L-systems (carrying geometric parameters) and environmentally-sensitive L-systems.";

            descriptions["Fractal Mountain"] =
                "Fractal terrain generation uses the diamond-square algorithm (midpoint displacement) to create realistic height fields with fractal properties. "
                "Algorithm: 1) Start with corner heights on a grid, 2) Diamond step: set midpoint = average of corners + random displacement, 3) Square step: "
                "set edge midpoints = average of adjacent points + random displacement, 4) Recursively subdivide, reducing displacement magnitude by roughness "
                "factor h (typically h=0.5-0.7). The result has fractal dimension D = 3 - h, with h controlling roughness (h→0 gives smooth terrain, h→1 gives "
                "rough/jagged mountains). Related techniques: Perlin noise for more control, hydraulic erosion simulation for realism, and thermal weathering. "
                "Applications: game terrain, landscape visualization, studying geomorphology, and demonstrates 1/f noise in natural surfaces.";

            descriptions["Barnsley Fern"] =
                "The Barnsley Fern uses an Iterated Function System (IFS) with four affine transformations applied probabilistically to generate a fern-like structure. "
                "The four functions: f₁ (stem, 1% probability), f₂ (small leaflet, 85%), f₃ (left pinna, 7%), f₄ (right pinna, 7%) are 2×2 matrices with translation "
                "vectors. Starting from origin, randomly select and apply transformations iteratively. After ~10,000 iterations, the point distribution forms a detailed "
                "fern shape. The attractor (limiting set) has fractal dimension ~1.5 and demonstrates self-similarity—each leaflet resembles the whole fern. This "
                "showcases: chaos theory (sensitive dependence), fractals in nature, image compression potential (entire fern encoded in 24 numbers), and stochastic "
                "geometry. IFS generalizes to model various natural forms through different transformation sets.";

            descriptions["Mandelbrot 2D"] =
                "The Mandelbrot set is defined in the complex plane as the set of points c for which iterating z_{n+1} = z_n² + c (starting from z₀=0) remains bounded. "
                "For each pixel (x,y) representing c = x + iy: iterate the function, count iterations until |z| > 2 (escape) or max iterations reached. Points in the "
                "set (no escape) are colored black; external points colored by escape time, creating the iconic fractal boundary. The set exhibits: infinite complexity "
                "at all zoom levels, self-similarity with variations (not exact), Hausdorff dimension = 2 boundary, and contains miniature copies of itself. Features "
                "include: main cardioid, circular bulbs, seahorse valley, elephant valley, and filaments connecting regions. The Mandelbrot set is the most famous "
                "fractal, revealing deep connections between complex dynamics, number theory, and computational mathematics.";

            descriptions["Mandelbrot 3D"] =
                "3D Mandelbrot sets extend the 2D definition using hypercomplex numbers (quaternions or triplex numbers). Mandelbulb formula: z_{n+1} = z_n^p + c "
                "where exponentiation uses spherical coordinates: z^p = r^p(cos(pθ)cos(pφ), sin(pθ)cos(pφ), sin(pφ)) with r=|z|, θ=arctan(y/x), φ=arctan(z/√(x²+y²)). "
                "Power p=8 (Mandelbulb) creates the most visually striking forms with detail at all scales. The 3D structure exhibits: bulbous lobes, intricate surface "
                "detail, recursive tentacle-like protrusions, and islands of different fractal behavior. Volume rendering or marching cubes reveals internal structure. "
                "Discovery (2009) generated excitement for apparently 'true' 3D Mandelbrot analog. Applications: mathematical visualization, fractal art, and exploring "
                "hypercomplex dynamics.";

            descriptions["Menger Sponge"] =
                "The Menger sponge is constructed by recursively subdividing a cube into 3×3×3 = 27 sub-cubes and removing the center cube and face-centered cubes "
                "(7 cubes removed, 20 remain). Repeating this process infinitely creates a fractal with: Hausdorff dimension log(20)/log(3) ≈ 2.727, zero volume "
                "(V_n = (20/27)^n → 0), infinite surface area (A_n = 6·(20/9)^n → ∞), and exact self-similarity at every third subdivision. The sponge demonstrates: "
                "sets with dimension between 2 and 3, space-filling curve properties, topological complexity (infinite genus), and appears in physics (fractal antenna, "
                "porous materials). Related to Sierpinski carpet (2D analog) and represents fractals with practical engineering applications—modeling porous media, "
                "catalyst substrates, and antennas with broadband frequency response.";

            descriptions["Julia Set 3D"] =
                "3D Julia sets extend 2D Julia sets to quaternion algebra: iterate Q_{n+1} = Q_n² + C where Q and C are quaternions. For each point (x,y,z) representing "
                "quaternion q = x + yi + zj + wk, iterate until |Q| > escape radius or max iterations reached. Points that remain bounded form the Julia set. Unlike "
                "Mandelbrot (one set), Julia sets form a family parameterized by C. The 4D dynamics projected to 3D create intricate, highly detailed structures: "
                "connected (for C in Mandelbrot set) or disconnected dust-like forms. Features: rotational symmetry, self-similar detail, and topology varying with "
                "parameter C. Quaternion Julia sets reveal relationships between complex dynamics, group theory, and 4D geometry, useful in mathematical visualization "
                "and studying hypercomplex iteration dynamics.";

            descriptions["Julia Set 2D"] =
                "2D Julia sets are defined for a fixed complex parameter c by iterating z_{n+1} = z_n² + c for each point z₀ in the complex plane. Points that "
                "remain bounded (don't escape to infinity) form the Julia set for that specific c value. The set's character depends critically on c: if c is in "
                "the Mandelbrot set, the Julia set is connected; otherwise, it's a Cantor dust (disconnected). Each c produces unique fractal: c = -0.4 + 0.6i "
                "(dendrite), c = 0.285 + 0.01i (dragon), c = -0.8 + 0.156i (douady rabbit), c = -0.7269 + 0.1889i (San Marco). Julia sets exhibit: self-similarity, "
                "intricate boundary structure, sensitive parameter dependence, and connection to dynamical systems. Named after Gaston Julia (1918), they're fundamental "
                "in complex dynamics, fractal art, and demonstrate the richness of quadratic maps.";

            descriptions["Sierpinski"] =
                "The Sierpinski triangle/pyramid is constructed by recursive subdivision: starting with equilateral triangle (2D) or tetrahedron (3D), connect edge "
                "midpoints creating 3 (2D) or 4 (3D) smaller copies, remove central portion, and repeat recursively. Properties: Hausdorff dimension log(3)/log(2) "
                "≈ 1.585 (2D) or log(4)/log(2) = 2 (3D), exact self-similarity (each part is 1/2 scale copy), area→0 or volume→0 in limit, but infinite boundary "
                "length/surface area. Alternative generation: chaos game (randomly jump halfway to triangle vertices). The Sierpinski fractal demonstrates: Pascal's "
                "triangle mod 2 correspondence, binomial coefficient patterns, cellular automata (Rule 90), and appears in: antenna design (multiband response), "
                "gaskets for sealing, and models for diffusion on fractals. It's one of the oldest and most studied fractals.";

            descriptions["Koch Snowflake"] =
                "The Koch snowflake is constructed iteratively: starting with equilateral triangle, for each edge: divide into thirds, replace middle third with "
                "two sides of equilateral triangle pointing outward. Each iteration multiplies edge count by 4/3. Properties: perimeter P_n = 3·(4/3)^n → ∞ "
                "(infinite perimeter), area A_n → (8/5)A₀ (finite area), Hausdorff dimension log(4)/log(3) ≈ 1.262. The snowflake demonstrates: continuous "
                "everywhere but differentiable nowhere (counterexample to intuition), finite area with infinite perimeter (mathematical pathology), and self-similarity "
                "at every scale. Discovered by Helge von Koch (1904), it models: coastline irregularity (coastline paradox), fractal antennas (broadband, compact), "
                "and illustrates fractal dimension between 1D line and 2D area. The snowflake showcases beautiful mathematics emerging from simple recursive rules.";

            descriptions["Dragon Curve"] =
                "The Heighway dragon curve is generated by repeatedly folding a strip of paper in half: the creases create ±90° turns that define the curve. "
                "Algorithmically: start with line segment FX, rules: X → X+YF+, Y → -FX-Y where F=forward, +/- =turn ±90°. Each iteration adds detail; the limiting "
                "curve is continuous but nowhere differentiable. Properties: Hausdorff dimension ≈1.52, never self-intersects, tiles the plane (dragon curves fit "
                "together perfectly), and exhibits self-similarity. The curve can be defined by complex numbers: at each step, multiply direction by (1+i)/√2 or "
                "(1-i)/√2 alternately. Dragon curves demonstrate: L-system generation, paper-folding geometry, complex number sequences, and appear in: computer science "
                "(space-filling curves), fractal art, and models for river meandering. The 3D extension creates elaborate spatial curves.";

            // ARCHITECTURAL
            descriptions["Building"] =
                "Parametric building generation uses procedural techniques: floor plan extrusion, floor-by-floor stacking with varying footprints (setbacks for "
                "skyscrapers), and modular facade elements. Core system: vertical circulation (elevators, stairs) centered or to side, structural grid following "
                "column spacing (typically 6-12m), and envelope with window distribution following functional/aesthetic rules. Advanced features: corner chamfering "
                "or rounding, terrace/balcony placement, roof forms (flat, pitched, mansard), and LOD (Level of Detail) for distant viewing. Parameters control: "
                "footprint dimensions, floor heights, setback ratios, facade articulation, and architectural style rules. Used in: urban planning visualization, "
                "game environments (procedural cities), real estate development, and demonstrates rule-based generation of complex architecture from high-level parameters.";

            descriptions["Voronoi Lattice"] =
                "3D Voronoi tessellation partitions space based on proximity to seed points: each cell contains points closer to its seed than to any other seed. "
                "Computing Voronoi diagrams: generate random seeds in 3D volume, construct Delaunay tetrahedralization (dual of Voronoi), extract Voronoi cell faces "
                "(perpendicular bisector planes), and create polyhedral cells. The lattice uses Voronoi edges as structural members, creating irregular but efficient "
                "framework. Properties: natural load distribution, no preferential directions (isotropic), and cells adapt to seed distribution. Applications: lightweight "
                "structures in aerospace, architectural space frames (natural aesthetic), bio-inspired scaffolds (similar to trabecular bone), and acoustic panels "
                "(irregular geometry reduces resonance). Demonstrates nature's optimization—Voronoi patterns appear in cell structures, giraffe spots, and cracked mud.";

            descriptions["Hexagonal Grid"] =
                "Hexagonal tessellation (honeycomb pattern) is optimal for 2D space-filling with maximum packing efficiency: hexagons tile the plane with least "
                "perimeter per area (isoperimetric property), each cell has 6 neighbors at equal distance, and creates equilateral triangular dual grid. In 3D, extruded "
                "hexagons form prismatic honeycomb structure. Properties: even stress distribution (no corners concentrating stress), efficient material usage (20% "
                "less than square grid for same rigidity), and isotropic in-plane behavior. Hexagonal coordinates: axial (q,r), cube (x,y,z with x+y+z=0), or offset. "
                "Applications: structural cores (aircraft panels, packaging), geodesic domes (buckyballs follow hexagonal/pentagonal pattern), game boards (Settlers of Catan), "
                "and nature demonstrates optimality—beehive honeycombs, compound eyes, columnar basalt.";

            descriptions["Facade Pattern"] =
                "Parametric facades use algorithm-generated panel patterns responding to functional requirements: solar orientation (shading where needed), structural "
                "loads (thicker elements in high-stress regions), interior space function (transparency variation), and aesthetic expression. Generation techniques: "
                "attractor-based deformation (panels react to curves/points), subdivision with variation (random or rule-based), pixelation of underlying images, "
                "and Voronoi/Delaunay tessellation for organic patterns. Panel parameters: depth (relief), aperture size (transparency), orientation (sun angles), "
                "and material properties. Advanced systems: kinetic facades (moveable panels responding to sun/weather), biomimetic patterns (learning from nature), "
                "and fractal recursion. Applications: building envelopes, shading systems, ventilation grilles, and demonstrates performance-driven generative design "
                "in contemporary architecture (Zaha Hadid, Norman Foster).";

            descriptions["Structural Frame"] =
                "3D structural frames consist of linear elements (beams, columns, braces) connected at nodes forming a load-bearing skeleton. Typical configurations: "
                "orthogonal grids (post-and-beam), space frames (triangulated 3D trusses), diagrid systems (diagonal grid—efficient for tall buildings), and braced "
                "frames (lateral stability). Design considerations: member sizing (strength, stiffness, buckling), node detailing (moment connections vs. pinned), "
                "load paths (gravity and lateral loads), and redundancy (alternate load paths). Analysis methods: finite element analysis, matrix structural analysis. "
                "Optimization goals: minimize weight/cost subject to stress, deflection, and code constraints. Applications: high-rise buildings, bridges, stadiums, "
                "industrial facilities. Demonstrates principles of statics, mechanics of materials, and structural optimization—architectural form following "
                "efficient load distribution.";

            // MECHANICAL
            descriptions["Gear"] =
                "Involute gears use tooth profiles defined by involute curves—the path traced by unwinding string from a base circle. This profile ensures: constant "
                "velocity ratio (smooth power transmission), pressure angle remains constant (typically 20° or 14.5°), and center distance tolerance (gears still mesh "
                "correctly with small spacing errors). Gear geometry: pitch circle (theoretical contact), base circle (involute generation), addendum (tooth height "
                "above pitch), dedendum (depth below pitch), and clearance. Key parameters: module m or diametral pitch P (tooth size), number of teeth N, pressure "
                "angle α, and face width. Tooth count determines gear ratio: ω₁/ω₂ = N₂/N₁. Applications: transmissions, reducers, clocks, and mechanisms requiring "
                "precise speed conversion. Demonstrates analytical geometry solving mechanical engineering problems—smooth, efficient power transmission.";

            descriptions["Threaded Bolt"] =
                "Screw threads consist of helical ridge (thread) wrapped around cylindrical or conical shaft. The helix follows: r(θ) = constant, z(θ) = (pitch/2π)·θ "
                "where pitch p is axial distance per revolution. Thread standards define: major diameter (outer), minor diameter (root), pitch diameter (theoretical "
                "contact), thread angle (60° for metric/unified, 55° for Whitworth), and crest/root profiles. Thread form generates mechanical advantage: force "
                "multiplication F_axial ≈ F_tangent·(2πr/p), converts rotation to translation, and provides clamping force through preload. Types: V-threads (fastening), "
                "square threads (power transmission—jacks, presses), ACME threads (heavier loads), and buttress threads (unidirectional loads). Applications: fasteners, "
                "adjusters, precision positioning (micrometers), and demonstrates simple machines—inclined plane wrapped around cylinder.";

            descriptions["Bearing"] =
                "Ball bearings reduce rotational friction using rolling elements between inner and outer races. Geometry: balls (spheres) sit in grooves (races) with "
                "specific curvature, cage/separator maintains ball spacing, and contact points follow kinematic constraints. Key parameters: bore diameter (inner race), "
                "outer diameter, width, ball diameter, and number of balls. The balls experience: rolling contact (low friction), contact stress (Hertzian contact "
                "mechanics), and centrifugal forces at high speeds. Load capacity depends on: contact angle (axial vs. radial load distribution), number of balls, "
                "ball diameter, and material properties. Types: deep groove (general purpose), angular contact (combined loads), thrust bearings (axial loads). "
                "Applications: motors, wheels, machinery, robotics. Demonstrates tribology, contact mechanics, and precision manufacturing—typical ball sphericity "
                "within 0.1 µm.";

            // STRUCTURAL
            descriptions["Diamond Lattice"] =
                "The diamond cubic lattice is the crystal structure of diamond and silicon: two interpenetrating face-centered cubic (FCC) lattices offset by 1/4 body "
                "diagonal. Each atom has 4 neighbors in tetrahedral coordination (109.47° angles), creating extremely strong covalent bonding network. The structure's "
                "topology provides: maximum rigidity with minimal material (strut arrangement resists deformation in all directions), isotropic mechanical properties, "
                "and high strength-to-weight ratio. Artificial diamond lattices in additive manufacturing use: tetrahedral unit cells with struts along <111> directions, "
                "periodic tiling of space, and parametric strut thickness. Applications: lightweight structures in aerospace, energy absorption in crash protection, "
                "bio-inspired orthopedic implants (mimics trabecular bone), and thermal management. Demonstrates how atomic-scale geometry inspires macro-scale "
                "engineering solutions.";

            descriptions["Pipe Network"] =
                "Branching pipe networks distribute fluids efficiently following principles analogous to vascular systems. Network topology determined by: space colonization "
                "algorithms (growth toward target points), flow-based optimization (minimize pressure loss), or Steiner tree approximation (minimize total pipe length). "
                "Key principles: Murray's law—optimal branch diameter ratios minimize pumping energy: D_parent³ = ΣD_children³, bifurcation angles optimizing flow "
                "dynamics, and hierarchical branching (large mains to small distribution). Pipe junctions (nodes) require: manifolds for flow splitting, gradual transitions "
                "(avoid separation), and structural support at changes in direction. Applications: HVAC systems, irrigation networks, cooling systems (electronics, data "
                "centers), and vascular stents. Demonstrates engineering optimization inspired by biology—cardiovascular systems evolved to minimize cardiac work while "
                "delivering nutrients to all cells.";

            descriptions["Lattice"] =
                "Lattice structures consist of periodic arrangements of struts forming unit cells that tile 3D space. Common topologies: cubic (aligned with axes), "
                "octet-truss (octahedron-tetrahedron tiling—optimal stiffness), body-centered cubic (BCC), face-centered cubic (FCC), and Kagome lattice (triangular "
                "coordination). Mechanical behavior characterized by: relative density ρ_rel = V_solid/V_total, Young's modulus scaling E ~ ρ_rel^n (n=1 stretching-dominated, "
                "n=2 bending-dominated), and deformation modes (elastic buckling, plastic yielding). Design parameters: strut diameter, unit cell size, and topology. "
                "Applications: aerospace structures (launch vehicle internals), heat exchangers (high surface area/volume), energy absorption (crash structures), and "
                "bio-scaffolds. Additive manufacturing enables complex lattices impossible with traditional methods. Demonstrates material-structure-property relationships—achieve "
                "desired properties through topology rather than material alone.";

            // TESSELLATIONS
            descriptions["Penrose Tiling"] =
                "Penrose tilings are non-periodic tessellations that cover the plane without translational symmetry but exhibit local five-fold symmetry. The most common "
                "version uses two rhombus tiles: thick (72°/108° angles) and thin (36°/144°), or equivalently kite and dart shapes. Matching rules ensure only non-periodic "
                "arrangements. Properties: quasicrystalline order (long-range order without periodicity), inflation/deflation symmetry (self-similarity at different scales), "
                "density of each tile type follows golden ratio φ, and any finite patch appears infinitely often. Discovered by Roger Penrose (1974), these tilings relate "
                "to: quasicrystals (2011 Nobel Prize—materials with Penrose-like atomic arrangements), recreational mathematics, and Islamic geometric patterns. "
                "Applications: decorative design, floor patterns, studying aperiodic order, and demonstrates mathematical sophistication in structure without periodicity.";

            descriptions["Islamic Pattern"] =
                "Islamic geometric patterns are constructed using compass and straightedge, based on principles of symmetry, polygonal tessellation, and interlacing. "
                "Construction method: establish symmetry framework (4, 6, 8, 10, or 12-fold), create polygonal grid (typically squares, hexagons, or 12-pointed stars), "
                "connect vertices/midpoints following design rules, and create interlacing (over-under weaving) for visual complexity. Patterns exhibit: multiple scales "
                "(small motifs to large compositions), wallpaper group symmetries (17 possible types), and sacred geometry (embodying unity/infinity). Common motifs: "
                "girih (strapwork), shamsa (sun), and khatayi (arabesque). The designs demonstrate mathematical sophistication in medieval Islamic culture (9th-17th centuries) "
                "and appear on architecture, textiles, manuscripts. Modern applications: architectural ornamentation, laser-cut screens, design inspiration, and study "
                "of mathematical symmetry and group theory in art.";

            descriptions["Truchet Tiles"] =
                "Truchet tiles are square tiles with patterns that create interesting designs when randomly rotated. Original design (Sébastien Truchet, 1704): square "
                "divided diagonally into two colored triangles. Modern variants: quarter circles in opposite corners, connecting paths/lines. Each tile has limited "
                "orientations (typically 4: 0°, 90°, 180°, 270°), and adjacent tiles always connect properly regardless of orientation. Random rotation creates: "
                "emergent patterns (apparent structure from randomness), continuous paths or mazes, balanced contrast (roughly equal amounts of each element), and "
                "no repeating pattern. Smith tiles extend the concept with edge constraints ensuring continuity. Applications: texture generation in computer graphics, "
                "procedural wallpaper patterns, game maze generation, and studying pattern emergence from local random rules. Demonstrates how simple tile rules and "
                "randomness produce visually complex, aesthetically pleasing results.";

            descriptions["Modular Circles"] =
                "Modular circle patterns explore relationships in modular arithmetic through geometric visualization: plotting points at angles 2πn·k (mod m) for various "
                "k creates circles connected by chords, generating surprising patterns. Multiplication table variant: connect point i to point k·i (mod N) on circle with "
                "N equally-spaced points. Different k values produce: prime numbers give star patterns, factors of N create sub-patterns, and composite numbers show "
                "symmetry related to divisors. These diagrams visualize: number theory relationships (factors, multiples, primes), group theory structure (cyclic groups), "
                "and modular arithmetic geometry. The patterns exhibit: rotational symmetry (order related to divisors), cardioid curves appear for certain parameters, "
                "and self-similarity at different scales. Applications: mathematical visualization, educational tools for number theory, generative art, and demonstrates "
                "beauty in mathematical structure—visual representation of algebraic relationships.";

            // COMPUTATIONAL
            descriptions["Reaction-Diffusion"] =
                "Reaction-diffusion systems model pattern formation through chemical reactions coupled with diffusion, described by partial differential equations: "
                "∂u/∂t = D_u∇²u + F(u,v), ∂v/∂t = D_v∇²v + G(u,v) where u, v are concentrations, D are diffusion coefficients, and F, G are reaction kinetics. "
                "Gray-Scott model is popular: F(u,v) = -uv² + f(1-u), G(u,v) = uv² - (f+k)v creating patterns depending on feed rate f and kill rate k. Produces: "
                "spots, stripes, spirals, labyrinths depending on parameters. Alan Turing (1952) proposed this mechanism explains biological patterns (animal coats, "
                "seashells). Applications: texture synthesis, pattern generation for graphics, studying morphogenesis, and chemical computing. Demonstrates self-organization—complex "
                "spatial patterns emerge from uniform initial conditions through local interactions, without global coordination or preset template.";

            descriptions["Cellular Automata"] =
                "Cellular automata (CA) consist of grids of cells in discrete states, evolving via local rules applied simultaneously. Each cell's next state depends on "
                "current state of itself and neighbors. Famous examples: Conway's Game of Life (2D), Rule 110 (1D), and 3D CA using 6-neighborhood (von Neumann) or "
                "26-neighborhood (Moore). Rules can create: static patterns, oscillators, gliders (moving structures), and chaotic evolution. Some CA are: Class 1 "
                "(homogeneous), Class 2 (periodic), Class 3 (chaotic), Class 4 (complex—capable of universal computation). CA demonstrate: emergent complexity from simple "
                "rules, self-organization, edge of chaos (boundary between order and chaos yields interesting behavior), and computational universality (Rule 110, Life "
                "are Turing complete). Applications: procedural generation, physics simulation, cryptography, and artificial life research. John Conway's Life (1970) "
                "remains most famous CA.";

            descriptions["Voronoi Surface"] =
                "Voronoi surfaces tessellate a surface (rather than plane/space) based on geodesic distance to seed points on the surface. Computing requires: placing "
                "seeds on surface, computing geodesic distances (shortest paths along surface), assigning each surface point to nearest seed, and extracting Voronoi "
                "cell boundaries. The cells follow surface curvature, creating organic divisions. Applications: surface segmentation (mesh decomposition), texture synthesis "
                "(organic patterns on 3D objects), architectural facades (panel division optimized for fabrication), and biological modeling (cell territories on curved "
                "surfaces). The pattern adapts to curvature—cells compress in high-curvature regions and expand in flat areas. Related to geodesic CVT (Centroidal Voronoi "
                "Tessellation) for mesh optimization. Demonstrates geometric algorithms extending from Euclidean space to curved manifolds—computational differential geometry "
                "meeting practical applications.";

            descriptions["Delaunay Surface"] =
                "Delaunay surfaces are constructed by computing Delaunay triangulation of points on or near a surface. The Delaunay condition ensures: empty circumsphere "
                "(no point inside circumcircle of any triangle), maximizes minimum angle (avoids sliver triangles), and is unique for points in general position. For "
                "surface reconstruction: sample point cloud, compute 3D Delaunay tetrahedralization, extract surface triangles (alpha shapes or Crust algorithm), and "
                "optimize mesh quality. Results in: well-shaped triangles, smooth surface interpolation, and good numerical properties for FEM. The dual of Delaunay "
                "triangulation is the Voronoi diagram. Applications: terrain reconstruction from LiDAR, medical imaging (organ surfaces), reverse engineering (3D scanning), "
                "and mesh generation for simulation. Demonstrates fundamental computational geometry—Delaunay triangulation is cornerstone algorithm used throughout graphics, "
                "GIS, and scientific computing.";

            descriptions["Voronoi Diagram"] =
                "2D Voronoi diagrams partition the plane based on proximity to a set of seed points (sites): Voronoi cell V(p_i) = {x : d(x,p_i) ≤ d(x,p_j) ∀j}. "
                "Each cell is a convex polygon bounded by perpendicular bisectors of sites. Properties: cell edges are perpendicular bisectors, vertices are equidistant "
                "from 3+ sites (circumcenters of Delaunay triangles), and unbounded cells for convex hull sites. Algorithms: Fortune's sweep line O(n log n), incremental "
                "construction, or Delaunay dual computation. Applications: nearest-neighbor queries (which site is closest?), facility location (optimal placement minimizing "
                "maximum distance), growth simulation (competing regions), and mobile network coverage (cell tower territories). Appears in nature: cell structures, "
                "territorial animals, cracked surfaces. Demonstrates fundamental spatial partition problem with optimal properties for many scenarios—elegant geometry "
                "with practical applications.";

            descriptions["Delaunay Triangulation"] =
                "2D Delaunay triangulation maximizes the minimum angle of all triangles, avoiding sliver triangles that cause numerical problems. Equivalent definitions: "
                "empty circumcircle property (no points inside circumcircle of any triangle), maximizes minimum angle, or dual of Voronoi diagram (sites become vertices, "
                "Voronoi edges become triangle edges). Algorithms: Bowyer-Watson incremental insertion, divide-and-conquer, or sweepline. Properties: unique (for points "
                "in general position), well-conditioned triangles, and optimal for interpolation (minimizes maximum gradient). Applications: terrain modeling (TIN—Triangulated "
                "Irregular Network), finite element mesh generation, computer graphics (texture mapping, lighting), and spatial interpolation. Delaunay triangulation "
                "is fundamental in computational geometry—optimal triangulation balancing various quality metrics, extensively used in GIS, engineering simulation, and "
                "graphics rendering.";

            descriptions["Perlin Noise Surface"] =
                "Perlin noise generates coherent pseudorandom textures by interpolating random gradients on a grid. Algorithm: 1) Define gradient vectors at grid points, "
                "2) For query point, identify surrounding grid cell, 3) Compute dot products of gradients with offset vectors, 4) Interpolate using smoothstep function "
                "s(t) = 3t² - 2t³ or smootherstep 6t⁵ - 15t⁴ + 10t³. Multiple octaves (frequencies) are summed with decreasing amplitude (persistence) and increasing "
                "frequency (lacunarity) to create fractal noise. Properties: continuous (smooth gradients), controllable frequency spectrum, and deterministic (same seed "
                "gives same pattern). Improved Perlin/Simplex noise fixes gradient artifacts. Applications: terrain generation (height fields), texture synthesis (clouds, "
                "marble, wood grain), procedural materials, and animation (controlled randomness). Ken Perlin (1985) revolutionized computer graphics—organic, natural-looking "
                "randomness without visual artifacts of white noise.";

            descriptions["Neural Network"] =
                "Neural networks for mesh generation use learned representations to create geometry. Approaches include: autoencoders learning latent shape space (interpolate "
                "between learned shapes), GANs (Generative Adversarial Networks) generating novel shapes, and implicit neural representations (NeRF, DeepSDF) encoding "
                "geometry as neural network weights. Training on dataset of 3D models learns: shape priors (typical structures), semantic features (parts, symmetries), "
                "and style distributions. Mesh generation from network: decode latent vector to point cloud/mesh, evaluate implicit function on grid (marching cubes), "
                "or directly output mesh connectivity. Applications: design exploration (generate shape variations), shape completion (infer missing data), procedural "
                "generation (learned rules), and compression. Demonstrates machine learning in geometry—networks capture complex distributions over shape space, enabling "
                "data-driven modeling beyond hand-crafted algorithms.";

            descriptions["CPPN-HyperNEAT"] =
                "Compositional Pattern-Producing Networks (CPPNs) generate geometric patterns by querying spatial coordinates through evolved neural networks. Given position "
                "(x,y,z), the network outputs density/material properties. Network uses diverse activation functions (sin, cos, Gaussian, sigmoid) creating compositional "
                "patterns—combining simple functions produces complex, symmetric, repetitive designs. HyperNEAT (Hypercube-based NeuroEvolution of Augmenting Topologies) "
                "evolves CPPN topology and weights using genetic algorithm, optimizing for fitness criteria (aesthetic preferences, structural requirements). Key properties: "
                "natural symmetry (function of coordinates preserves geometric regularities), infinite resolution (query any position), and compact representation (small "
                "network encodes complex pattern). Applications: evolutionary art, 3D printing (generative designs), robot morphology evolution, and neural architecture "
                "search. Demonstrates neuroevolution—evolving neural networks as generative models, producing organic, symmetric forms reminiscent of natural patterns.";

            descriptions["Evolutionary Shape"] =
                "Evolutionary algorithms optimize 3D shapes by iteratively varying parameters and selecting based on fitness criteria. Process: 1) Initialize population "
                "of shapes (random parameters), 2) Evaluate fitness (structural performance, aesthetics, manufacturability), 3) Selection (keep high-fitness individuals), "
                "4) Variation (mutation + crossover of parameters), 5) Repeat for generations. Representation options: parametric (adjust algorithm parameters), direct "
                "(vertex positions as genes), constructive (operations sequence), or implicit (function coefficients). Fitness evaluation might include: FEA simulation "
                "(stress/strain), CFD (drag reduction), packing efficiency, or aesthetic metrics. Multi-objective optimization (Pareto fronts) balances competing goals. "
                "Applications: aerodynamic shapes (airfoils, car bodies), structural topology optimization (lightweight, strong), antenna design, and art. Demonstrates "
                "computational design optimization—exploring vast design space automatically, discovering novel solutions beyond human intuition.";

            // ATTRACTORS
            descriptions["Lorenz Attractor"] =
                "The Lorenz attractor is a chaotic solution to the Lorenz system: dx/dt = σ(y-x), dy/dt = x(ρ-z)-y, dz/dt = xy-βz, discovered by Edward Lorenz (1963) "
                "studying atmospheric convection. Standard parameters: σ=10, ρ=28, β=8/3 produce the characteristic butterfly-shaped attractor. Properties: sensitive "
                "dependence on initial conditions (butterfly effect—tiny differences grow exponentially), strange attractor (fractal dimension ~2.06), and bounded chaos "
                "(stays within finite region but never repeats). The system is deterministic but unpredictable long-term. Demonstrates: limits of weather prediction, "
                "chaos theory foundations, and deterministic chaos (simple equations producing complex behavior). Visualization creates beautiful 3D trajectory through "
                "state space, iconic in chaos theory and dynamical systems. Applications: understanding turbulence, climate dynamics, and studying nonlinear dynamics. "
                "Revolutionized science—showed small uncertainties can make long-term prediction impossible even in deterministic systems.";

            // MODIFIERS
            descriptions["Scale"] =
                "Scale transformation multiplies vertex positions by scale factors: x' = s_x·x, y' = s_y·y, z' = s_z·z. Uniform scaling (s_x = s_y = s_z) preserves "
                "shape (similarity transformation), while non-uniform scaling distorts proportions, creating elongation or compression along axes. Scaling about origin "
                "is straightforward; scaling about arbitrary point p requires: translate by -p, scale, translate by +p. Matrix form: S = diag(s_x, s_y, s_z). Negative "
                "scales create mirroring. Effects: volume changes by s_x·s_y·s_z, normals require inverse-transpose transformation (N' = (S^(-T))N), and textures stretch "
                "accordingly. Applications: model sizing, proportional editing, squash-and-stretch animation, and resolution adaptation. Fundamental linear transformation "
                "in computer graphics, combined with rotation and translation forms affine transformations (rigid body motion + scaling).";

            descriptions["Translate"] =
                "Translation moves all points by constant offset vector t = (t_x, t_y, t_z): p' = p + t. It's an affine transformation preserving: distances (isometry), "
                "angles, parallelism, and shapes (congruence). Translation has no fixed points (everything moves) unlike rotation/scaling. Matrix representation in "
                "homogeneous coordinates: 4×4 matrix with translation in rightmost column. Concatenating translations: T₂·T₁ = T(t₁+t₂) (commutative). Applications: "
                "object positioning in scene, animation (linear motion), camera movement, and coordinate system transformation. Combined with rotation forms rigid body "
                "transformation (preserves distances and angles). Translation is fundamental in: scene graph hierarchies (local→world coordinates), physics simulation "
                "(kinematic updates), and spatial data structures (organizing 3D space). Simple concept, but essential in every 3D application—positioning objects in "
                "world space.";

            descriptions["Rotate"] =
                "Rotation transformation rotates points around axis by angle θ. About coordinate axes: R_x, R_y, R_z use trigonometric functions in rotation matrices. "
                "Arbitrary axis rotation uses Rodrigues' formula: R(n,θ) = I + sin(θ)[n]_× + (1-cos(θ))[n]_×², where [n]_× is skew-symmetric matrix of axis n. "
                "Properties: preserves distances and angles (orthogonal transformation), det(R)=1 (orientation-preserving), and R^T = R^(-1) (inverse is transpose). "
                "Euler angles (roll, pitch, yaw) suffer gimbal lock; quaternions provide robust alternative for interpolation. Rotations don't commute: R_x·R_y ≠ R_y·R_x. "
                "Applications: object orientation, skeletal animation (joint rotations), camera controls (orbit, tumble), and physics (angular momentum). Normal vectors "
                "transform with same rotation matrix. Rotation is fundamental in 3D graphics—understanding rotation representations (matrices, Euler angles, quaternions, "
                "axis-angle) is essential.";

            descriptions["Noise"] =
                "Noise displacement applies random offsets to vertex positions: p' = p + r·n where r is random magnitude and n is direction (typically normal or random). "
                "Pure random noise creates uniform random displacement in all directions or along normals. Effects: breaks regularity, adds geometric detail, creates "
                "organic irregularity, and increases mesh complexity visually without actual subdivision. Control parameters: displacement amount, noise seed (reproducibility), "
                "direction constraint (normal, radial, arbitrary), and per-vertex vs. correlated noise. Applications: rock/terrain roughness, organic surface imperfection, "
                "breaking artificial symmetry, and detail addition. Simple noise is white spectrum (uncorrelated between vertices), appearing jagged. For smoother results, "
                "use coherent noise (Perlin, simplex) or post-process with smoothing. Quick way to add visual complexity and natural irregularity to geometric models.";

            descriptions["Perlin Noise"] =
                "Perlin noise displacement uses coherent gradient noise for smooth, natural-looking surface variation: p' = p + A·noise(f·p)·n where A is amplitude, f "
                "is frequency, and n is displacement direction (usually normal). Multiple octaves combine different frequencies: Σ(A_i·noise(f_i·p)) with f_i = 2^i·f₀ "
                "(doubling frequency) and A_i = A₀·persistence^i (decreasing amplitude). This creates fractal-like detail—large features plus progressively finer detail. "
                "Parameters: amplitude (displacement strength), frequency (feature size), octaves (detail levels), persistence (amplitude reduction), and lacunarity (frequency "
                "increase). Applications: terrain generation (hills, valleys), organic surface detail (skin pores, bark texture), water surfaces (waves), and procedural "
                "detail without modeling. Perlin noise provides controllable, natural-looking randomness superior to white noise—smooth gradients prevent jarring discontinuities. "
                "Essential procedural technique in computer graphics.";

            descriptions["Pulse"] =
                "Pulse deformation creates radial expansion/contraction from center point: displacement = A·f(r)·r̂ where r is distance from center, r̂ is radial direction, "
                "A is amplitude, and f(r) is falloff function. Common falloff: Gaussian f(r) = exp(-r²/σ²), linear f(r) = max(0, 1-r/R), or smooth-step. The pulse can "
                "be: expansion (A > 0), contraction (A < 0), or oscillating (time-varying A). Multiple pulses can overlap. Effect propagation: expanding ripples use "
                "time-dependent radius R(t). Applications: impact deformations, heartbeat pulsation animation, wave propagation visualization, and radial emphasis (draw "
                "attention to region). Advanced versions: directional pulses (along surface), anisotropic falloff, and obstacle-aware propagation. Simple but effective "
                "animation technique—creates organic, natural-looking motion from center point, useful in both physics simulation and stylized animation.";

            descriptions["Repeat Domain"] =
                "Domain repetition (spatial tiling) creates infinite patterns by taking spatial modulo: p' = (x mod L_x, y mod L_y, z mod L_z) where L are tile sizes. "
                "This maps infinite space to bounded domain [0,L_x]×[0,L_y]×[0,L_z], effectively tiling objects infinitely. In SDFs (Signed Distance Fields), domain "
                "repetition before evaluation creates repeated geometry without duplicating data. Advanced variants: finite repetition (limited count), polar repetition "
                "(around axis with n-fold symmetry), and non-uniform tiling. Effects: creates patterns, arrays, and periodic structures from single instance. Applications: "
                "architectural elements (columns, windows), texture tiling (seamless repeat), crystalline structures (unit cell repetition), and game environments (efficient "
                "pattern without memory overhead). Common in shader programming and procedural modeling—achieve complexity through repetition of simple base element. "
                "Demonstrates mathematical periodicity creating visual complexity.";

            descriptions["Smooth"] =
                "Laplacian smoothing moves each vertex toward the average of its neighbors: p'_i = p_i + λ(L(p_i) - p_i) where L(p_i) = (1/|N|)Σp_j (j in neighborhood N), "
                "and λ ∈ (0,1) is relaxation factor. Iterating this reduces high-frequency detail, shrinks the mesh slightly, and can degrade quality (triangle shape). "
                "Taubin smoothing alternates positive and negative λ to reduce shrinkage. Cotangent Laplacian weights neighbors by angles: more geometrically accurate. "
                "Properties: low-pass filter (removes noise), degrades sharp features (undesired), and simple to implement. Applications: noise removal from 3D scans, "
                "mesh fairing (creating smooth surfaces), medical imaging (smoothing segmented surfaces), and relaxation for mesh quality improvement. Alternative methods: "
                "bilateral smoothing (feature-preserving), anisotropic diffusion, or guided smoothing. Fundamental mesh processing operation—balance between noise reduction "
                "and feature preservation critical in many applications.";

            descriptions["Inflate"] =
                "Inflation moves vertices outward along normals: p' = p + d·n where d is inflation distance and n is vertex normal. This uniformly expands the mesh, creating "
                "offset surface. Effects: increases volume, rounds sharp edges (convex regions expand more), and smooths concave areas. Constant inflation distance creates "
                "parallel surface (equidistant from original). For distance d: volume changes by ~A·d + perimeter·d², where A is surface area. Normal calculation: average "
                "of incident face normals (weighted by area or angle). Applications: character modeling (smooth muscles, padding), creating shells/casings, modeling growth/"
                "swelling, and preparing meshes for mollification. Can be combined with smoothing for organic forms. Related to morphological dilation in image processing. "
                "Negative inflation (deflation) shrinks mesh. Simple operation with significant visual impact—useful in modeling workflows for quickly adjusting forms.";

            descriptions["Solidify"] =
                "Solidify creates solid object from surface mesh by: duplicating vertices, offsetting new vertices along normals (inward or outward), connecting boundary "
                "edges with quads to form walls, and capping open edges. Parameters: thickness, offset direction (even/external/internal), and boundary capping. The operation "
                "converts open surface (manifold or non-manifold) into closed solid with thickness. Effects: creates printable models (3D printing requires volume), adds "
                "structural rigidity, enables boolean operations, and allows interior visualization (cross-sections). Challenges: self-intersections in high-curvature regions, "
                "normals consistency, and maintaining mesh quality. Applications: preparing surface scans for 3D printing, creating wall thickness for architectural models, "
                "mechanical part modeling (sheet metal), and game assets (shells for collision). Essential operation when working with surface-only data that needs physical "
                "realization—converts abstract surfaces to manufacturable solids.";

            descriptions["Subdivide"] =
                "Subdivision increases mesh resolution by splitting faces into smaller faces. Simple subdivision: split each edge at midpoint, connect new vertices to form "
                "4 triangles (per original triangle) or 4 quads (per quad). Advanced schemes: Catmull-Clark (quad meshes, limit surface is smooth), Loop (triangle meshes, "
                "smooth), and Butterfly (interpolating). These use weighted averages of surrounding vertices, creating smooth limit surface as subdivision→∞. Properties: "
                "refinement (more faces), smoothness (converging to limit surface), and stability (shrinkage in some schemes). Applications: smooth modeling (control "
                "cage→smooth surface), adaptive mesh refinement (detail where needed), animation (subdivision surfaces for smooth deformation), and multiresolution editing. "
                "Subdivide combined with displacement enables fine detail. Trade-off: quality vs. polygon count. Subdivision surfaces revolutionized character modeling—artists "
                "work with low-poly control mesh, automatic subdivision creates smooth result.";

            descriptions["Decimate"] =
                "Decimation reduces mesh polygon count while preserving shape fidelity. Algorithms: edge collapse (merge edge endpoints, remove incident faces), vertex "
                "clustering (grid-based merging), or face deletion. Quadric Error Metrics (QEM) guides edge collapse priority: quadric Q measures squared distance to adjacent "
                "planes; collapse edges minimizing error Q(v'). Iteratively collapse until target polygon count or maximum error reached. Properties: controllable quality vs. "
                "complexity trade-off, preserves features (sharp edges, boundaries), and adapts resolution to curvature. Applications: LOD (Level of Detail) generation for "
                "games, mesh simplification from 3D scanning (millions→thousands of polygons), real-time rendering optimization, and transmission (lower bandwidth). "
                "Alternative: remeshing creates uniform triangulation. Decimation enables manageable polygon counts while maintaining visual quality—essential for performance "
                "in interactive applications.";

            descriptions["Twist Deform"] =
                "Twist deformation rotates mesh progressively along axis: at height z, rotation angle = α(z/h) where α is total twist angle and h is height. Each vertex "
                "(x,y,z) rotates around axis by angle proportional to its axial position: x' = x·cos(θ(z)) - y·sin(θ(z)), y' = x·sin(θ(z)) + y·cos(θ(z)), z' = z. "
                "Linear twist: θ(z) = α·z/h. Non-linear: exponential, sinusoidal, or custom curves. Effects: creates helical form, shows torsion visually, and demonstrates "
                "shear deformation. Applications: modeling twisted columns, DNA double helix visualization, candy canes, spiral staircases, and demonstrating material torsion. "
                "The deformation is smooth (tangent continuous) if twist function is smooth. Multiple axes or time-varying twist creates animation. Fundamental space "
                "deformation—simple mathematical transformation creates visually interesting organic or mechanical forms. Common in 3D modeling packages as procedural deformer.";

            descriptions["Bend Deform"] =
                "Bend deformation curves mesh along axis with specified angle. Implementation maps straight line to circular arc: vertices at distance d along axis move "
                "to arc position. For bend angle α and radius R: arc length = R·α should equal original length. Vertex (x,y,z) along y-axis bends: y' = R·sin(θ), z' = R·(1-cos(θ)) "
                "where θ = (z/L)·α, L is length. The transformation smoothly curves geometry. Parameters: bend angle, bend axis, and radius (larger radius = gentler curve). "
                "Applications: modeling curved objects (arches, pipes), animation (flexible objects), character rigging (spine, tail), and mechanical demonstration (beam "
                "bending). Lattice-based bend uses FFD (Free-Form Deformation). Bend combined with taper creates complex organic forms. Intuitive deformation for creating "
                "arcs, hooks, and C-shapes from straight objects—procedural modeling efficiency.";

            descriptions["Taper Deform"] =
                "Taper deformation scales cross-sections progressively along axis: at position z, scale = 1 + (factor)(z/h) where factor controls taper rate and h is height. "
                "Linear taper: s(z) = s₀ + (s₁-s₀)(z/h) varies from scale s₀ at base to s₁ at top. Each vertex (x,y,z): x' = x·s(z), y' = y·s(z), z' = z. Positive "
                "factor creates cone-like expansion, negative creates narrowing. Non-uniform taper applies different scaling to x and y. Effects: creates tapered columns, "
                "conical forms, and elongated/compressed shapes. Applications: tree trunks (wider at base), architecture (tapered towers), product design (bottle necks), "
                "and character proportions. The deformation maintains straight lines along taper axis while scaling perpendicular directions. Combined with other deformers "
                "(twist, bend) creates complex organic forms. Simple but versatile—fundamental shape variation in procedural modeling and architecture.";

            descriptions["Wave Deform"] =
                "Wave deformation displaces vertices using sinusoidal function: d(x,y) = A·sin(kx·x + ky·y + φ) where A is amplitude, k is wave number (2π/wavelength), "
                "and φ is phase. Displacement typically along normal or z-axis: p' = p + d(x,y)·n̂. Parameters: amplitude (wave height), wavelength (period), direction "
                "(wave propagation), and phase (offset). Time-varying phase creates animation: φ(t) = ω·t (angular frequency). Multiple waves superimpose: ocean surfaces "
                "combine many wavelengths. Applications: water surface modeling, terrain undulation, fabric ripples, audio visualization, and flag waving. Advanced: damping "
                "(amplitude decay), directional waves (anisotropic), and interacting waves (interference patterns). Wave deformation demonstrates harmonic motion and periodic "
                "functions—creates natural-looking oscillatory displacement, essential in animation, effects, and physical simulation of wave phenomena.";

            descriptions["Spherify"] =
                "Spherify pushes vertices toward spherical shape centered at origin (or arbitrary center). Implementation: interpolate between current position p and spherical "
                "position p_sphere = p/|p|·R: p' = (1-t)p + t·p_sphere where t∈[0,1] is spherify amount and R is target radius. Alternatively, use radial displacement: "
                "p' = p + (R - |p|)·(p/|p|)·t. Effect: gradually transforms any shape toward sphere, preserves vertex count, and smooths irregularities. At t=1, all vertices "
                "lie on sphere of radius R. Applications: creating spherical proxies (simplified collision shapes), smoothing rough spherical objects (planets, balls), "
                "morphing animation (shape→sphere transition), and normalizing scale. Related to normalized cube approach for sphere generation. Demonstrates radial projection—useful "
                "primitive operation when spherical symmetry is desired. Combined with other deformers creates organic spheroidal forms.";

            descriptions["Lattice Deform"] =
                "Lattice (Free-Form Deformation - FFD) embeds mesh in 3D grid; moving lattice control points deforms mesh. Algorithm: 1) Create lattice (3D grid of control "
                "points), 2) Compute barycentric coordinates of each vertex within lattice cells, 3) Move lattice points (user manipulation), 4) Reconstruct vertex positions "
                "using trilinear/trivariate interpolation. Bernstein polynomials or B-splines provide smooth influence. Benefits: intuitive sculpting (adjust few control "
                "points affects many vertices), smooth deformation, and independent of mesh resolution. Applications: character animation (facial expressions), modeling (broad "
                "shape changes), morphing, and medical imaging (organ shape alignment). Resolution trade-off: coarse lattice = broad deformation, fine lattice = localized "
                "control. FFD is fundamental in animation—separates control (low-res lattice) from detail (high-res mesh), enabling efficient, artist-friendly deformation.";

            descriptions["Mirror"] =
                "Mirror transformation reflects geometry across plane: typically XY (z→-z), YZ (x→-x), or XZ (y→-y) planes. Point reflection across plane ax+by+cz+d=0: "
                "p' = p - 2(n·p + d)n where n=(a,b,c) is plane normal. For coordinate plane mirrors: simply negate appropriate coordinate. Topology: mirror doesn't change "
                "structure, but winding order reverses (inside→outside), requiring face orientation flip. Applications: modeling symmetric objects (faces, cars, buildings—create "
                "half, mirror for full), creating reflections, generating symmetric textures, and ensuring bilateral symmetry. Often combined with mesh merging to create "
                "continuous symmetric model. Mirror modifier common in 3D software—edit one side, other updates automatically. Demonstrates geometric symmetry operations—reduces "
                "modeling effort by exploiting symmetry, essential for efficient creation of symmetric objects.";

            descriptions["Array"] =
                "Array modifier creates multiple copies of object with offset transformation: each copy is transformed by translation, rotation, and/or scale. Linear array: "
                "offset vector Δ applied n times: position_i = position_0 + i·Δ. Radial array: rotate by angle θ = 360°/n around axis. Combined transformations enable: "
                "staircases (translate up + rotate), spirals (translate + rotate), and scaling progressions. Parameters: count, offset type (constant/relative/object), and "
                "transformations per step. End caps can close array (first/last different geometry). Applications: architectural elements (columns, windows), mechanical parts "
                "(bolts, teeth), procedural patterns, and animation (motion trails). Arrays with deformers (curve modifier) create complex forms (spiral staircases following "
                "path). Efficient instancing—render many objects from single geometry, common in games/real-time graphics. Demonstrates transformation sequences and geometric "
                "patterns through iteration.";

            descriptions["Shell"] =
                "Shell modifier creates offset surface at fixed distance from original: generates surface parallel to input at distance d. Implementation: 1) Calculate vertex "
                "normals, 2) Offset vertices: p' = p ± d·n, 3) Optional: connect original and offset surfaces with walls (quads between boundaries). Similar to solidify but "
                "doesn't necessarily close volume—can create open shell. Applications: architectural modeling (walls with thickness), mechanical parts (sheet metal), creating "
                "standoff surfaces, and level-set evolution visualization. Challenges: self-intersections in high-curvature concave regions (offset may fold), normal consistency, "
                "and crease handling. Exact offset surfaces are complex (non-trivial geometry); practical implementations approximate. Used in CAD/CAM for tool path generation "
                "(milling with ball-end mill creates offset surface). Demonstrates surface evolution and geometric offset—fundamental in manufacturing and design.";

            descriptions["Displace"] =
                "Displacement maps vertex positions using texture or procedural function: p' = p + D(u,v)·n where D(u,v) is displacement amount from texture/function at UV "
                "coordinates (u,v), and n is displacement direction (typically normal). Grayscale texture encodes height: black=negative displacement, white=positive. Vector "
                "displacement uses RGB for (x,y,z) offsets, enabling overhangs and complex detail. Benefits: adds geometric detail without modeling, resolution-independent "
                "(sub-pixel detail in rendering), and dynamic (can be animated). Applications: terrain (height maps), fine surface detail (wrinkles, pores, engravings), "
                "ocean waves, and material weathering. Requires sufficient mesh resolution (can't displace between vertices). Displacement vs. normal mapping: displacement "
                "is true geometry (correct silhouettes, shadowing) while normal mapping is lighting trick. Essential for film-quality rendering—adds massive geometric detail "
                "efficiently.";

            descriptions["Remesh"] =
                "Remeshing creates new mesh with desired properties (uniform triangles, isotropy, alignment) from input mesh. Approaches: Delaunay-based (Voronoi diagram→Delaunay "
                "triangulation), isotropic remeshing (iterative edge splits/flips/collapses targeting uniform edge length), or quad remeshing (field-guided quadrangulation). "
                "Goals: uniform element size, isotropic triangles (equilateral), alignment to features (edges, ridges), or specific topology. CVT (Centroidal Voronoi Tessellation) "
                "distributes points optimally, then Delaunay triangulates. Applications: improving mesh quality for simulation (FEM requires good elements), animation (clean "
                "topology deforms better), 3D printing (removing artifacts), and converting between triangle/quad meshes. Remeshing fixes common problems: stretched triangles, "
                "inconsistent density, and poor connectivity. Essential mesh processing operation—transforms arbitrary mesh into high-quality, application-appropriate representation. "
                "Balances triangle count, quality, and feature preservation.";

            descriptions["Extrude"] =
                "Extrusion duplicates faces and moves them along direction (normal or specified vector), connecting original and new faces with side walls. Per-face extrusion: "
                "each face extrudes independently. Region extrusion: selected faces extrude as connected group. Algorithm: 1) Duplicate selected faces, 2) Offset along direction "
                "d: p'= p + d, 3) Create quads connecting original edges to new edges. Parameters: distance, direction (along normals/axis/custom), and scale (taper). Effects: "
                "adds depth, creates 3D from 2D profiles, and builds geometric complexity. Applications: architectural modeling (windows, decorative elements), mechanical parts "
                "(boss features, ribs), text (3D text from 2D curves), and incremental modeling (building geometry step-by-step). Related to sweep operations (extrude along path). "
                "Fundamental modeling operation—most complex shapes built by extruding and modifying simple profiles. Essential in CAD and artistic modeling workflows.";

            descriptions["Voxelize"] =
                "Voxelization converts mesh to volumetric grid of voxels (3D pixels): cells inside mesh are solid, outside are empty. Approaches: surface voxelization (mark "
                "cells surface intersects) or solid voxelization (determine inside/outside via ray casting, checking if ray from cell intersects mesh odd/even times). Resolution "
                "determines accuracy vs. memory. Outputs: binary occupancy, distance field (signed distance to surface), or density values. Applications: 3D printing (toolpath "
                "generation), physics simulation (fluid/smoke interaction with objects), CSG operations (boolean operations on voxel grids), medical imaging (volumetric rendering), "
                "and minecraft-style representations. Voxel data enables efficient spatial queries, boolean operations, and volumetric effects. Trade-off: resolution vs. memory—high-res "
                "voxelization is memory intensive. Conversion: voxels→mesh via marching cubes. Demonstrates discretization of continuous geometry—bridging mesh and volume representations.";

            descriptions["Laplacian Relax"] =
                "Laplacian relaxation repositions vertices to minimize bending energy, creating fair (smooth) surfaces. Each iteration solves: L(v) = 0 where L is Laplace-Beltrami "
                "operator (cotangent-weighted Laplacian). This makes each vertex weighted barycenter of neighbors. The method preserves area better than uniform Laplacian and "
                "respects surface metric. Iterative solver or direct solve (sparse linear system). Boundary handling: fix boundary vertices (constrained smoothing) or allow movement. "
                "Effects: reduces curvature variation, creates minimal-variation surfaces, and removes noise while preserving overall shape better than uniform smoothing. Applications: "
                "mesh fairing (aesthetic smoothness), data smoothing (medical imaging, scanning artifacts), and shape optimization. Related to discrete differential geometry—Laplacian "
                "encodes surface curvature. Demonstrates geometric PDE discretization—continuum fairness concept becomes discrete algorithm operating on mesh structure.";

            descriptions["CMA-ES Optimize"] =
                "Covariance Matrix Adaptation Evolution Strategy (CMA-ES) is a state-of-the-art evolutionary algorithm for derivative-free optimization. It evolves distribution "
                "of candidate solutions: 1) Sample population from multivariate Gaussian, 2) Evaluate fitness, 3) Update mean toward better solutions, 4) Update covariance "
                "matrix (adapts search distribution to fitness landscape). The algorithm learns: step size (global learning rate), principal directions (which parameter combinations "
                "are important), and scaling (relative importance of parameters). Advantages: invariant to transformations, handles ill-conditioned problems, and requires minimal "
                "parameter tuning. Applications in mesh optimization: minimize drag (aerodynamic shapes), maximize strength-to-weight (topology optimization), optimize aesthetics "
                "(user-defined criteria), and evolve parametric designs. CMA-ES explores design space efficiently, discovering non-obvious optimal shapes. Demonstrates modern "
                "optimization—sophisticated evolutionary algorithm outperforming simpler genetic algorithms, especially in continuous parameter spaces with complex fitness landscapes.";

            descriptions["NSGA-II Multi-Objective"] =
                "Non-dominated Sorting Genetic Algorithm II (NSGA-II) performs multi-objective optimization, finding Pareto-optimal solutions (trade-offs where improving one "
                "objective worsens another). Algorithm: 1) Generate population, 2) Evaluate multiple objectives per solution, 3) Non-dominated sorting (rank by Pareto dominance), "
                "4) Crowding distance (maintain diversity), 5) Selection (prefer lower rank, higher crowding distance), 6) Crossover and mutation, 7) Repeat. Output: Pareto "
                "front—set of solutions representing trade-offs. In mesh optimization, objectives might include: structural strength vs. weight, cost vs. performance, or aesthetics "
                "vs. manufacturability. Benefits: finds multiple solutions showing trade-off space, maintains diversity (avoids converging to single solution), and efficient (elitism). "
                "Applications: engineering design (conflicting requirements), multi-criteria decision making, and robot morphology evolution. Demonstrates multi-objective optimization—recognition "
                "that real problems have competing goals, need family of solutions representing trade-offs rather than single 'best' solution.";

            descriptions["RBF Surrogate"] =
                "Radial Basis Function (RBF) surrogate-based optimization uses interpolation to approximate expensive objective function, enabling efficient optimization. Process: "
                "1) Sample design space (e.g., Latin hypercube), 2) Evaluate true objective at samples, 3) Build RBF interpolant: f̃(x) = Σw_i·φ(||x-x_i||) where φ is basis "
                "function (Gaussian, multiquadric, thin-plate spline), 4) Optimize surrogate (cheap) to find promising point, 5) Evaluate true objective there, 6) Update surrogate, "
                "7) Repeat. Benefits: reduces expensive evaluations (FEA, CFD), smooth approximation guides search, and handles black-box functions. Applications: aerodynamic "
                "shape optimization (where each evaluation requires CFD simulation), structural optimization (FEA for each design), and hyperparameter tuning (expensive training). "
                "RBF surrogates balance exploration (global search) and exploitation (local refinement). Demonstrates smart optimization—learning approximate model enables "
                "finding good solutions with far fewer expensive evaluations than direct search.";

            // POINT CLOUD
            descriptions["Random Scatter"] =
                "Random scatter distributes points uniformly within 3D volume using pseudorandom number generation. For axis-aligned box: x = x_min + rand()·(x_max-x_min), "
                "similarly for y, z. For sphere: generate (r,θ,φ) with r^(1/3) (uniform volume distribution, not surface), θ ∈ [0,2π], φ = arccos(2·rand()-1) (uniform "
                "on sphere surface). Convert to Cartesian. For arbitrary volume: rejection sampling (generate in bounding volume, reject if outside shape). Benefits: simple "
                "implementation, unbiased sampling, and pseudorandom seed enables reproducibility. Downsides: no blue noise properties (clustering possible), not maximal coverage. "
                "Applications: particle system initialization, Monte Carlo simulations, vegetation placement, and procedural decoration. Starting point for more sophisticated "
                "sampling—often followed by relaxation (Lloyd's algorithm) for better distribution. Demonstrates basic stochastic sampling—foundation for probabilistic methods "
                "in graphics and simulation.";

            descriptions["Grid Sample"] =
                "Regular grid sampling places points at lattice positions: p_ijk = origin + (i·Δx, j·Δy, k·Δz) for integers i,j,k. Spacing Δ controls density. Grid types: "
                "cubic (aligned with axes), FCC (face-centered cubic—denser packing), BCC (body-centered cubic), or hexagonal. Benefits: perfectly uniform distribution, predictable "
                "topology (known neighbors), and efficient indexing (spatial queries via arithmetic). Downsides: aliasing patterns in rendering/simulation, anisotropy (directional "
                "bias), and artificial regularity. Applications: voxel grids (Minecraft-style), fluid simulation (MAC grid, marker-and-cell), volume rendering (CT/MRI data), "
                "and structured mesh generation. Jittered grid adds small random offsets, reducing aliasing while maintaining structure. Demonstrates spatial discretization—fundamental "
                "in numerical methods, volume graphics, and simulation where regular topology simplifies algorithms and data structures.";

            descriptions["Poisson Disk"] =
                "Poisson disk sampling creates blue noise distribution: no two points closer than minimum distance r, while maximizing coverage. Properties: no clustering (unlike "
                "random), uniform coverage, and visually pleasing distribution. Algorithms: dart throwing (rejection sampling—slow), Bridson's algorithm (O(n)—grid-accelerated), "
                "or tile-based (use pre-computed tiles). The process avoids low-frequency clumping and high-frequency aliasing. Parameters: minimum distance r (controls density), "
                "maximum attempts (rejection tolerance). Applications: stippling and artistic rendering, object placement (trees, rocks—natural-looking distribution), texture "
                "synthesis (pattern generation), and sampling for integration (Monte Carlo with stratification). Poisson disk achieves good coverage with fewer samples than random—optimal "
                "sampling for many graphics/simulation tasks. Demonstrates spatial statistics—blue noise is 'gold standard' for sampling, avoiding both clustering and regularity artifacts.";

            descriptions["Blue Noise"] =
                "Blue noise sampling creates point distributions with no low-frequency components—energy concentrated at high frequencies. Characterized by radial power spectrum "
                "with reduced energy near origin. Compared to white noise (uniform spectrum) and Poisson disk (hard minimum distance), blue noise allows variable spacing while "
                "maintaining good coverage. Generation methods: Lloyd relaxation (iterative optimization), simulated annealing, or void-and-cluster algorithm. Properties: perceptually "
                "pleasing (human visual system sensitive to low-frequency clustering), good for anti-aliasing (sampling with blue noise reduces aliasing), and optimal convergence "
                "in integration. Applications: rendering (sample patterns for Monte Carlo), dithering (blue noise dithering superior to ordered dither), and printing (halftoning). "
                "Blue noise is optimal for many sampling tasks—balances uniformity with randomness. Demonstrates signal processing meeting computer graphics—spectral properties "
                "of sampling patterns profoundly affect visual quality and numerical accuracy.";

            descriptions["Surface Sample"] =
                "Surface sampling generates points on mesh surface: distributed by area (more samples on larger faces) or uniformly (perceptually even coverage). Area-weighted: "
                "select face with probability proportional to area, then random barycentric coordinates (u,v) with u+v≤1: p = (1-u-v)·v₀ + u·v₁ + v·v₂. Uniform distribution "
                "requires: computing face areas, building cumulative distribution, and sampling. Normal information preserved: at each point, store surface normal for oriented "
                "sampling. Applications: texture baking (capture surface detail in point cloud), particle system emission (sparks from geometry), surface detail transfer, and "
                "point-based rendering. Adaptive sampling concentrates points in high-curvature regions. Related to photon mapping (light transport) and many-light methods "
                "(rendering). Demonstrates sampling geometric manifolds—converting continuous surface to discrete point representation while preserving spatial and geometric properties.";

            descriptions["Volume Sample"] =
                "Volume sampling generates points inside 3D volume (interior of mesh). Methods: grid sampling (regular 3D lattice), Monte Carlo (random points, reject if outside—requires "
                "inside/outside test), or tetrahedral mesh sampling (if volume meshed). Inside/outside test: ray casting (count intersections—odd=inside) or winding number (more "
                "robust). For signed distance field (SDF): sample where SDF < 0. Adaptive sampling: denser in regions of interest, guided by field gradients or geometric features. "
                "Applications: volumetric effects (smoke, fire particles), physics simulation (material point method), medical imaging (volumetric analysis), and mass property "
                "computation (center of mass, moment of inertia via sampling). Volume sampling enables volumetric representations from surface meshes—converting boundary representation "
                "to interior points. Demonstrates volume discretization—enabling volumetric algorithms (SPH, material point method) and analysis from boundary-defined geometry.";

            descriptions["Fibonacci Sphere"] =
                "Fibonacci sphere distributes points uniformly on sphere using golden angle φ = 180°(3-√5) ≈ 137.5°. Algorithm: for i in [0,n-1], set z = 1 - 2i/(n-1) (uniform "
                "in z from -1 to 1), θ = i·φ (cumulative golden angle), r = √(1-z²) (radius at height z), then (x,y,z) = (r·cos(θ), r·sin(θ), z). Result: remarkably uniform "
                "distribution—no polar clustering, even spacing, low discrepancy. Mathematics: golden angle is most irrational number—maximally avoids resonance, preventing aligned "
                "spirals. Properties: deterministic (reproducible), efficient O(n), and excellent uniformity. Applications: global illumination (environment map sampling), photographic "
                "spherical panoramas (viewpoint selection), molecular modeling (surface points on proteins), and icosahedron alternative. Demonstrates number theory in geometry—golden "
                "ratio's unique properties enable elegant sphere sampling algorithm, beautiful spiraling pattern on sphere surface.";

            descriptions["Halton Sequence"] =
                "Halton sequence is low-discrepancy (quasi-random) sequence: deterministic but appears random, fills space more uniformly than random numbers. Construction: for "
                "dimension d, use prime base p_d. Halton number H_n(p) = Σ(a_i·p^(-i-1)) where a_i are digits of n in base p. Multi-dimensional: combine different bases (typically "
                "(2,3), (2,3,5) for 2D, 3D). Example: base 2: n=5 (binary 101) → 0.101₂ = 0.625. Properties: space-filling (asymptotically uniform), stratification (progressive "
                "coverage), and deterministic (reproducible). Benefits: better convergence than random in Monte Carlo integration, structured coverage, and incremental (add samples "
                "maintaining quality). Applications: rendering (sample patterns for anti-aliasing), numerical integration, and spatial sampling. Correlation between dimensions "
                "(issue for high dimensions). Demonstrates quasi-random sequences—deterministic sequences with randomness-like distribution properties, superior to pseudorandom for "
                "many applications requiring uniform coverage.";

            descriptions["Sobol Sequence"] =
                "Sobol sequence is another low-discrepancy quasi-random sequence with excellent multi-dimensional uniformity. Based on direction numbers and primitive polynomials "
                "over GF(2) (binary field). Constructed via: successive integers XORed with Gray code, then XORed with direction numbers. Properties: 2D projections maintain low "
                "discrepancy better than Halton (less correlation), base-2 construction enables efficient generation, and stratification properties (successive powers of 2 points "
                "form stratified samples). Benefits: better for higher dimensions than Halton, cache-friendly generation, and widely used in finance (Monte Carlo valuation). "
                "Applications: quasi-Monte Carlo integration, rendering (sampling light transport paths), parameter space exploration, and sensitivity analysis. Requires initialization "
                "tables (direction numbers) but then very fast. Demonstrates advanced quasi-random sequences—carefully designed for multi-dimensional uniformity, essential in "
                "high-dimensional integration and simulation where random/pseudorandom perform poorly.";

            descriptions["Stratified Sample"] =
                "Stratified sampling divides domain into strata (subdivisions) and samples within each stratum, ensuring coverage while maintaining randomness. 2D: divide into grid, "
                "one sample per cell at random location within cell. 3D: divide volume into voxels, sample randomly in each. Benefits: guaranteed coverage (no large empty regions), "
                "reduced variance in integration (stratification theorem), and controllable density. Compared to random (possible clustering) and regular grid (aliasing), stratified "
                "balances uniformity and randomness. Jittered sampling is common variant: grid cells with random offsets. Applications: rendering (anti-aliasing—one sample per "
                "pixel region), Monte Carlo integration (variance reduction), and statistical sampling (market research surveys sample from demographic strata). Multi-dimensional "
                "stratification (Latin hypercube) ensures projection properties. Demonstrates variance reduction techniques—structure in sampling reduces statistical error compared "
                "to pure randomness, fundamental in Monte Carlo methods.";

            descriptions["Sphere Packing"] =
                "Sphere packing places non-overlapping spheres to fill space efficiently. Dense packing: FCC (face-centered cubic) or HCP (hexagonal close-packed) achieve ~74% "
                "density (Kepler conjecture, proved 1998). Algorithms: advancing front (place spheres incrementally), molecular dynamics (repulsive forces converge to packing), "
                "or sequential random packing. Random sphere packing: place spheres randomly, ensuring no overlap, typically achieves ~64% density. Parameters: sphere radii (mono-disperse "
                "or poly-disperse), packing density, and boundary conditions. Applications: granular materials simulation (sand, grains), porous media modeling (catalysts, filters), "
                "colloidal systems, and procedural object placement (rocks, bubbles). Point centers become point cloud. Demonstrates geometric optimization and physics—sphere packing "
                "relates to materials science, crystallography, and combinatorial geometry. Optimal packing remains open problem for many configurations.";

            descriptions["Voxel Downsample"] =
                "Voxel downsampling reduces point cloud density by partitioning space into voxel grid and keeping one representative point per occupied voxel. Algorithm: 1) Determine "
                "bounding box and voxel size, 2) Assign each point to voxel, 3) For each voxel, compute representative (centroid, median, or random point from that voxel), 4) Output "
                "representatives. Effect: uniform density reduction, removes redundant points, and preserves overall shape. Benefits: controllable output size (voxel size determines "
                "reduction), fast O(n), and uniform sampling. Applications: point cloud simplification (millions→thousands), pre-processing for registration (ICP works better with "
                "reduced data), real-time processing (LiDAR downsampling for robotics), and memory reduction. Voxel size trade-off: large→aggressive reduction but lost detail, "
                "small→more detail but less reduction. Demonstrates spatial discretization for simplification—efficiently reduces data volume while maintaining spatial distribution.";

            descriptions["Normal Estimate"] =
                "Normal estimation computes surface normal vectors for point clouds lacking explicit normals. Common method: PCA (Principal Component Analysis) on k-nearest neighbors—normal "
                "is eigenvector corresponding to smallest eigenvalue (direction of minimum variance). Algorithm: 1) Find k neighbors (kd-tree), 2) Compute covariance matrix "
                "C = Σ((p_i-p̄)(p_i-p̄)ᵀ), 3) Eigendecomposition—smallest eigenvalue's eigenvector is normal, 4) Orient consistently (propagate orientation across neighbors). Parameters: "
                "neighborhood size k (larger→smoother normals, lost detail; smaller→noisy). Alternatives: robust fitting (RANSAC plane fit), MLS (moving least squares), or Hoppe's "
                "method. Challenges: orientation ambiguity (point toward viewer or away?), noise sensitivity, and sharp feature handling. Applications: rendering (lighting requires "
                "normals), surface reconstruction (Poisson reconstruction uses normals), and feature extraction. Demonstrates differential geometry discretization—computing tangent "
                "space from points approximates continuous surface properties, essential for point-based processing.";

            descriptions["Outlier Removal"] =
                "Outlier removal filters point clouds to eliminate noise points far from surface. Statistical method: for each point, compute mean distance to k neighbors d̄ and "
                "standard deviation σ; remove points where d̄ > μ + n·σ (μ, σ are global statistics, n is threshold). Radius-based: remove points with fewer than min_pts neighbors "
                "within radius r. RANSAC-based: fit surface (plane, sphere), remove points far from consensus model. Benefits: cleans scan data (sensor noise, reflections), improves "
                "downstream algorithms (registration, reconstruction), and removes artifacts. Parameters: k neighbors, standard deviation multiplier, or radius + min_pts. Trade-off: "
                "aggressive→removes features, conservative→retains noise. Applications: 3D scanning (LiDAR, structured light—always has noise), autonomous vehicles (environment perception), "
                "and pre-processing for meshing. Demonstrates robust statistics in geometry—identifying and removing anomalous data critical for reliable processing of real-world scans.";

            descriptions["ICP Align"] =
                "Iterative Closest Point (ICP) aligns two point clouds by finding rigid transformation (rotation + translation) minimizing point-to-point distance. Algorithm: "
                "1) Find correspondences (nearest neighbor—each source point's closest target point), 2) Compute optimal transformation minimizing Σ||R·p_s + t - p_t||² (closed-form "
                "solution via SVD), 3) Apply transformation to source, 4) Iterate until convergence. Variants: point-to-plane (uses target normals—faster convergence), robust metrics "
                "(reduce outlier influence), and multi-scale (coarse-to-fine). Requirements: good initialization (close starting alignment) or global registration first. Benefits: "
                "simple and effective, provable convergence (local minimum), widely implemented. Applications: 3D scanning (aligning multiple scans), SLAM (robot localization), medical "
                "imaging (registration), and AR (tracking). Limitations: local minimum (requires good initialization), sensitive to outliers, and non-rigid deformations. Demonstrates "
                "geometric optimization—iteratively refining alignment fundamental in 3D vision and robotics, enabling reconstruction from multiple viewpoints.";
        }

        return descriptions.value(algorithmName, "Detailed information not available for this algorithm.");
    }
};
