#include "MeshExporter.h"
#include <fstream>
#include <sstream>
#include <iomanip>
#include <algorithm>
#include <ctime>
#include <cstdint>

std::string MeshExporter::getFileExtension(const std::string& filepath) {
    size_t dotPos = filepath.find_last_of('.');
    if (dotPos == std::string::npos) {
        return "";
    }
    return toLower(filepath.substr(dotPos + 1));
}

std::string MeshExporter::toLower(const std::string& str) {
    std::string result = str;
    std::transform(result.begin(), result.end(), result.begin(), ::tolower);
    return result;
}

bool MeshExporter::exportMesh(const std::string& filepath, const Mesh& mesh) {
    std::string ext = getFileExtension(filepath);

    if (ext == "obj") {
        return exportOBJ(filepath, mesh);
    } else if (ext == "stl") {
        return exportSTL(filepath, mesh);
    } else if (ext == "ply") {
        return exportPLY(filepath, mesh);
    } else if (ext == "json") {
        return exportJSON(filepath, mesh);
    } else if (ext == "x3d") {
        return exportX3D(filepath, mesh);
    } else if (ext == "wrl") {
        return exportWRL(filepath, mesh);
    } else if (ext == "dxf") {
        return exportDXF(filepath, mesh);
    } else if (ext == "3ds") {
        return export3DS(filepath, mesh);
    }

    return false; // Unknown format
}

// OBJ Format - Wavefront Object
bool MeshExporter::exportOBJ(const std::string& filepath, const Mesh& mesh) {
    std::ofstream file(filepath);
    if (!file.is_open()) return false;

    file << "# Exported from codewelt.com/mesher\n";
    file << "# Vertices: " << mesh.getVertexCount() << "\n";
    file << "# Faces: " << mesh.getTriangleCount() << "\n\n";

    file << "o " << mesh.getName() << "\n\n";

    // Write vertices
    for (size_t i = 0; i < mesh.getVertexCount(); i++) {
        const Vec3& pos = mesh.getVertex(i).position;
        file << "v " << pos.x << " " << pos.y << " " << pos.z << "\n";
    }

    file << "\n";

    // Write normals
    for (size_t i = 0; i < mesh.getVertexCount(); i++) {
        const Vec3& normal = mesh.getVertex(i).normal;
        file << "vn " << normal.x << " " << normal.y << " " << normal.z << "\n";
    }

    file << "\n";

    // Write texture coordinates
    for (size_t i = 0; i < mesh.getVertexCount(); i++) {
        const Vertex& v = mesh.getVertex(i);
        file << "vt " << v.u << " " << v.v << "\n";
    }

    file << "\n";

    // Write faces (OBJ indices start at 1)
    for (size_t i = 0; i < mesh.getTriangleCount(); i++) {
        const Triangle& tri = mesh.getTriangle(i);
        file << "f " << (tri.v0 + 1) << "/" << (tri.v0 + 1) << "/" << (tri.v0 + 1) << " "
             << (tri.v1 + 1) << "/" << (tri.v1 + 1) << "/" << (tri.v1 + 1) << " "
             << (tri.v2 + 1) << "/" << (tri.v2 + 1) << "/" << (tri.v2 + 1) << "\n";
    }

    file.close();
    return true;
}

// STL Format - STereoLithography (ASCII)
bool MeshExporter::exportSTL(const std::string& filepath, const Mesh& mesh) {
    std::ofstream file(filepath);
    if (!file.is_open()) return false;

    file << "solid " << mesh.getName() << "\n";

    for (size_t i = 0; i < mesh.getTriangleCount(); i++) {
        const Triangle& tri = mesh.getTriangle(i);
        const Vec3& p0 = mesh.getVertex(tri.v0).position;
        const Vec3& p1 = mesh.getVertex(tri.v1).position;
        const Vec3& p2 = mesh.getVertex(tri.v2).position;

        // Calculate face normal
        Vec3 edge1 = p1 - p0;
        Vec3 edge2 = p2 - p0;
        Vec3 normal = edge1.cross(edge2);
        normal.normalize();

        file << "  facet normal " << normal.x << " " << normal.y << " " << normal.z << "\n";
        file << "    outer loop\n";
        file << "      vertex " << p0.x << " " << p0.y << " " << p0.z << "\n";
        file << "      vertex " << p1.x << " " << p1.y << " " << p1.z << "\n";
        file << "      vertex " << p2.x << " " << p2.y << " " << p2.z << "\n";
        file << "    endloop\n";
        file << "  endfacet\n";
    }

    file << "endsolid " << mesh.getName() << "\n";
    file.close();
    return true;
}

// PLY Format - Stanford Polygon File Format (ASCII)
bool MeshExporter::exportPLY(const std::string& filepath, const Mesh& mesh) {
    std::ofstream file(filepath);
    if (!file.is_open()) return false;

    file << "ply\n";
    file << "format ascii 1.0\n";
    file << "comment Exported from codewelt.com/mesher\n";
    file << "element vertex " << mesh.getVertexCount() << "\n";
    file << "property float x\n";
    file << "property float y\n";
    file << "property float z\n";
    file << "property float nx\n";
    file << "property float ny\n";
    file << "property float nz\n";
    file << "property uchar red\n";
    file << "property uchar green\n";
    file << "property uchar blue\n";
    file << "element face " << mesh.getTriangleCount() << "\n";
    file << "property list uchar int vertex_indices\n";
    file << "end_header\n";

    // Write vertices with normals and colors
    for (size_t i = 0; i < mesh.getVertexCount(); i++) {
        const Vertex& v = mesh.getVertex(i);
        int r = static_cast<int>(v.color.x * 255);
        int g = static_cast<int>(v.color.y * 255);
        int b = static_cast<int>(v.color.z * 255);

        file << v.position.x << " " << v.position.y << " " << v.position.z << " "
             << v.normal.x << " " << v.normal.y << " " << v.normal.z << " "
             << r << " " << g << " " << b << "\n";
    }

    // Write faces
    for (size_t i = 0; i < mesh.getTriangleCount(); i++) {
        const Triangle& tri = mesh.getTriangle(i);
        file << "3 " << tri.v0 << " " << tri.v1 << " " << tri.v2 << "\n";
    }

    file.close();
    return true;
}

// JSON Format - Simple JavaScript Object Notation
bool MeshExporter::exportJSON(const std::string& filepath, const Mesh& mesh) {
    std::ofstream file(filepath);
    if (!file.is_open()) return false;

    file << "{\n";
    file << "  \"name\": \"" << mesh.getName() << "\",\n";
    file << "  \"vertexCount\": " << mesh.getVertexCount() << ",\n";
    file << "  \"triangleCount\": " << mesh.getTriangleCount() << ",\n";
    file << "  \"vertices\": [\n";

    for (size_t i = 0; i < mesh.getVertexCount(); i++) {
        const Vertex& v = mesh.getVertex(i);
        file << "    {\n";
        file << "      \"position\": [" << v.position.x << ", " << v.position.y << ", " << v.position.z << "],\n";
        file << "      \"normal\": [" << v.normal.x << ", " << v.normal.y << ", " << v.normal.z << "],\n";
        file << "      \"color\": [" << v.color.x << ", " << v.color.y << ", " << v.color.z << "],\n";
        file << "      \"uv\": [" << v.u << ", " << v.v << "]\n";
        file << "    }";
        if (i < mesh.getVertexCount() - 1) file << ",";
        file << "\n";
    }

    file << "  ],\n";
    file << "  \"triangles\": [\n";

    for (size_t i = 0; i < mesh.getTriangleCount(); i++) {
        const Triangle& tri = mesh.getTriangle(i);
        file << "    [" << tri.v0 << ", " << tri.v1 << ", " << tri.v2 << "]";
        if (i < mesh.getTriangleCount() - 1) file << ",";
        file << "\n";
    }

    file << "  ]\n";
    file << "}\n";

    file.close();
    return true;
}

// X3D Format - Extensible 3D Graphics
bool MeshExporter::exportX3D(const std::string& filepath, const Mesh& mesh) {
    std::ofstream file(filepath);
    if (!file.is_open()) return false;

    file << "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n";
    file << "<!DOCTYPE X3D PUBLIC \"ISO//Web3D//DTD X3D 3.3//EN\" \"http://www.web3d.org/specifications/x3d-3.3.dtd\">\n";
    file << "<X3D profile='Interchange' version='3.3'>\n";
    file << "  <Scene>\n";
    file << "    <Shape>\n";
    file << "      <Appearance>\n";
    file << "        <Material diffuseColor='0.8 0.8 0.8'/>\n";
    file << "      </Appearance>\n";
    file << "      <IndexedFaceSet solid='true' coordIndex='";

    // Write indices
    for (size_t i = 0; i < mesh.getTriangleCount(); i++) {
        const Triangle& tri = mesh.getTriangle(i);
        file << tri.v0 << " " << tri.v1 << " " << tri.v2 << " -1 ";
    }

    file << "'>\n";
    file << "        <Coordinate point='";

    // Write vertices
    for (size_t i = 0; i < mesh.getVertexCount(); i++) {
        const Vec3& pos = mesh.getVertex(i).position;
        file << pos.x << " " << pos.y << " " << pos.z;
        if (i < mesh.getVertexCount() - 1) file << ", ";
    }

    file << "'/>\n";
    file << "        <Normal vector='";

    // Write normals
    for (size_t i = 0; i < mesh.getVertexCount(); i++) {
        const Vec3& normal = mesh.getVertex(i).normal;
        file << normal.x << " " << normal.y << " " << normal.z;
        if (i < mesh.getVertexCount() - 1) file << ", ";
    }

    file << "'/>\n";
    file << "      </IndexedFaceSet>\n";
    file << "    </Shape>\n";
    file << "  </Scene>\n";
    file << "</X3D>\n";

    file.close();
    return true;
}

// WRL Format - VRML (Virtual Reality Modeling Language)
bool MeshExporter::exportWRL(const std::string& filepath, const Mesh& mesh) {
    std::ofstream file(filepath);
    if (!file.is_open()) return false;

    file << "#VRML V2.0 utf8\n";
    file << "# Exported from codewelt.com/mesher\n\n";

    file << "Shape {\n";
    file << "  appearance Appearance {\n";
    file << "    material Material {\n";
    file << "      diffuseColor 0.8 0.8 0.8\n";
    file << "      specularColor 0.5 0.5 0.5\n";
    file << "      shininess 0.5\n";
    file << "    }\n";
    file << "  }\n";
    file << "  geometry IndexedFaceSet {\n";
    file << "    coord Coordinate {\n";
    file << "      point [\n";

    // Write vertices
    for (size_t i = 0; i < mesh.getVertexCount(); i++) {
        const Vec3& pos = mesh.getVertex(i).position;
        file << "        " << pos.x << " " << pos.y << " " << pos.z;
        if (i < mesh.getVertexCount() - 1) file << ",";
        file << "\n";
    }

    file << "      ]\n";
    file << "    }\n";
    file << "    normal Normal {\n";
    file << "      vector [\n";

    // Write normals
    for (size_t i = 0; i < mesh.getVertexCount(); i++) {
        const Vec3& normal = mesh.getVertex(i).normal;
        file << "        " << normal.x << " " << normal.y << " " << normal.z;
        if (i < mesh.getVertexCount() - 1) file << ",";
        file << "\n";
    }

    file << "      ]\n";
    file << "    }\n";
    file << "    coordIndex [\n";

    // Write indices
    for (size_t i = 0; i < mesh.getTriangleCount(); i++) {
        const Triangle& tri = mesh.getTriangle(i);
        file << "      " << tri.v0 << ", " << tri.v1 << ", " << tri.v2 << ", -1";
        if (i < mesh.getTriangleCount() - 1) file << ",";
        file << "\n";
    }

    file << "    ]\n";
    file << "    solid FALSE\n";
    file << "  }\n";
    file << "}\n";

    file.close();
    return true;
}

// DXF Format - Drawing Exchange Format (simplified 3D mesh)
bool MeshExporter::exportDXF(const std::string& filepath, const Mesh& mesh) {
    std::ofstream file(filepath);
    if (!file.is_open()) return false;

    file << "0\nSECTION\n";
    file << "2\nHEADER\n";
    file << "9\n$ACADVER\n1\nAC1015\n";
    file << "0\nENDSEC\n";

    file << "0\nSECTION\n";
    file << "2\nENTITIES\n";

    // Write each triangle as a 3DFACE entity
    for (size_t i = 0; i < mesh.getTriangleCount(); i++) {
        const Triangle& tri = mesh.getTriangle(i);
        const Vec3& p0 = mesh.getVertex(tri.v0).position;
        const Vec3& p1 = mesh.getVertex(tri.v1).position;
        const Vec3& p2 = mesh.getVertex(tri.v2).position;

        file << "0\n3DFACE\n";
        file << "8\n0\n"; // Layer
        file << "10\n" << p0.x << "\n";
        file << "20\n" << p0.y << "\n";
        file << "30\n" << p0.z << "\n";
        file << "11\n" << p1.x << "\n";
        file << "21\n" << p1.y << "\n";
        file << "31\n" << p1.z << "\n";
        file << "12\n" << p2.x << "\n";
        file << "22\n" << p2.y << "\n";
        file << "32\n" << p2.z << "\n";
        file << "13\n" << p2.x << "\n"; // Repeat last point for triangle
        file << "23\n" << p2.y << "\n";
        file << "33\n" << p2.z << "\n";
    }

    file << "0\nENDSEC\n";
    file << "0\nEOF\n";

    file.close();
    return true;
}

// 3DS Format - 3D Studio (fixed implementation)
bool MeshExporter::export3DS(const std::string& filepath, const Mesh& mesh) {
    std::ofstream file(filepath, std::ios::binary);
    if (!file.is_open()) return false;

    // Helper lambda to write chunk header
    auto writeChunkHeader = [&file](uint16_t id, uint32_t size) {
        file.write(reinterpret_cast<const char*>(&id), 2);
        file.write(reinterpret_cast<const char*>(&size), 4);
    };

    // Helper lambda to write string
    auto writeString = [&file](const std::string& str) {
        file.write(str.c_str(), str.length() + 1);
    };

    // Calculate chunk sizes
    std::string meshName = mesh.getName();
    if (meshName.empty()) meshName = "Mesh";
    
    uint32_t nameSize = meshName.length() + 1;
    uint32_t vertexListSize = 6 + 2 + mesh.getVertexCount() * 12; // header + count + vertices
    uint32_t faceListSize = 6 + 2 + mesh.getTriangleCount() * 8;  // header + count + faces
    uint32_t triMeshSize = 6 + vertexListSize + faceListSize;      // 0x4100 chunk
    uint32_t objectSize = 6 + nameSize + triMeshSize;              // 0x4000 chunk
    uint32_t editorSize = 6 + objectSize;                          // 0x3D3D chunk
    uint32_t mainSize = 6 + 10 + editorSize;                       // 0x4D4D chunk (includes version chunk)

    // Main chunk (0x4D4D) - Primary chunk
    writeChunkHeader(0x4D4D, mainSize);

    // Version chunk (0x0002)
    writeChunkHeader(0x0002, 10);
    uint32_t version = 3;
    file.write(reinterpret_cast<const char*>(&version), 4);

    // Editor chunk (0x3D3D) - 3D Editor chunk
    writeChunkHeader(0x3D3D, editorSize);

    // Object chunk (0x4000) - Named object
    writeChunkHeader(0x4000, objectSize);
    writeString(meshName);

    // Triangle mesh (0x4100) - Triangular mesh
    writeChunkHeader(0x4100, triMeshSize);

    // Vertex list (0x4110)
    writeChunkHeader(0x4110, vertexListSize);
    uint16_t vertexCount = static_cast<uint16_t>(mesh.getVertexCount());
    file.write(reinterpret_cast<const char*>(&vertexCount), 2);

    for (size_t i = 0; i < mesh.getVertexCount(); i++) {
        const Vec3& pos = mesh.getVertex(i).position;
        float x = pos.x, y = pos.y, z = pos.z;
        file.write(reinterpret_cast<const char*>(&x), 4);
        file.write(reinterpret_cast<const char*>(&y), 4);
        file.write(reinterpret_cast<const char*>(&z), 4);
    }

    // Face list (0x4120) - This is the critical chunk that was missing proper structure
    writeChunkHeader(0x4120, faceListSize);
    uint16_t faceCount = static_cast<uint16_t>(mesh.getTriangleCount());
    file.write(reinterpret_cast<const char*>(&faceCount), 2);

    for (size_t i = 0; i < mesh.getTriangleCount(); i++) {
        const Triangle& tri = mesh.getTriangle(i);
        uint16_t v0 = static_cast<uint16_t>(tri.v0);
        uint16_t v1 = static_cast<uint16_t>(tri.v1);
        uint16_t v2 = static_cast<uint16_t>(tri.v2);
        uint16_t flags = 0x0007; // All edges visible

        file.write(reinterpret_cast<const char*>(&v0), 2);
        file.write(reinterpret_cast<const char*>(&v1), 2);
        file.write(reinterpret_cast<const char*>(&v2), 2);
        file.write(reinterpret_cast<const char*>(&flags), 2);
    }

    file.close();
    return true;
}
