/**
 * @file main.cpp
 * @brief Entry point for Mesher - Parametric Modeler application
 *
 * This file contains the main function that initializes the Qt application,
 * sets up application metadata, applies global styles, and launches the main window.
 *
 * Copyright © 2025 Linus Suter
 * Released under the GNU/GPL License
 */

#include <QApplication>
#include <QFont>
#include <QPalette>
#include <QStyleFactory>
#include "MainWindow.h"

/**
 * @brief Application entry point
 *
 * Initializes the Qt application framework, configures application settings,
 * applies global stylesheet for tooltips, and displays the main window.
 *
 * @param argc Number of command line arguments
 * @param argv Array of command line argument strings
 * @return Application exit code (0 for normal exit)
 */
int main(int argc, char *argv[]) {
    // Initialize Qt application
    QApplication app(argc, argv);

    // Set global font size
    QFont font = app.font();
    font.setPointSize(9);  // Decreased from default (typically 10-11)
    app.setFont(font);

    // Enable Qt dark theme
    app.setStyle(QStyleFactory::create("Fusion"));

    QPalette darkPalette;
    darkPalette.setColor(QPalette::Window, QColor(53, 53, 53));
    darkPalette.setColor(QPalette::WindowText, Qt::white);
    darkPalette.setColor(QPalette::Base, QColor(25, 25, 25));
    darkPalette.setColor(QPalette::AlternateBase, QColor(53, 53, 53));
    darkPalette.setColor(QPalette::ToolTipBase, QColor(42, 42, 42));
    darkPalette.setColor(QPalette::ToolTipText, QColor(204, 204, 204));
    darkPalette.setColor(QPalette::Text, Qt::white);
    darkPalette.setColor(QPalette::Button, QColor(53, 53, 53));
    darkPalette.setColor(QPalette::ButtonText, Qt::white);
    darkPalette.setColor(QPalette::BrightText, Qt::red);
    darkPalette.setColor(QPalette::Link, QColor(42, 130, 218));
    darkPalette.setColor(QPalette::Highlight, QColor(42, 130, 218));
    darkPalette.setColor(QPalette::HighlightedText, Qt::black);
    darkPalette.setColor(QPalette::Disabled, QPalette::Text, QColor(127, 127, 127));
    darkPalette.setColor(QPalette::Disabled, QPalette::ButtonText, QColor(127, 127, 127));

    app.setPalette(darkPalette);

    // Set application metadata for settings storage and about dialogs
    app.setApplicationName("Mesher • Parametric Modeler");
    app.setOrganizationName("CodeWelt");
    app.setOrganizationDomain("codewelt.com");
    app.setApplicationVersion("1.0");

    // Set application icon (uses .ico on Windows, .png on other platforms)
    app.setWindowIcon(QIcon(":/winico.png"));

    // Apply global grayscale tooltip style for consistent dark theme
    app.setStyleSheet("QToolTip { background-color: #2a2a2a; color: #cccccc; border: 1px solid #555555; padding: 4px; }");

    // Create and show the main application window
    MainWindow window;
    window.show();

    // Enter the Qt event loop
    return app.exec();
}
