#include "MeshGenerator.h"
#include <map>
#include <cstdint>

// Marching cubes tables
int MeshGenerator::edgeTable[256] = {
    0x0, 0x109, 0x203, 0x30a, 0x406, 0x50f, 0x605, 0x70c,
    0x80c, 0x905, 0xa0f, 0xb06, 0xc0a, 0xd03, 0xe09, 0xf00,
    0x190, 0x99, 0x393, 0x29a, 0x596, 0x49f, 0x795, 0x69c,
    0x99c, 0x895, 0xb9f, 0xa96, 0xd9a, 0xc93, 0xf99, 0xe90,
    0x230, 0x339, 0x33, 0x13a, 0x636, 0x73f, 0x435, 0x53c,
    0xa3c, 0xb35, 0x83f, 0x936, 0xe3a, 0xf33, 0xc39, 0xd30,
    0x3a0, 0x2a9, 0x1a3, 0xaa, 0x7a6, 0x6af, 0x5a5, 0x4ac,
    0xbac, 0xaa5, 0x9af, 0x8a6, 0xfaa, 0xea3, 0xda9, 0xca0,
    0x460, 0x569, 0x663, 0x76a, 0x66, 0x16f, 0x265, 0x36c,
    0xc6c, 0xd65, 0xe6f, 0xf66, 0x86a, 0x963, 0xa69, 0xb60,
    0x5f0, 0x4f9, 0x7f3, 0x6fa, 0x1f6, 0xff, 0x3f5, 0x2fc,
    0xdfc, 0xcf5, 0xfff, 0xef6, 0x9fa, 0x8f3, 0xbf9, 0xaf0,
    0x650, 0x759, 0x453, 0x55a, 0x256, 0x35f, 0x55, 0x15c,
    0xe5c, 0xf55, 0xc5f, 0xd56, 0xa5a, 0xb53, 0x859, 0x950,
    0x7c0, 0x6c9, 0x5c3, 0x4ca, 0x3c6, 0x2cf, 0x1c5, 0xcc,
    0xfcc, 0xec5, 0xdcf, 0xcc6, 0xbca, 0xac3, 0x9c9, 0x8c0,
    0x8c0, 0x9c9, 0xac3, 0xbca, 0xcc6, 0xdcf, 0xec5, 0xfcc,
    0xcc, 0x1c5, 0x2cf, 0x3c6, 0x4ca, 0x5c3, 0x6c9, 0x7c0,
    0x950, 0x859, 0xb53, 0xa5a, 0xd56, 0xc5f, 0xf55, 0xe5c,
    0x15c, 0x55, 0x35f, 0x256, 0x55a, 0x453, 0x759, 0x650,
    0xaf0, 0xbf9, 0x8f3, 0x9fa, 0xef6, 0xfff, 0xcf5, 0xdfc,
    0x2fc, 0x3f5, 0xff, 0x1f6, 0x6fa, 0x7f3, 0x4f9, 0x5f0,
    0xb60, 0xa69, 0x963, 0x86a, 0xf66, 0xe6f, 0xd65, 0xc6c,
    0x36c, 0x265, 0x16f, 0x66, 0x76a, 0x663, 0x569, 0x460,
    0xca0, 0xda9, 0xea3, 0xfaa, 0x8a6, 0x9af, 0xaa5, 0xbac,
    0x4ac, 0x5a5, 0x6af, 0x7a6, 0xaa, 0x1a3, 0x2a9, 0x3a0,
    0xd30, 0xc39, 0xf33, 0xe3a, 0x936, 0x83f, 0xb35, 0xa3c,
    0x53c, 0x435, 0x73f, 0x636, 0x13a, 0x33, 0x339, 0x230,
    0xe90, 0xf99, 0xc93, 0xd9a, 0xa96, 0xb9f, 0x895, 0x99c,
    0x69c, 0x795, 0x49f, 0x596, 0x29a, 0x393, 0x99, 0x190,
    0xf00, 0xe09, 0xd03, 0xc0a, 0xb06, 0xa0f, 0x905, 0x80c,
    0x70c, 0x605, 0x50f, 0x406, 0x30a, 0x203, 0x109, 0x0
};

int MeshGenerator::triTable[256][16] = {
    {-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 8, 3, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 1, 9, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {1, 8, 3, 9, 8, 1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {1, 2, 10, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 8, 3, 1, 2, 10, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {9, 2, 10, 0, 2, 9, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {2, 8, 3, 2, 10, 8, 10, 9, 8, -1, -1, -1, -1, -1, -1, -1},
    {3, 11, 2, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 11, 2, 8, 11, 0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {1, 9, 0, 2, 3, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {1, 11, 2, 1, 9, 11, 9, 8, 11, -1, -1, -1, -1, -1, -1, -1},
    {3, 10, 1, 11, 10, 3, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 10, 1, 0, 8, 10, 8, 11, 10, -1, -1, -1, -1, -1, -1, -1},
    {3, 9, 0, 3, 11, 9, 11, 10, 9, -1, -1, -1, -1, -1, -1, -1},
    {9, 8, 10, 10, 8, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {4, 7, 8, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {4, 3, 0, 7, 3, 4, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 1, 9, 8, 4, 7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {4, 1, 9, 4, 7, 1, 7, 3, 1, -1, -1, -1, -1, -1, -1, -1},
    {1, 2, 10, 8, 4, 7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {3, 4, 7, 3, 0, 4, 1, 2, 10, -1, -1, -1, -1, -1, -1, -1},
    {9, 2, 10, 9, 0, 2, 8, 4, 7, -1, -1, -1, -1, -1, -1, -1},
    {2, 10, 9, 2, 9, 7, 2, 7, 3, 7, 9, 4, -1, -1, -1, -1},
    {8, 4, 7, 3, 11, 2, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {11, 4, 7, 11, 2, 4, 2, 0, 4, -1, -1, -1, -1, -1, -1, -1},
    {9, 0, 1, 8, 4, 7, 2, 3, 11, -1, -1, -1, -1, -1, -1, -1},
    {4, 7, 11, 9, 4, 11, 9, 11, 2, 9, 2, 1, -1, -1, -1, -1},
    {3, 10, 1, 3, 11, 10, 7, 8, 4, -1, -1, -1, -1, -1, -1, -1},
    {1, 11, 10, 1, 4, 11, 1, 0, 4, 7, 11, 4, -1, -1, -1, -1},
    {4, 7, 8, 9, 0, 11, 9, 11, 10, 11, 0, 3, -1, -1, -1, -1},
    {4, 7, 11, 4, 11, 9, 9, 11, 10, -1, -1, -1, -1, -1, -1, -1},
    {9, 5, 4, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {9, 5, 4, 0, 8, 3, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 5, 4, 1, 5, 0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {8, 5, 4, 8, 3, 5, 3, 1, 5, -1, -1, -1, -1, -1, -1, -1},
    {1, 2, 10, 9, 5, 4, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {3, 0, 8, 1, 2, 10, 4, 9, 5, -1, -1, -1, -1, -1, -1, -1},
    {5, 2, 10, 5, 4, 2, 4, 0, 2, -1, -1, -1, -1, -1, -1, -1},
    {2, 10, 5, 3, 2, 5, 3, 5, 4, 3, 4, 8, -1, -1, -1, -1},
    {9, 5, 4, 2, 3, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 11, 2, 0, 8, 11, 4, 9, 5, -1, -1, -1, -1, -1, -1, -1},
    {0, 5, 4, 0, 1, 5, 2, 3, 11, -1, -1, -1, -1, -1, -1, -1},
    {2, 1, 5, 2, 5, 8, 2, 8, 11, 4, 8, 5, -1, -1, -1, -1},
    {10, 3, 11, 10, 1, 3, 9, 5, 4, -1, -1, -1, -1, -1, -1, -1},
    {4, 9, 5, 0, 8, 1, 8, 10, 1, 8, 11, 10, -1, -1, -1, -1},
    {5, 4, 0, 5, 0, 11, 5, 11, 10, 11, 0, 3, -1, -1, -1, -1},
    {5, 4, 8, 5, 8, 10, 10, 8, 11, -1, -1, -1, -1, -1, -1, -1},
    {9, 7, 8, 5, 7, 9, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {9, 3, 0, 9, 5, 3, 5, 7, 3, -1, -1, -1, -1, -1, -1, -1},
    {0, 7, 8, 0, 1, 7, 1, 5, 7, -1, -1, -1, -1, -1, -1, -1},
    {1, 5, 3, 3, 5, 7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {9, 7, 8, 9, 5, 7, 10, 1, 2, -1, -1, -1, -1, -1, -1, -1},
    {10, 1, 2, 9, 5, 0, 5, 3, 0, 5, 7, 3, -1, -1, -1, -1},
    {8, 0, 2, 8, 2, 5, 8, 5, 7, 10, 5, 2, -1, -1, -1, -1},
    {2, 10, 5, 2, 5, 3, 3, 5, 7, -1, -1, -1, -1, -1, -1, -1},
    {7, 9, 5, 7, 8, 9, 3, 11, 2, -1, -1, -1, -1, -1, -1, -1},
    {9, 5, 7, 9, 7, 2, 9, 2, 0, 2, 7, 11, -1, -1, -1, -1},
    {2, 3, 11, 0, 1, 8, 1, 7, 8, 1, 5, 7, -1, -1, -1, -1},
    {11, 2, 1, 11, 1, 7, 7, 1, 5, -1, -1, -1, -1, -1, -1, -1},
    {9, 5, 8, 8, 5, 7, 10, 1, 3, 10, 3, 11, -1, -1, -1, -1},
    {5, 7, 0, 5, 0, 9, 7, 11, 0, 1, 0, 10, 11, 10, 0, -1},
    {11, 10, 0, 11, 0, 3, 10, 5, 0, 8, 0, 7, 5, 7, 0, -1},
    {11, 10, 5, 7, 11, 5, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {10, 6, 5, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 8, 3, 5, 10, 6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {9, 0, 1, 5, 10, 6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {1, 8, 3, 1, 9, 8, 5, 10, 6, -1, -1, -1, -1, -1, -1, -1},
    {1, 6, 5, 2, 6, 1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {1, 6, 5, 1, 2, 6, 3, 0, 8, -1, -1, -1, -1, -1, -1, -1},
    {9, 6, 5, 9, 0, 6, 0, 2, 6, -1, -1, -1, -1, -1, -1, -1},
    {5, 9, 8, 5, 8, 2, 5, 2, 6, 3, 2, 8, -1, -1, -1, -1},
    {2, 3, 11, 10, 6, 5, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {11, 0, 8, 11, 2, 0, 10, 6, 5, -1, -1, -1, -1, -1, -1, -1},
    {0, 1, 9, 2, 3, 11, 5, 10, 6, -1, -1, -1, -1, -1, -1, -1},
    {5, 10, 6, 1, 9, 2, 9, 11, 2, 9, 8, 11, -1, -1, -1, -1},
    {6, 3, 11, 6, 5, 3, 5, 1, 3, -1, -1, -1, -1, -1, -1, -1},
    {0, 8, 11, 0, 11, 5, 0, 5, 1, 5, 11, 6, -1, -1, -1, -1},
    {3, 11, 6, 0, 3, 6, 0, 6, 5, 0, 5, 9, -1, -1, -1, -1},
    {6, 5, 9, 6, 9, 11, 11, 9, 8, -1, -1, -1, -1, -1, -1, -1},
    {5, 10, 6, 4, 7, 8, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {4, 3, 0, 4, 7, 3, 6, 5, 10, -1, -1, -1, -1, -1, -1, -1},
    {1, 9, 0, 5, 10, 6, 8, 4, 7, -1, -1, -1, -1, -1, -1, -1},
    {10, 6, 5, 1, 9, 7, 1, 7, 3, 7, 9, 4, -1, -1, -1, -1},
    {6, 1, 2, 6, 5, 1, 4, 7, 8, -1, -1, -1, -1, -1, -1, -1},
    {1, 2, 5, 5, 2, 6, 3, 0, 4, 3, 4, 7, -1, -1, -1, -1},
    {8, 4, 7, 9, 0, 5, 0, 6, 5, 0, 2, 6, -1, -1, -1, -1},
    {7, 3, 9, 7, 9, 4, 3, 2, 9, 5, 9, 6, 2, 6, 9, -1},
    {3, 11, 2, 7, 8, 4, 10, 6, 5, -1, -1, -1, -1, -1, -1, -1},
    {5, 10, 6, 4, 7, 2, 4, 2, 0, 2, 7, 11, -1, -1, -1, -1},
    {0, 1, 9, 4, 7, 8, 2, 3, 11, 5, 10, 6, -1, -1, -1, -1},
    {9, 2, 1, 9, 11, 2, 9, 4, 11, 7, 11, 4, 5, 10, 6, -1},
    {8, 4, 7, 3, 11, 5, 3, 5, 1, 5, 11, 6, -1, -1, -1, -1},
    {5, 1, 11, 5, 11, 6, 1, 0, 11, 7, 11, 4, 0, 4, 11, -1},
    {0, 5, 9, 0, 6, 5, 0, 3, 6, 11, 6, 3, 8, 4, 7, -1},
    {6, 5, 9, 6, 9, 11, 4, 7, 9, 7, 11, 9, -1, -1, -1, -1},
    {10, 4, 9, 6, 4, 10, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {4, 10, 6, 4, 9, 10, 0, 8, 3, -1, -1, -1, -1, -1, -1, -1},
    {10, 0, 1, 10, 6, 0, 6, 4, 0, -1, -1, -1, -1, -1, -1, -1},
    {8, 3, 1, 8, 1, 6, 8, 6, 4, 6, 1, 10, -1, -1, -1, -1},
    {1, 4, 9, 1, 2, 4, 2, 6, 4, -1, -1, -1, -1, -1, -1, -1},
    {3, 0, 8, 1, 2, 9, 2, 4, 9, 2, 6, 4, -1, -1, -1, -1},
    {0, 2, 4, 4, 2, 6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {8, 3, 2, 8, 2, 4, 4, 2, 6, -1, -1, -1, -1, -1, -1, -1},
    {10, 4, 9, 10, 6, 4, 11, 2, 3, -1, -1, -1, -1, -1, -1, -1},
    {0, 8, 2, 2, 8, 11, 4, 9, 10, 4, 10, 6, -1, -1, -1, -1},
    {3, 11, 2, 0, 1, 6, 0, 6, 4, 6, 1, 10, -1, -1, -1, -1},
    {6, 4, 1, 6, 1, 10, 4, 8, 1, 2, 1, 11, 8, 11, 1, -1},
    {9, 6, 4, 9, 3, 6, 9, 1, 3, 11, 6, 3, -1, -1, -1, -1},
    {8, 11, 1, 8, 1, 0, 11, 6, 1, 9, 1, 4, 6, 4, 1, -1},
    {3, 11, 6, 3, 6, 0, 0, 6, 4, -1, -1, -1, -1, -1, -1, -1},
    {6, 4, 8, 11, 6, 8, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {7, 10, 6, 7, 8, 10, 8, 9, 10, -1, -1, -1, -1, -1, -1, -1},
    {0, 7, 3, 0, 10, 7, 0, 9, 10, 6, 7, 10, -1, -1, -1, -1},
    {10, 6, 7, 1, 10, 7, 1, 7, 8, 1, 8, 0, -1, -1, -1, -1},
    {10, 6, 7, 10, 7, 1, 1, 7, 3, -1, -1, -1, -1, -1, -1, -1},
    {1, 2, 6, 1, 6, 8, 1, 8, 9, 8, 6, 7, -1, -1, -1, -1},
    {2, 6, 9, 2, 9, 1, 6, 7, 9, 0, 9, 3, 7, 3, 9, -1},
    {7, 8, 0, 7, 0, 6, 6, 0, 2, -1, -1, -1, -1, -1, -1, -1},
    {7, 3, 2, 6, 7, 2, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {2, 3, 11, 10, 6, 8, 10, 8, 9, 8, 6, 7, -1, -1, -1, -1},
    {2, 0, 7, 2, 7, 11, 0, 9, 7, 6, 7, 10, 9, 10, 7, -1},
    {1, 8, 0, 1, 7, 8, 1, 10, 7, 6, 7, 10, 2, 3, 11, -1},
    {11, 2, 1, 11, 1, 7, 10, 6, 1, 6, 7, 1, -1, -1, -1, -1},
    {8, 9, 6, 8, 6, 7, 9, 1, 6, 11, 6, 3, 1, 3, 6, -1},
    {0, 9, 1, 11, 6, 7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {7, 8, 0, 7, 0, 6, 3, 11, 0, 11, 6, 0, -1, -1, -1, -1},
    {7, 11, 6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {7, 6, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {3, 0, 8, 11, 7, 6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 1, 9, 11, 7, 6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {8, 1, 9, 8, 3, 1, 11, 7, 6, -1, -1, -1, -1, -1, -1, -1},
    {10, 1, 2, 6, 11, 7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {1, 2, 10, 3, 0, 8, 6, 11, 7, -1, -1, -1, -1, -1, -1, -1},
    {2, 9, 0, 2, 10, 9, 6, 11, 7, -1, -1, -1, -1, -1, -1, -1},
    {6, 11, 7, 2, 10, 3, 10, 8, 3, 10, 9, 8, -1, -1, -1, -1},
    {7, 2, 3, 6, 2, 7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {7, 0, 8, 7, 6, 0, 6, 2, 0, -1, -1, -1, -1, -1, -1, -1},
    {2, 7, 6, 2, 3, 7, 0, 1, 9, -1, -1, -1, -1, -1, -1, -1},
    {1, 6, 2, 1, 8, 6, 1, 9, 8, 8, 7, 6, -1, -1, -1, -1},
    {10, 7, 6, 10, 1, 7, 1, 3, 7, -1, -1, -1, -1, -1, -1, -1},
    {10, 7, 6, 1, 7, 10, 1, 8, 7, 1, 0, 8, -1, -1, -1, -1},
    {0, 3, 7, 0, 7, 10, 0, 10, 9, 6, 10, 7, -1, -1, -1, -1},
    {7, 6, 10, 7, 10, 8, 8, 10, 9, -1, -1, -1, -1, -1, -1, -1},
    {6, 8, 4, 11, 8, 6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {3, 6, 11, 3, 0, 6, 0, 4, 6, -1, -1, -1, -1, -1, -1, -1},
    {8, 6, 11, 8, 4, 6, 9, 0, 1, -1, -1, -1, -1, -1, -1, -1},
    {9, 4, 6, 9, 6, 3, 9, 3, 1, 11, 3, 6, -1, -1, -1, -1},
    {6, 8, 4, 6, 11, 8, 2, 10, 1, -1, -1, -1, -1, -1, -1, -1},
    {1, 2, 10, 3, 0, 11, 0, 6, 11, 0, 4, 6, -1, -1, -1, -1},
    {4, 11, 8, 4, 6, 11, 0, 2, 9, 2, 10, 9, -1, -1, -1, -1},
    {10, 9, 3, 10, 3, 2, 9, 4, 3, 11, 3, 6, 4, 6, 3, -1},
    {8, 2, 3, 8, 4, 2, 4, 6, 2, -1, -1, -1, -1, -1, -1, -1},
    {0, 4, 2, 4, 6, 2, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {1, 9, 0, 2, 3, 4, 2, 4, 6, 4, 3, 8, -1, -1, -1, -1},
    {1, 9, 4, 1, 4, 2, 2, 4, 6, -1, -1, -1, -1, -1, -1, -1},
    {8, 1, 3, 8, 6, 1, 8, 4, 6, 6, 10, 1, -1, -1, -1, -1},
    {10, 1, 0, 10, 0, 6, 6, 0, 4, -1, -1, -1, -1, -1, -1, -1},
    {4, 6, 3, 4, 3, 8, 6, 10, 3, 0, 3, 9, 10, 9, 3, -1},
    {10, 9, 4, 6, 10, 4, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {4, 9, 5, 7, 6, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 8, 3, 4, 9, 5, 11, 7, 6, -1, -1, -1, -1, -1, -1, -1},
    {5, 0, 1, 5, 4, 0, 7, 6, 11, -1, -1, -1, -1, -1, -1, -1},
    {11, 7, 6, 8, 3, 4, 3, 5, 4, 3, 1, 5, -1, -1, -1, -1},
    {9, 5, 4, 10, 1, 2, 7, 6, 11, -1, -1, -1, -1, -1, -1, -1},
    {6, 11, 7, 1, 2, 10, 0, 8, 3, 4, 9, 5, -1, -1, -1, -1},
    {7, 6, 11, 5, 4, 10, 4, 2, 10, 4, 0, 2, -1, -1, -1, -1},
    {3, 4, 8, 3, 5, 4, 3, 2, 5, 10, 5, 2, 11, 7, 6, -1},
    {7, 2, 3, 7, 6, 2, 5, 4, 9, -1, -1, -1, -1, -1, -1, -1},
    {9, 5, 4, 0, 8, 6, 0, 6, 2, 6, 8, 7, -1, -1, -1, -1},
    {3, 6, 2, 3, 7, 6, 1, 5, 0, 5, 4, 0, -1, -1, -1, -1},
    {6, 2, 8, 6, 8, 7, 2, 1, 8, 4, 8, 5, 1, 5, 8, -1},
    {9, 5, 4, 10, 1, 6, 1, 7, 6, 1, 3, 7, -1, -1, -1, -1},
    {1, 6, 10, 1, 7, 6, 1, 0, 7, 8, 7, 0, 9, 5, 4, -1},
    {4, 0, 10, 4, 10, 5, 0, 3, 10, 6, 10, 7, 3, 7, 10, -1},
    {7, 6, 10, 7, 10, 8, 5, 4, 10, 4, 8, 10, -1, -1, -1, -1},
    {6, 9, 5, 6, 11, 9, 11, 8, 9, -1, -1, -1, -1, -1, -1, -1},
    {3, 6, 11, 0, 6, 3, 0, 5, 6, 0, 9, 5, -1, -1, -1, -1},
    {0, 11, 8, 0, 5, 11, 0, 1, 5, 5, 6, 11, -1, -1, -1, -1},
    {6, 11, 3, 6, 3, 5, 5, 3, 1, -1, -1, -1, -1, -1, -1, -1},
    {1, 2, 10, 9, 5, 11, 9, 11, 8, 11, 5, 6, -1, -1, -1, -1},
    {0, 11, 3, 0, 6, 11, 0, 9, 6, 5, 6, 9, 1, 2, 10, -1},
    {11, 8, 5, 11, 5, 6, 8, 0, 5, 10, 5, 2, 0, 2, 5, -1},
    {6, 11, 3, 6, 3, 5, 2, 10, 3, 10, 5, 3, -1, -1, -1, -1},
    {5, 8, 9, 5, 2, 8, 5, 6, 2, 3, 8, 2, -1, -1, -1, -1},
    {9, 5, 6, 9, 6, 0, 0, 6, 2, -1, -1, -1, -1, -1, -1, -1},
    {1, 5, 8, 1, 8, 0, 5, 6, 8, 3, 8, 2, 6, 2, 8, -1},
    {1, 5, 6, 2, 1, 6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {1, 3, 6, 1, 6, 10, 3, 8, 6, 5, 6, 9, 8, 9, 6, -1},
    {10, 1, 0, 10, 0, 6, 9, 5, 0, 5, 6, 0, -1, -1, -1, -1},
    {0, 3, 8, 5, 6, 10, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {10, 5, 6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {11, 5, 10, 7, 5, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {11, 5, 10, 11, 7, 5, 8, 3, 0, -1, -1, -1, -1, -1, -1, -1},
    {5, 11, 7, 5, 10, 11, 1, 9, 0, -1, -1, -1, -1, -1, -1, -1},
    {10, 7, 5, 10, 11, 7, 9, 8, 1, 8, 3, 1, -1, -1, -1, -1},
    {11, 1, 2, 11, 7, 1, 7, 5, 1, -1, -1, -1, -1, -1, -1, -1},
    {0, 8, 3, 1, 2, 7, 1, 7, 5, 7, 2, 11, -1, -1, -1, -1},
    {9, 7, 5, 9, 2, 7, 9, 0, 2, 2, 11, 7, -1, -1, -1, -1},
    {7, 5, 2, 7, 2, 11, 5, 9, 2, 3, 2, 8, 9, 8, 2, -1},
    {2, 5, 10, 2, 3, 5, 3, 7, 5, -1, -1, -1, -1, -1, -1, -1},
    {8, 2, 0, 8, 5, 2, 8, 7, 5, 10, 2, 5, -1, -1, -1, -1},
    {9, 0, 1, 5, 10, 3, 5, 3, 7, 3, 10, 2, -1, -1, -1, -1},
    {9, 8, 2, 9, 2, 1, 8, 7, 2, 10, 2, 5, 7, 5, 2, -1},
    {1, 3, 5, 3, 7, 5, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 8, 7, 0, 7, 1, 1, 7, 5, -1, -1, -1, -1, -1, -1, -1},
    {9, 0, 3, 9, 3, 5, 5, 3, 7, -1, -1, -1, -1, -1, -1, -1},
    {9, 8, 7, 5, 9, 7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {5, 8, 4, 5, 10, 8, 10, 11, 8, -1, -1, -1, -1, -1, -1, -1},
    {5, 0, 4, 5, 11, 0, 5, 10, 11, 11, 3, 0, -1, -1, -1, -1},
    {0, 1, 9, 8, 4, 10, 8, 10, 11, 10, 4, 5, -1, -1, -1, -1},
    {10, 11, 4, 10, 4, 5, 11, 3, 4, 9, 4, 1, 3, 1, 4, -1},
    {2, 5, 1, 2, 8, 5, 2, 11, 8, 4, 5, 8, -1, -1, -1, -1},
    {0, 4, 11, 0, 11, 3, 4, 5, 11, 2, 11, 1, 5, 1, 11, -1},
    {0, 2, 5, 0, 5, 9, 2, 11, 5, 4, 5, 8, 11, 8, 5, -1},
    {9, 4, 5, 2, 11, 3, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {2, 5, 10, 3, 5, 2, 3, 4, 5, 3, 8, 4, -1, -1, -1, -1},
    {5, 10, 2, 5, 2, 4, 4, 2, 0, -1, -1, -1, -1, -1, -1, -1},
    {3, 10, 2, 3, 5, 10, 3, 8, 5, 4, 5, 8, 0, 1, 9, -1},
    {5, 10, 2, 5, 2, 4, 1, 9, 2, 9, 4, 2, -1, -1, -1, -1},
    {8, 4, 5, 8, 5, 3, 3, 5, 1, -1, -1, -1, -1, -1, -1, -1},
    {0, 4, 5, 1, 0, 5, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {8, 4, 5, 8, 5, 3, 9, 0, 5, 0, 3, 5, -1, -1, -1, -1},
    {9, 4, 5, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {4, 11, 7, 4, 9, 11, 9, 10, 11, -1, -1, -1, -1, -1, -1, -1},
    {0, 8, 3, 4, 9, 7, 9, 11, 7, 9, 10, 11, -1, -1, -1, -1},
    {1, 10, 11, 1, 11, 4, 1, 4, 0, 7, 4, 11, -1, -1, -1, -1},
    {3, 1, 4, 3, 4, 8, 1, 10, 4, 7, 4, 11, 10, 11, 4, -1},
    {4, 11, 7, 9, 11, 4, 9, 2, 11, 9, 1, 2, -1, -1, -1, -1},
    {9, 7, 4, 9, 11, 7, 9, 1, 11, 2, 11, 1, 0, 8, 3, -1},
    {11, 7, 4, 11, 4, 2, 2, 4, 0, -1, -1, -1, -1, -1, -1, -1},
    {11, 7, 4, 11, 4, 2, 8, 3, 4, 3, 2, 4, -1, -1, -1, -1},
    {2, 9, 10, 2, 7, 9, 2, 3, 7, 7, 4, 9, -1, -1, -1, -1},
    {9, 10, 7, 9, 7, 4, 10, 2, 7, 8, 7, 0, 2, 0, 7, -1},
    {3, 7, 10, 3, 10, 2, 7, 4, 10, 1, 10, 0, 4, 0, 10, -1},
    {1, 10, 2, 8, 7, 4, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {4, 9, 1, 4, 1, 7, 7, 1, 3, -1, -1, -1, -1, -1, -1, -1},
    {4, 9, 1, 4, 1, 7, 0, 8, 1, 8, 7, 1, -1, -1, -1, -1},
    {4, 0, 3, 7, 4, 3, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {4, 8, 7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {9, 10, 8, 10, 11, 8, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {3, 0, 9, 3, 9, 11, 11, 9, 10, -1, -1, -1, -1, -1, -1, -1},
    {0, 1, 10, 0, 10, 8, 8, 10, 11, -1, -1, -1, -1, -1, -1, -1},
    {3, 1, 10, 11, 3, 10, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {1, 2, 11, 1, 11, 9, 9, 11, 8, -1, -1, -1, -1, -1, -1, -1},
    {3, 0, 9, 3, 9, 11, 1, 2, 9, 2, 11, 9, -1, -1, -1, -1},
    {0, 2, 11, 8, 0, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {3, 2, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {2, 3, 8, 2, 8, 10, 10, 8, 9, -1, -1, -1, -1, -1, -1, -1},
    {9, 10, 2, 0, 9, 2, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {2, 3, 8, 2, 8, 10, 0, 1, 8, 1, 10, 8, -1, -1, -1, -1},
    {1, 10, 2, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {1, 3, 8, 9, 1, 8, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 9, 1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 3, 8, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1}
};

Vec3 MeshGenerator::vertexInterp(float isolevel, const Vec3& p1, const Vec3& p2, float v1, float v2) {
    if (std::abs(isolevel - v1) < 0.00001f) return p1;
    if (std::abs(isolevel - v2) < 0.00001f) return p2;
    if (std::abs(v1 - v2) < 0.00001f) return p1;
    float mu = (isolevel - v1) / (v2 - v1);
    return p1 + (p2 - p1) * mu;
}

Mesh MeshGenerator::surfels(const SDFEngine& sdf, int resolution) {
    Mesh mesh;
    Vec3 min = sdf.getBoundsMin();
    Vec3 max = sdf.getBoundsMax();
    float stepSize = (max.x - min.x) / resolution;

    // Surfel-based surface extraction
    // For each voxel near the surface, create a small quad facing outward
    for (int x = 0; x < resolution; x++) {
        for (int y = 0; y < resolution; y++) {
            for (int z = 0; z < resolution; z++) {
                float px = min.x + x * stepSize;
                float py = min.y + y * stepSize;
                float pz = min.z + z * stepSize;

                Vec3 center(px, py, pz);
                float val = sdf.evaluate(center);

                // Check if this voxel is near the surface (contains the zero crossing)
                bool nearSurface = (std::abs(val) < stepSize * 0.6f);

                if (nearSurface) {
                    // Create a small quad facing outward based on SDF gradient
                    Vec3 gradient;
                    float h = stepSize * 0.01f;
                    gradient.x = sdf.evaluate(Vec3(px + h, py, pz)) - sdf.evaluate(Vec3(px - h, py, pz));
                    gradient.y = sdf.evaluate(Vec3(px, py + h, pz)) - sdf.evaluate(Vec3(px, py - h, pz));
                    gradient.z = sdf.evaluate(Vec3(px, py, pz + h)) - sdf.evaluate(Vec3(px, py, pz - h));
                    gradient.normalize();

                    // Create perpendicular vectors
                    Vec3 tangent1, tangent2;
                    if (std::abs(gradient.x) < 0.9f) {
                        tangent1 = Vec3(1, 0, 0).cross(gradient).normalized();
                    } else {
                        tangent1 = Vec3(0, 1, 0).cross(gradient).normalized();
                    }
                    tangent2 = gradient.cross(tangent1).normalized();

                    float quadSize = stepSize * 0.5f;

                    // Create a quad
                    Vec3 p0 = center + tangent1 * quadSize + tangent2 * quadSize;
                    Vec3 p1 = center - tangent1 * quadSize + tangent2 * quadSize;
                    Vec3 p2 = center - tangent1 * quadSize - tangent2 * quadSize;
                    Vec3 p3 = center + tangent1 * quadSize - tangent2 * quadSize;

                    unsigned int baseIdx = mesh.getVertexCount();

                    Vertex v0(p0, gradient);
                    Vertex v1(p1, gradient);
                    Vertex v2(p2, gradient);
                    Vertex v3(p3, gradient);

                    v0.color = Vec3(0.7f, 0.7f, 0.8f);
                    v1.color = Vec3(0.7f, 0.7f, 0.8f);
                    v2.color = Vec3(0.7f, 0.7f, 0.8f);
                    v3.color = Vec3(0.7f, 0.7f, 0.8f);

                    mesh.addVertex(v0);
                    mesh.addVertex(v1);
                    mesh.addVertex(v2);
                    mesh.addVertex(v3);

                    // Two triangles for the quad
                    mesh.addTriangle(baseIdx, baseIdx + 1, baseIdx + 2);
                    mesh.addTriangle(baseIdx, baseIdx + 2, baseIdx + 3);
                }
            }
        }
    }

    return mesh;
}

Mesh MeshGenerator::marchingCubes(const SDFEngine& sdf, int resolution) {
    Mesh mesh;
    Vec3 min = sdf.getBoundsMin();
    Vec3 max = sdf.getBoundsMax();
    float stepSize = (max.x - min.x) / resolution;

    for (int x = 0; x < resolution - 1; x++) {
        for (int y = 0; y < resolution - 1; y++) {
            for (int z = 0; z < resolution - 1; z++) {
                Vec3 corners[8];
                float values[8];

                // Standard marching cubes corner ordering
                // 0:(0,0,0) 1:(1,0,0) 2:(1,1,0) 3:(0,1,0) 4:(0,0,1) 5:(1,0,1) 6:(1,1,1) 7:(0,1,1)
                int cornerOffsets[8][3] = {
                    {0,0,0}, {1,0,0}, {1,1,0}, {0,1,0},
                    {0,0,1}, {1,0,1}, {1,1,1}, {0,1,1}
                };

                for (int i = 0; i < 8; i++) {
                    float px = min.x + (x + cornerOffsets[i][0]) * stepSize;
                    float py = min.y + (y + cornerOffsets[i][1]) * stepSize;
                    float pz = min.z + (z + cornerOffsets[i][2]) * stepSize;
                    corners[i] = Vec3(px, py, pz);
                    values[i] = sdf.evaluate(corners[i]);
                }

                int cubeIndex = 0;
                float isolevel = 0.0f;
                for (int i = 0; i < 8; i++) {
                    if (values[i] < isolevel) cubeIndex |= (1 << i);
                }

                if (edgeTable[cubeIndex] == 0) continue;

                Vec3 vertList[12];
                if (edgeTable[cubeIndex] & 1) vertList[0] = vertexInterp(isolevel, corners[0], corners[1], values[0], values[1]);
                if (edgeTable[cubeIndex] & 2) vertList[1] = vertexInterp(isolevel, corners[1], corners[2], values[1], values[2]);
                if (edgeTable[cubeIndex] & 4) vertList[2] = vertexInterp(isolevel, corners[2], corners[3], values[2], values[3]);
                if (edgeTable[cubeIndex] & 8) vertList[3] = vertexInterp(isolevel, corners[3], corners[0], values[3], values[0]);
                if (edgeTable[cubeIndex] & 16) vertList[4] = vertexInterp(isolevel, corners[4], corners[5], values[4], values[5]);
                if (edgeTable[cubeIndex] & 32) vertList[5] = vertexInterp(isolevel, corners[5], corners[6], values[5], values[6]);
                if (edgeTable[cubeIndex] & 64) vertList[6] = vertexInterp(isolevel, corners[6], corners[7], values[6], values[7]);
                if (edgeTable[cubeIndex] & 128) vertList[7] = vertexInterp(isolevel, corners[7], corners[4], values[7], values[4]);
                if (edgeTable[cubeIndex] & 256) vertList[8] = vertexInterp(isolevel, corners[0], corners[4], values[0], values[4]);
                if (edgeTable[cubeIndex] & 512) vertList[9] = vertexInterp(isolevel, corners[1], corners[5], values[1], values[5]);
                if (edgeTable[cubeIndex] & 1024) vertList[10] = vertexInterp(isolevel, corners[2], corners[6], values[2], values[6]);
                if (edgeTable[cubeIndex] & 2048) vertList[11] = vertexInterp(isolevel, corners[3], corners[7], values[3], values[7]);

                // Helper function to calculate normal at a point
                auto calcNormal = [&](const Vec3& p) -> Vec3 {
                    float h = stepSize * 0.01f;
                    Vec3 grad;
                    grad.x = sdf.evaluate(Vec3(p.x + h, p.y, p.z)) - sdf.evaluate(Vec3(p.x - h, p.y, p.z));
                    grad.y = sdf.evaluate(Vec3(p.x, p.y + h, p.z)) - sdf.evaluate(Vec3(p.x, p.y - h, p.z));
                    grad.z = sdf.evaluate(Vec3(p.x, p.y, p.z + h)) - sdf.evaluate(Vec3(p.x, p.y, p.z - h));
                    float len = grad.length();
                    return (len > 0.00001f) ? (grad / len) : Vec3(0, 1, 0);
                };

                // Create triangles using the triTable
                for (int i = 0; triTable[cubeIndex][i] != -1; i += 3) {
                    Vec3 p0 = vertList[triTable[cubeIndex][i]];
                    Vec3 p1 = vertList[triTable[cubeIndex][i + 1]];
                    Vec3 p2 = vertList[triTable[cubeIndex][i + 2]];

                    // Calculate per-vertex normals from SDF gradient
                    Vec3 n0 = calcNormal(p0);
                    Vec3 n1 = calcNormal(p1);
                    Vec3 n2 = calcNormal(p2);

                    unsigned int v0 = mesh.getVertexCount();
                    unsigned int v1 = v0 + 1;
                    unsigned int v2 = v0 + 2;

                    Vertex vert0(p0, n0);
                    Vertex vert1(p1, n1);
                    Vertex vert2(p2, n2);

                    vert0.color = Vec3(0.7f, 0.7f, 0.8f);
                    vert1.color = Vec3(0.7f, 0.7f, 0.8f);
                    vert2.color = Vec3(0.7f, 0.7f, 0.8f);

                    mesh.addVertex(vert0);
                    mesh.addVertex(vert1);
                    mesh.addVertex(vert2);

                    mesh.addTriangle(v0, v1, v2);
                }
            }
        }
    }

    return mesh;
}

Mesh MeshGenerator::createSphere(float radius, int segments, int rings) {
    Mesh mesh;

    for (int r = 0; r <= rings; r++) {
        float phi = M_PI * r / rings;
        for (int s = 0; s <= segments; s++) {
            float theta = 2.0f * M_PI * s / segments;

            float x = radius * std::sin(phi) * std::cos(theta);
            float y = radius * std::cos(phi);
            float z = radius * std::sin(phi) * std::sin(theta);

            Vertex v(Vec3(x, y, z));
            v.normal = Vec3(x, y, z).normalized();
            v.u = (float)s / segments;
            v.v = (float)r / rings;
            mesh.addVertex(v);
        }
    }

    for (int r = 0; r < rings; r++) {
        for (int s = 0; s < segments; s++) {
            int current = r * (segments + 1) + s;
            int next = current + segments + 1;

            mesh.addTriangle(current, next, current + 1);
            mesh.addTriangle(current + 1, next, next + 1);
        }
    }

    return mesh;
}

Mesh MeshGenerator::createCube(float size) {
    Mesh mesh;
    float h = size / 2.0f;

    Vec3 positions[8] = {
        Vec3(-h, -h, -h), Vec3(h, -h, -h), Vec3(h, h, -h), Vec3(-h, h, -h),
        Vec3(-h, -h, h), Vec3(h, -h, h), Vec3(h, h, h), Vec3(-h, h, h)
    };

    int faces[6][4] = {
        {0, 1, 2, 3}, {5, 4, 7, 6}, {4, 0, 3, 7},
        {1, 5, 6, 2}, {4, 5, 1, 0}, {3, 2, 6, 7}
    };

    Vec3 normals[6] = {
        Vec3(0, 0, -1), Vec3(0, 0, 1), Vec3(-1, 0, 0),
        Vec3(1, 0, 0), Vec3(0, -1, 0), Vec3(0, 1, 0)
    };

    for (int f = 0; f < 6; f++) {
        unsigned int baseIdx = mesh.getVertexCount();
        for (int i = 0; i < 4; i++) {
            Vertex v(positions[faces[f][i]]);
            v.normal = normals[f];
            mesh.addVertex(v);
        }
        mesh.addTriangle(baseIdx, baseIdx + 1, baseIdx + 2);
        mesh.addTriangle(baseIdx, baseIdx + 2, baseIdx + 3);
    }

    return mesh;
}

Mesh MeshGenerator::createCylinder(float radius, float height, int segments) {
    Mesh mesh;
    float halfHeight = height / 2.0f;

    // Bottom cap
    mesh.addVertex(Vertex(Vec3(0, -halfHeight, 0), Vec3(0, -1, 0)));
    for (int i = 0; i <= segments; i++) {
        float theta = 2.0f * M_PI * i / segments;
        float x = radius * std::cos(theta);
        float z = radius * std::sin(theta);
        mesh.addVertex(Vertex(Vec3(x, -halfHeight, z), Vec3(0, -1, 0)));
    }

    // Top cap
    unsigned int topCenter = mesh.getVertexCount();
    mesh.addVertex(Vertex(Vec3(0, halfHeight, 0), Vec3(0, 1, 0)));
    for (int i = 0; i <= segments; i++) {
        float theta = 2.0f * M_PI * i / segments;
        float x = radius * std::cos(theta);
        float z = radius * std::sin(theta);
        mesh.addVertex(Vertex(Vec3(x, halfHeight, z), Vec3(0, 1, 0)));
    }

    // Sides
    unsigned int sideStart = mesh.getVertexCount();
    for (int i = 0; i <= segments; i++) {
        float theta = 2.0f * M_PI * i / segments;
        float x = radius * std::cos(theta);
        float z = radius * std::sin(theta);
        Vec3 normal = Vec3(x, 0, z).normalized();

        mesh.addVertex(Vertex(Vec3(x, -halfHeight, z), normal));
        mesh.addVertex(Vertex(Vec3(x, halfHeight, z), normal));
    }

    // Bottom triangles
    for (int i = 0; i < segments; i++) {
        mesh.addTriangle(0, i + 2, i + 1);
    }

    // Top triangles
    for (int i = 0; i < segments; i++) {
        mesh.addTriangle(topCenter, topCenter + i + 1, topCenter + i + 2);
    }

    // Side triangles
    for (int i = 0; i < segments; i++) {
        unsigned int base = sideStart + i * 2;
        mesh.addTriangle(base, base + 2, base + 1);
        mesh.addTriangle(base + 1, base + 2, base + 3);
    }

    return mesh;
}

Mesh MeshGenerator::createTorus(float majorRadius, float minorRadius, int segments, int sides) {
    Mesh mesh;

    for (int i = 0; i <= segments; i++) {
        float u = 2.0f * M_PI * i / segments;
        for (int j = 0; j <= sides; j++) {
            float v = 2.0f * M_PI * j / sides;

            float x = (majorRadius + minorRadius * std::cos(v)) * std::cos(u);
            float y = minorRadius * std::sin(v);
            float z = (majorRadius + minorRadius * std::cos(v)) * std::sin(u);

            Vec3 pos(x, y, z);
            Vec3 center(majorRadius * std::cos(u), 0, majorRadius * std::sin(u));
            Vec3 normal = (pos - center).normalized();

            Vertex vertex(pos, normal);
            mesh.addVertex(vertex);
        }
    }

    for (int i = 0; i < segments; i++) {
        for (int j = 0; j < sides; j++) {
            int current = i * (sides + 1) + j;
            int next = current + sides + 1;

            mesh.addTriangle(current, next, current + 1);
            mesh.addTriangle(current + 1, next, next + 1);
        }
    }

    return mesh;
}

Mesh MeshGenerator::createCone(float radius, float height, int segments) {
    Mesh mesh;

    // Base center
    mesh.addVertex(Vertex(Vec3(0, 0, 0), Vec3(0, -1, 0)));

    // Base vertices
    for (int i = 0; i <= segments; i++) {
        float theta = 2.0f * M_PI * i / segments;
        float x = radius * std::cos(theta);
        float z = radius * std::sin(theta);
        mesh.addVertex(Vertex(Vec3(x, 0, z), Vec3(0, -1, 0)));
    }

    // Apex
    unsigned int apex = mesh.getVertexCount();
    mesh.addVertex(Vertex(Vec3(0, height, 0), Vec3(0, 1, 0)));

    // Base triangles
    for (int i = 0; i < segments; i++) {
        mesh.addTriangle(0, i + 1, i + 2);
    }

    // Side triangles
    for (int i = 0; i < segments; i++) {
        mesh.addTriangle(i + 1, apex, i + 2);
    }

    mesh.computeNormals();
    return mesh;
}

Mesh MeshGenerator::createPlane(float width, float height, int segmentsX, int segmentsY) {
    Mesh mesh;

    for (int y = 0; y <= segmentsY; y++) {
        for (int x = 0; x <= segmentsX; x++) {
            float px = (float)x / segmentsX * width - width / 2.0f;
            float pz = (float)y / segmentsY * height - height / 2.0f;

            Vertex v(Vec3(px, 0, pz), Vec3(0, 1, 0));
            v.u = (float)x / segmentsX;
            v.v = (float)y / segmentsY;
            mesh.addVertex(v);
        }
    }

    for (int y = 0; y < segmentsY; y++) {
        for (int x = 0; x < segmentsX; x++) {
            int i0 = y * (segmentsX + 1) + x;
            int i1 = i0 + 1;
            int i2 = i0 + segmentsX + 1;
            int i3 = i2 + 1;

            mesh.addTriangle(i0, i2, i1);
            mesh.addTriangle(i1, i2, i3);
        }
    }

    return mesh;
}

Mesh MeshGenerator::createIcosahedron(float size) {
    Mesh mesh;
    const float t = (1.0f + std::sqrt(5.0f)) / 2.0f;

    Vec3 vertices[12] = {
        Vec3(-1, t, 0), Vec3(1, t, 0), Vec3(-1, -t, 0), Vec3(1, -t, 0),
        Vec3(0, -1, t), Vec3(0, 1, t), Vec3(0, -1, -t), Vec3(0, 1, -t),
        Vec3(t, 0, -1), Vec3(t, 0, 1), Vec3(-t, 0, -1), Vec3(-t, 0, 1)
    };

    for (int i = 0; i < 12; i++) {
        mesh.addVertex(Vertex(vertices[i].normalized() * size));
    }

    int faces[20][3] = {
        {0, 11, 5}, {0, 5, 1}, {0, 1, 7}, {0, 7, 10}, {0, 10, 11},
        {1, 5, 9}, {5, 11, 4}, {11, 10, 2}, {10, 7, 6}, {7, 1, 8},
        {3, 9, 4}, {3, 4, 2}, {3, 2, 6}, {3, 6, 8}, {3, 8, 9},
        {4, 9, 5}, {2, 4, 11}, {6, 2, 10}, {8, 6, 7}, {9, 8, 1}
    };

    for (int i = 0; i < 20; i++) {
        mesh.addTriangle(faces[i][0], faces[i][1], faces[i][2]);
    }

    mesh.computeNormals();
    return mesh;
}

Mesh MeshGenerator::createTetrahedron(float size) {
    Mesh mesh;
    float a = size / std::sqrt(2.0f);

    mesh.addVertex(Vertex(Vec3(a, a, a)));
    mesh.addVertex(Vertex(Vec3(-a, -a, a)));
    mesh.addVertex(Vertex(Vec3(-a, a, -a)));
    mesh.addVertex(Vertex(Vec3(a, -a, -a)));

    mesh.addTriangle(0, 1, 2);
    mesh.addTriangle(0, 3, 1);
    mesh.addTriangle(0, 2, 3);
    mesh.addTriangle(1, 3, 2);

    mesh.computeNormals();
    return mesh;
}

Mesh MeshGenerator::createOctahedron(float size) {
    Mesh mesh;

    mesh.addVertex(Vertex(Vec3(size, 0, 0)));
    mesh.addVertex(Vertex(Vec3(-size, 0, 0)));
    mesh.addVertex(Vertex(Vec3(0, size, 0)));
    mesh.addVertex(Vertex(Vec3(0, -size, 0)));
    mesh.addVertex(Vertex(Vec3(0, 0, size)));
    mesh.addVertex(Vertex(Vec3(0, 0, -size)));

    mesh.addTriangle(0, 2, 4); mesh.addTriangle(0, 4, 3);
    mesh.addTriangle(0, 3, 5); mesh.addTriangle(0, 5, 2);
    mesh.addTriangle(1, 4, 2); mesh.addTriangle(1, 3, 4);
    mesh.addTriangle(1, 5, 3); mesh.addTriangle(1, 2, 5);

    mesh.computeNormals();
    return mesh;
}

Mesh MeshGenerator::createDodecahedron(float size) {
    Mesh mesh;
    const float phi = (1.0f + std::sqrt(5.0f)) / 2.0f;
    const float invPhi = 1.0f / phi;

    Vec3 vertices[20] = {
        Vec3(1, 1, 1), Vec3(1, 1, -1), Vec3(1, -1, 1), Vec3(1, -1, -1),
        Vec3(-1, 1, 1), Vec3(-1, 1, -1), Vec3(-1, -1, 1), Vec3(-1, -1, -1),
        Vec3(0, invPhi, phi), Vec3(0, invPhi, -phi), Vec3(0, -invPhi, phi), Vec3(0, -invPhi, -phi),
        Vec3(invPhi, phi, 0), Vec3(invPhi, -phi, 0), Vec3(-invPhi, phi, 0), Vec3(-invPhi, -phi, 0),
        Vec3(phi, 0, invPhi), Vec3(phi, 0, -invPhi), Vec3(-phi, 0, invPhi), Vec3(-phi, 0, -invPhi)
    };

    for (int i = 0; i < 20; i++) {
        mesh.addVertex(Vertex(vertices[i].normalized() * size));
    }

    int faces[12][5] = {
        {0, 16, 2, 10, 8}, {0, 8, 4, 14, 12}, {16, 17, 1, 12, 0},
        {1, 9, 11, 3, 17}, {1, 12, 14, 5, 9}, {2, 13, 15, 6, 10},
        {13, 3, 17, 16, 2}, {3, 11, 7, 15, 13}, {4, 8, 10, 6, 18},
        {14, 5, 19, 18, 4}, {5, 19, 7, 11, 9}, {15, 7, 19, 18, 6}
    };

    for (int i = 0; i < 12; i++) {
        mesh.addTriangle(faces[i][0], faces[i][1], faces[i][2]);
        mesh.addTriangle(faces[i][0], faces[i][2], faces[i][3]);
        mesh.addTriangle(faces[i][0], faces[i][3], faces[i][4]);
    }

    mesh.computeNormals();
    return mesh;
}

Mesh MeshGenerator::createTeapot(float scale, int subdivisions) {
    Mesh mesh;

    // Create simplified teapot using basic shapes
    // Body: Sphere with slight vertical stretch
    Mesh body = createSphere(0.5f * scale, subdivisions, subdivisions / 2);
    for (size_t i = 0; i < body.getVertexCount(); i++) {
        Vertex& v = body.getVertex(i);
        v.position.y *= 0.85f;  // Flatten slightly
        v.position.y -= 0.1f * scale;  // Lower the body
    }

    // Spout: Tapered cylinder
    Mesh spout = createCylinder(0.08f * scale, 0.4f * scale, subdivisions / 2);
    for (size_t i = 0; i < spout.getVertexCount(); i++) {
        Vertex& v = spout.getVertex(i);
        // Rotate to horizontal and position
        float y = v.position.y;
        float z = v.position.z;
        v.position.y = z * 0.5f + 0.1f * scale;
        v.position.z = y * 0.3f;
        v.position.x += 0.5f * scale;
        // Taper the spout
        float t = (v.position.x - 0.3f * scale) / (0.5f * scale);
        v.position = v.position * (1.0f - t * 0.6f) + Vec3(v.position.x, 0, 0) * t * 0.6f;
    }

    // Handle: Torus section
    Mesh handle = createTorus(0.25f * scale, 0.05f * scale, subdivisions, subdivisions / 4);
    for (size_t i = 0; i < handle.getVertexCount(); i++) {
        Vertex& v = handle.getVertex(i);
        // Keep only half of torus and position on back
        if (v.position.x < 0.0f) {
            v.position.x -= 0.35f * scale;
        }
    }

    // Lid: Small sphere on top
    Mesh lid = createSphere(0.3f * scale, subdivisions / 2, subdivisions / 4);
    for (size_t i = 0; i < lid.getVertexCount(); i++) {
        Vertex& v = lid.getVertex(i);
        v.position.y *= 0.4f;  // Flatten
        v.position.y += 0.4f * scale;  // Position on top
    }

    // Knob: Tiny sphere on lid
    Mesh knob = createSphere(0.08f * scale, subdivisions / 4, subdivisions / 8);
    for (size_t i = 0; i < knob.getVertexCount(); i++) {
        Vertex& v = knob.getVertex(i);
        v.position.y += 0.52f * scale;
    }

    // Merge all parts
    mesh.merge(body);
    mesh.merge(spout);
    mesh.merge(handle);
    mesh.merge(lid);
    mesh.merge(knob);

    mesh.computeNormals();
    return mesh;
}

Mesh MeshGenerator::createUtahTeapot(float scale, int subdivisions) {
    Mesh mesh;

    // Create more authentic Utah teapot with better proportions
    // Body: Sphere with specific proportions
    Mesh body = createSphere(0.6f * scale, subdivisions * 1.2, subdivisions / 1.5);
    for (size_t i = 0; i < body.getVertexCount(); i++) {
        Vertex& v = body.getVertex(i);
        // More accurate teapot body shape
        v.position.y *= 0.75f;  // Shorter and wider
        v.position.y -= 0.15f * scale;
        // Add slight bulge at bottom
        float distFromBottom = (v.position.y + 0.6f * scale) / (0.6f * scale);
        if (distFromBottom < 0.5f) {
            float bulge = 1.0f + (0.5f - distFromBottom) * 0.15f;
            v.position.x *= bulge;
            v.position.z *= bulge;
        }
    }

    // Spout: More curved and elegant
    Mesh spout = createCylinder(0.06f * scale, 0.45f * scale, subdivisions / 2);
    for (size_t i = 0; i < spout.getVertexCount(); i++) {
        Vertex& v = spout.getVertex(i);
        float y = v.position.y;
        float z = v.position.z;
        // Create curved spout
        float t = (y + 0.225f * scale) / (0.45f * scale);
        v.position.y = z * 0.6f + 0.05f * scale;
        v.position.z = y * 0.2f - t * t * 0.1f * scale;
        v.position.x += 0.55f * scale;
        // Taper the spout more elegantly
        float taper = 1.0f - t * 0.7f;
        v.position = v.position * taper + Vec3(v.position.x, v.position.y, 0) * (1.0f - taper);
    }

    // Handle: More characteristic C-shape
    Mesh handle = createTorus(0.28f * scale, 0.04f * scale, subdivisions, subdivisions / 4);
    for (size_t i = 0; i < handle.getVertexCount(); i++) {
        Vertex& v = handle.getVertex(i);
        // Keep only back half and position
        if (v.position.x < 0.0f) {
            v.position.x -= 0.4f * scale;
            v.position.y -= 0.05f * scale;
        }
    }

    // Lid: Flatter and more detailed
    Mesh lid = createSphere(0.35f * scale, subdivisions / 1.5, subdivisions / 3);
    for (size_t i = 0; i < lid.getVertexCount(); i++) {
        Vertex& v = lid.getVertex(i);
        v.position.y *= 0.3f;  // Very flat
        v.position.y += 0.45f * scale;
    }

    // Knob: More prominent
    Mesh knob = createSphere(0.1f * scale, subdivisions / 3, subdivisions / 6);
    for (size_t i = 0; i < knob.getVertexCount(); i++) {
        Vertex& v = knob.getVertex(i);
        v.position.y *= 0.6f;  // Slightly flattened
        v.position.y += 0.58f * scale;
    }

    // Merge all parts
    mesh.merge(body);
    mesh.merge(spout);
    mesh.merge(handle);
    mesh.merge(lid);
    mesh.merge(knob);

    mesh.computeNormals();
    return mesh;
}

Mesh MeshGenerator::createOilTank(float radius, float height, int segments) {
    Mesh mesh;

    // Main cylindrical body
    Mesh body = createCylinder(radius, height, segments);

    // Top cap (hemisphere)
    Mesh topCap = createSphere(radius, segments, segments / 2);
    for (size_t i = 0; i < topCap.getVertexCount(); i++) {
        Vertex& v = topCap.getVertex(i);
        // Keep only top half and flatten slightly
        if (v.position.y > 0) {
            v.position.y *= 0.7f;  // Flatten the dome
            v.position.y += height / 2.0f;
        }
    }

    // Bottom cap (hemisphere)
    Mesh bottomCap = createSphere(radius, segments, segments / 2);
    for (size_t i = 0; i < bottomCap.getVertexCount(); i++) {
        Vertex& v = bottomCap.getVertex(i);
        // Keep only bottom half and flatten slightly
        if (v.position.y < 0) {
            v.position.y *= 0.7f;  // Flatten the dome
            v.position.y -= height / 2.0f;
        }
    }

    // Support legs (4 cylinders at corners)
    float legRadius = radius * 0.05f;
    float legHeight = height * 0.3f;
    float legOffset = radius * 0.7f;

    for (int i = 0; i < 4; i++) {
        float angle = i * M_PI / 2.0f;
        Mesh leg = createCylinder(legRadius, legHeight, segments / 4);
        for (size_t j = 0; j < leg.getVertexCount(); j++) {
            Vertex& v = leg.getVertex(j);
            float x = v.position.x + legOffset * std::cos(angle);
            float z = v.position.z + legOffset * std::sin(angle);
            v.position.x = x;
            v.position.z = z;
            v.position.y -= (height / 2.0f + legHeight / 2.0f);
        }
        mesh.merge(leg);
    }

    // Add access ladder on one side
    float ladderWidth = radius * 0.15f;
    int ladderRungs = 6;
    for (int i = 0; i < ladderRungs; i++) {
        float yPos = -height * 0.4f + (i * height * 0.8f / (ladderRungs - 1));
        Mesh rung = createCylinder(legRadius * 0.5f, ladderWidth, 8);
        for (size_t j = 0; j < rung.getVertexCount(); j++) {
            Vertex& v = rung.getVertex(j);
            float temp = v.position.y;
            v.position.y = yPos;
            v.position.z = v.position.z + radius;
            v.position.x = temp;
        }
        mesh.merge(rung);
    }

    // Merge main components
    mesh.merge(body);
    mesh.merge(topCap);
    mesh.merge(bottomCap);

    mesh.computeNormals();
    return mesh;
}

// ========== NEW PRIMITIVES ==========

Mesh MeshGenerator::createGeoSphere(float radius, int subdivisions) {
    Mesh mesh = createIcosahedron(radius);
    subdivide(mesh, subdivisions);

    // Normalize vertices to sphere surface
    for (size_t i = 0; i < mesh.getVertexCount(); i++) {
        Vertex& v = mesh.getVertex(i);
        float len = std::sqrt(v.position.x * v.position.x +
                             v.position.y * v.position.y +
                             v.position.z * v.position.z);
        if (len > 0.0f) {
            v.position.x = (v.position.x / len) * radius;
            v.position.y = (v.position.y / len) * radius;
            v.position.z = (v.position.z / len) * radius;
        }
    }

    mesh.computeNormals();
    return mesh;
}

Mesh MeshGenerator::createTube(float radius, float height, float thickness, int segments) {
    Mesh mesh;
    Mesh outer = createCylinder(radius, height, segments);
    Mesh inner = createCylinder(radius - thickness, height, segments);

    // Flip inner cylinder normals
    for (size_t i = 0; i < inner.getTriangleCount(); i++) {
        Triangle& tri = inner.getTriangle(i);
        std::swap(tri.v0, tri.v2);
    }

    mesh.merge(outer);
    mesh.merge(inner);
    mesh.computeNormals();
    return mesh;
}

Mesh MeshGenerator::createPyramid(float baseSize, float height, int subdivisions) {
    Mesh mesh;
    float half = baseSize / 2.0f;

    // Base vertices
    Vec3 base[4] = {
        Vec3(-half, -height/2, -half),
        Vec3( half, -height/2, -half),
        Vec3( half, -height/2,  half),
        Vec3(-half, -height/2,  half)
    };

    // Apex
    Vec3 apex(0, height/2, 0);

    // Add base
    mesh.addVertex(Vertex(base[0]));
    mesh.addVertex(Vertex(base[1]));
    mesh.addVertex(Vertex(base[2]));
    mesh.addVertex(Vertex(base[3]));
    mesh.addTriangle(Triangle(0, 2, 1));
    mesh.addTriangle(Triangle(0, 3, 2));

    // Add apex
    mesh.addVertex(Vertex(apex));
    int apexIdx = 4; // Index of the apex vertex

    // Add faces
    for (int i = 0; i < 4; i++) {
        int next = (i + 1) % 4;
        mesh.addTriangle(Triangle(i, next, apexIdx));
    }

    mesh.computeNormals();
    return mesh;
}

Mesh MeshGenerator::createHedra(float radius, int sides, int subdivision) {
    Mesh mesh;

    // Create a basic polyhedron based on sides
    if (sides == 4) mesh = createTetrahedron(radius);
    else if (sides == 8) mesh = createOctahedron(radius);
    else if (sides == 12) mesh = createDodecahedron(radius);
    else if (sides == 20) mesh = createIcosahedron(radius);
    else mesh = createCube(radius);

    if (subdivision > 0) {
        subdivide(mesh, subdivision);
    }

    return mesh;
}

Mesh MeshGenerator::createTorusKnot(int p, int q, float tubeRadius, int segments) {
    Mesh mesh;
    const int tubeSides = 16;

    for (int i = 0; i < segments; i++) {
        float t = 2.0f * M_PI * i / segments;
        float tnext = 2.0f * M_PI * (i + 1) / segments;

        // Torus knot parametric equations
        float r = std::cos(q * t) + 2.0f;
        float x = r * std::cos(p * t);
        float y = r * std::sin(p * t);
        float z = -std::sin(q * t);

        Vec3 center(x, y, z);

        // Create tube cross-section
        for (int j = 0; j < tubeSides; j++) {
            float angle = 2.0f * M_PI * j / tubeSides;
            Vec3 offset(tubeRadius * std::cos(angle), tubeRadius * std::sin(angle), 0);
            mesh.addVertex(Vertex(center + offset));
        }
    }

    // Connect vertices
    for (int i = 0; i < segments; i++) {
        int next = (i + 1) % segments;
        for (int j = 0; j < tubeSides; j++) {
            int jnext = (j + 1) % tubeSides;
            int i0 = i * tubeSides + j;
            int i1 = i * tubeSides + jnext;
            int i2 = next * tubeSides + jnext;
            int i3 = next * tubeSides + j;

            mesh.addTriangle(Triangle(i0, i1, i2));
            mesh.addTriangle(Triangle(i0, i2, i3));
        }
    }

    mesh.computeNormals();
    return mesh;
}


Mesh MeshGenerator::createCapsule(float radius, float height, int segments) {
    Mesh mesh;

    // Cylinder body
    Mesh body = createCylinder(radius, height - 2*radius, segments);

    // Top hemisphere
    Mesh topSphere = createSphere(radius, segments, segments/2);
    for (size_t i = 0; i < topSphere.getVertexCount(); i++) {
        Vertex& v = topSphere.getVertex(i);
        if (v.position.y >= 0) {
            v.position.y += (height - 2*radius) / 2;
        }
    }

    // Bottom hemisphere
    Mesh bottomSphere = createSphere(radius, segments, segments/2);
    for (size_t i = 0; i < bottomSphere.getVertexCount(); i++) {
        Vertex& v = bottomSphere.getVertex(i);
        if (v.position.y <= 0) {
            v.position.y -= (height - 2*radius) / 2;
        }
    }

    mesh.merge(body);
    mesh.merge(topSphere);
    mesh.merge(bottomSphere);
    mesh.computeNormals();
    return mesh;
}

Mesh MeshGenerator::createSpindle(float radius, float height, int segments) {
    Mesh mesh;

    // Create two cones joined at their bases
    Mesh topCone = createCone(radius, height/2, segments);
    Mesh bottomCone = createCone(radius, height/2, segments);

    // Position cones
    for (size_t i = 0; i < topCone.getVertexCount(); i++) {
        topCone.getVertex(i).position.y += height/4;
    }

    for (size_t i = 0; i < bottomCone.getVertexCount(); i++) {
        Vertex& v = bottomCone.getVertex(i);
        v.position.y = -v.position.y - height/4;
    }

    mesh.merge(topCone);
    mesh.merge(bottomCone);
    mesh.computeNormals();
    return mesh;
}

Mesh MeshGenerator::createLExt(float width, float height, float thickness, float depth) {
    Mesh mesh;

    // Create L-shape as two boxes
    Mesh horizontal = createCube(1.0f);
    Mesh vertical = createCube(1.0f);

    // Scale and position horizontal part
    for (size_t i = 0; i < horizontal.getVertexCount(); i++) {
        Vertex& v = horizontal.getVertex(i);
        v.position.x *= width / 2;
        v.position.y *= thickness / 2;
        v.position.z *= depth / 2;
        v.position.y -= height/2 - thickness/2;
    }

    // Scale and position vertical part
    for (size_t i = 0; i < vertical.getVertexCount(); i++) {
        Vertex& v = vertical.getVertex(i);
        v.position.x *= thickness / 2;
        v.position.y *= height / 2;
        v.position.z *= depth / 2;
        v.position.x -= width/2 - thickness/2;
    }

    mesh.merge(horizontal);
    mesh.merge(vertical);
    mesh.computeNormals();
    return mesh;
}

Mesh MeshGenerator::createCExt(float width, float height, float thickness, float depth) {
    Mesh mesh;

    // Create C-shape as three boxes
    Mesh top = createCube(1.0f);
    Mesh middle = createCube(1.0f);
    Mesh bottom = createCube(1.0f);

    // Scale and position parts
    for (size_t i = 0; i < top.getVertexCount(); i++) {
        Vertex& v = top.getVertex(i);
        v.position.x *= width / 2;
        v.position.y *= thickness / 2;
        v.position.z *= depth / 2;
        v.position.y += height/2 - thickness/2;
    }

    for (size_t i = 0; i < middle.getVertexCount(); i++) {
        Vertex& v = middle.getVertex(i);
        v.position.x *= thickness / 2;
        v.position.y *= (height - 2*thickness) / 2;
        v.position.z *= depth / 2;
        v.position.x -= width/2 - thickness/2;
    }

    for (size_t i = 0; i < bottom.getVertexCount(); i++) {
        Vertex& v = bottom.getVertex(i);
        v.position.x *= width / 2;
        v.position.y *= thickness / 2;
        v.position.z *= depth / 2;
        v.position.y -= height/2 - thickness/2;
    }

    mesh.merge(top);
    mesh.merge(middle);
    mesh.merge(bottom);
    mesh.computeNormals();
    return mesh;
}

Mesh MeshGenerator::createGengon(float radius, int sides, float fillet, float height) {
    Mesh mesh;

    // Create a prism and apply fillet
    for (int i = 0; i <= sides; i++) {
        float angle = 2.0f * M_PI * i / sides;
        float nextAngle = 2.0f * M_PI * (i + 1) / sides;

        float x = radius * std::cos(angle);
        float z = radius * std::sin(angle);

        // Add vertices at top and bottom
        mesh.addVertex(Vertex(Vec3(x, height/2, z)));
        mesh.addVertex(Vertex(Vec3(x, -height/2, z)));
    }

    // Create faces
    for (int i = 0; i < sides; i++) {
        int i0 = i * 2;
        int i1 = i * 2 + 1;
        int i2 = ((i + 1) % sides) * 2;
        int i3 = ((i + 1) % sides) * 2 + 1;

        mesh.addTriangle(Triangle(i0, i2, i1));
        mesh.addTriangle(Triangle(i1, i2, i3));
    }

    mesh.computeNormals();
    return mesh;
}

Mesh MeshGenerator::createRingWave(float radius, float thickness, float waveAmplitude, int waveFrequency, int segments) {
    Mesh mesh = createTorus(radius, thickness, segments, 16);

    // Apply wave deformation
    for (size_t i = 0; i < mesh.getVertexCount(); i++) {
        Vertex& v = mesh.getVertex(i);
        float angle = std::atan2(v.position.z, v.position.x);
        float wave = waveAmplitude * std::sin(waveFrequency * angle);

        // Apply wave to radial direction
        float currentRadius = std::sqrt(v.position.x * v.position.x + v.position.z * v.position.z);
        float newRadius = currentRadius + wave;
        if (currentRadius > 0) {
            v.position.x *= newRadius / currentRadius;
            v.position.z *= newRadius / currentRadius;
        }
    }

    mesh.computeNormals();
    return mesh;
}

Mesh MeshGenerator::createHose(float radius, float length, float bendAngle, int segments) {
    Mesh mesh;
    const int tubeSides = 16;
    float bendRadius = length / 2.0f;
    float angleRad = bendAngle * M_PI / 180.0f;

    for (int i = 0; i <= segments; i++) {
        float t = angleRad * i / segments;
        float x = bendRadius * std::sin(t);
        float y = bendRadius * (1.0f - std::cos(t));

        // Create tube cross-section
        for (int j = 0; j < tubeSides; j++) {
            float angle = 2.0f * M_PI * j / tubeSides;
            float dx = radius * std::cos(angle) * std::cos(t);
            float dy = radius * std::sin(angle);
            float dz = radius * std::cos(angle) * std::sin(t);

            mesh.addVertex(Vertex(Vec3(x + dx, y + dy, dz)));
        }
    }

    // Connect vertices
    for (int i = 0; i < segments; i++) {
        for (int j = 0; j < tubeSides; j++) {
            int jnext = (j + 1) % tubeSides;
            int i0 = i * tubeSides + j;
            int i1 = i * tubeSides + jnext;
            int i2 = (i + 1) * tubeSides + jnext;
            int i3 = (i + 1) * tubeSides + j;

            mesh.addTriangle(Triangle(i0, i1, i2));
            mesh.addTriangle(Triangle(i0, i2, i3));
        }
    }

    mesh.computeNormals();
    return mesh;
}

Mesh MeshGenerator::createPrism(float radius, int sides, float height, int heightSegments) {
    Mesh mesh;

    // Create vertices
    for (int h = 0; h <= heightSegments; h++) {
        float y = -height/2 + (height * h) / heightSegments;

        for (int i = 0; i < sides; i++) {
            float angle = 2.0f * M_PI * i / sides;
            float x = radius * std::cos(angle);
            float z = radius * std::sin(angle);
            mesh.addVertex(Vertex(Vec3(x, y, z)));
        }
    }

    // Create side faces
    for (int h = 0; h < heightSegments; h++) {
        for (int i = 0; i < sides; i++) {
            int next = (i + 1) % sides;
            int i0 = h * sides + i;
            int i1 = h * sides + next;
            int i2 = (h + 1) * sides + next;
            int i3 = (h + 1) * sides + i;

            mesh.addTriangle(Triangle(i0, i1, i2));
            mesh.addTriangle(Triangle(i0, i2, i3));
        }
    }

    // Add top and bottom caps
    mesh.addVertex(Vertex(Vec3(0, height/2, 0)));
    mesh.addVertex(Vertex(Vec3(0, -height/2, 0)));
    int topCenter = (heightSegments + 1) * sides;
    int bottomCenter = topCenter + 1;

    for (int i = 0; i < sides; i++) {
        int next = (i + 1) % sides;
        mesh.addTriangle(Triangle(topCenter, heightSegments * sides + i, heightSegments * sides + next));
        mesh.addTriangle(Triangle(bottomCenter, next, i));
    }

    mesh.computeNormals();
    return mesh;
}

Mesh MeshGenerator::createMobiusStrip(float radius, float width, int segments) {
    Mesh mesh;

    for (int i = 0; i <= segments; i++) {
        float u = 2.0f * M_PI * i / segments;
        float halfTwist = u / 2.0f;

        for (int j = -1; j <= 1; j += 2) {
            float v = j * width / 2.0f;
            float x = (radius + v * std::cos(halfTwist)) * std::cos(u);
            float y = (radius + v * std::cos(halfTwist)) * std::sin(u);
            float z = v * std::sin(halfTwist);

            mesh.addVertex(Vertex(Vec3(x, y, z)));
        }
    }

    for (int i = 0; i < segments; i++) {
        int base = i * 2;
        mesh.addTriangle(base, base + 2, base + 1);
        mesh.addTriangle(base + 1, base + 2, base + 3);
    }

    mesh.computeNormals();
    return mesh;
}

Mesh MeshGenerator::createKleinBottle(int segments) {
    Mesh mesh;

    for (int i = 0; i <= segments; i++) {
        float u = 2.0f * M_PI * i / segments;
        for (int j = 0; j <= segments; j++) {
            float v = 2.0f * M_PI * j / segments;

            float r = 4.0f * (1.0f - std::cos(u) / 2.0f);
            float x = 6.0f * std::cos(u) * (1.0f + std::sin(u)) + r * std::cos(u) * std::cos(v);
            float y = 16.0f * std::sin(u) + r * std::sin(u) * std::cos(v);
            float z = r * std::sin(v);

            mesh.addVertex(Vertex(Vec3(x / 10.0f, y / 10.0f, z / 10.0f)));
        }
    }

    for (int i = 0; i < segments; i++) {
        for (int j = 0; j < segments; j++) {
            int base = i * (segments + 1) + j;
            int next = base + segments + 1;
            mesh.addTriangle(base, next, base + 1);
            mesh.addTriangle(base + 1, next, next + 1);
        }
    }

    mesh.computeNormals();
    return mesh;
}

Mesh MeshGenerator::createTrefoilKnot(float scale, int segments) {
    // Generate trefoil knot curve points
    std::vector<Vec3> curvePoints;
    for (int i = 0; i <= segments; i++) {
        float t = 2.0f * M_PI * i / segments;
        float x = std::sin(t) + 2.0f * std::sin(2.0f * t);
        float y = std::cos(t) - 2.0f * std::cos(2.0f * t);
        float z = -std::sin(3.0f * t);
        curvePoints.push_back(Vec3(x * scale, y * scale, z * scale));
    }

    // Create tube along the curve
    Mesh mesh;
    float tubeRadius = 0.15f * scale;
    int tubeSegments = 12;

    for (size_t i = 0; i < curvePoints.size(); i++) {
        size_t nextIdx = (i + 1) % curvePoints.size();
        Vec3 p1 = curvePoints[i];
        Vec3 p2 = curvePoints[nextIdx];
        Vec3 dir = p2 - p1;
        dir.normalize();

        // Create perpendicular vectors
        Vec3 up = (std::abs(dir.y) < 0.9f) ? Vec3(0, 1, 0) : Vec3(1, 0, 0);
        Vec3 right = dir.cross(up);
        right.normalize();
        up = right.cross(dir);
        up.normalize();

        // Create ring of vertices
        for (int j = 0; j < tubeSegments; j++) {
            float angle = 2.0f * M_PI * j / tubeSegments;
            Vec3 offset = right * std::cos(angle) * tubeRadius + up * std::sin(angle) * tubeRadius;
            mesh.addVertex(Vertex(p1 + offset));
        }
    }

    // Create triangles
    for (int i = 0; i < (int)curvePoints.size(); i++) {
        for (int j = 0; j < tubeSegments; j++) {
            int curr = i * tubeSegments + j;
            int next = i * tubeSegments + (j + 1) % tubeSegments;
            int currNext = ((i + 1) % curvePoints.size()) * tubeSegments + j;
            int nextNext = ((i + 1) % curvePoints.size()) * tubeSegments + (j + 1) % tubeSegments;

            mesh.addTriangle(curr, currNext, next);
            mesh.addTriangle(next, currNext, nextNext);
        }
    }

    mesh.computeNormals();
    return mesh;
}

Mesh MeshGenerator::createGrid(int width, int height, float spacing) {
    return createPlane(width * spacing, height * spacing, width, height);
}

Mesh MeshGenerator::createHexGrid(int rings, float radius) {
    Mesh mesh;
    // Simplified hex grid
    return createPlane(rings * radius * 2, rings * radius * 2, rings, rings);
}

void MeshGenerator::subdivide(Mesh& mesh, int iterations) {
    for (int iter = 0; iter < iterations; iter++) {
        std::vector<Triangle> newTriangles;
        size_t oldVertexCount = mesh.getVertexCount();

        std::map<std::pair<unsigned int, unsigned int>, unsigned int> edgeMidpoints;

        auto getMidpoint = [&](unsigned int v0, unsigned int v1) -> unsigned int {
            if (v0 > v1) std::swap(v0, v1);
            auto key = std::make_pair(v0, v1);

            if (edgeMidpoints.find(key) != edgeMidpoints.end()) {
                return edgeMidpoints[key];
            }

            const Vertex& vert0 = mesh.getVertex(v0);
            const Vertex& vert1 = mesh.getVertex(v1);

            Vertex mid;
            mid.position = (vert0.position + vert1.position) * 0.5f;
            mid.normal = (vert0.normal + vert1.normal).normalized();
            mid.color = (vert0.color + vert1.color) * 0.5f;
            mid.u = (vert0.u + vert1.u) * 0.5f;
            mid.v = (vert0.v + vert1.v) * 0.5f;

            unsigned int idx = mesh.getVertexCount();
            mesh.addVertex(mid);
            edgeMidpoints[key] = idx;
            return idx;
        };

        size_t triangleCount = mesh.getTriangleCount();
        for (size_t i = 0; i < triangleCount; i++) {
            const Triangle& tri = mesh.getTriangle(i);

            unsigned int m01 = getMidpoint(tri.v0, tri.v1);
            unsigned int m12 = getMidpoint(tri.v1, tri.v2);
            unsigned int m20 = getMidpoint(tri.v2, tri.v0);

            newTriangles.push_back(Triangle(tri.v0, m01, m20));
            newTriangles.push_back(Triangle(tri.v1, m12, m01));
            newTriangles.push_back(Triangle(tri.v2, m20, m12));
            newTriangles.push_back(Triangle(m01, m12, m20));
        }

        mesh.getTriangles().clear();
        for (const auto& tri : newTriangles) {
            mesh.addTriangle(tri);
        }
    }
}

void MeshGenerator::extrude(Mesh& mesh, const Vec3& direction, float distance) {
    size_t originalVertexCount = mesh.getVertexCount();

    for (size_t i = 0; i < originalVertexCount; i++) {
        Vertex v = mesh.getVertex(i);
        v.position += direction * distance;
        mesh.addVertex(v);
    }
}

Mesh MeshGenerator::lathe(const std::vector<Vec3>& profile, int segments) {
    Mesh mesh;

    for (int i = 0; i < segments; i++) {
        float angle = 2.0f * M_PI * i / segments;
        float c = std::cos(angle);
        float s = std::sin(angle);

        for (const auto& p : profile) {
            float x = p.x * c - p.z * s;
            float y = p.y;
            float z = p.x * s + p.z * c;
            mesh.addVertex(Vertex(Vec3(x, y, z)));
        }
    }

    int profileSize = profile.size();
    for (int i = 0; i < segments; i++) {
        int nextSegment = (i + 1) % segments;
        for (int j = 0; j < profileSize - 1; j++) {
            int v0 = i * profileSize + j;
            int v1 = nextSegment * profileSize + j;
            int v2 = i * profileSize + j + 1;
            int v3 = nextSegment * profileSize + j + 1;

            mesh.addTriangle(v0, v1, v2);
            mesh.addTriangle(v2, v1, v3);
        }
    }

    mesh.computeNormals();
    return mesh;
}

Mesh MeshGenerator::loft(const std::vector<std::vector<Vec3>>& sections) {
    Mesh mesh;

    for (const auto& section : sections) {
        for (const auto& point : section) {
            mesh.addVertex(Vertex(point));
        }
    }

    if (sections.size() < 2) return mesh;

    int pointsPerSection = sections[0].size();
    for (size_t i = 0; i < sections.size() - 1; i++) {
        for (int j = 0; j < pointsPerSection; j++) {
            int next = (j + 1) % pointsPerSection;
            int v0 = i * pointsPerSection + j;
            int v1 = i * pointsPerSection + next;
            int v2 = (i + 1) * pointsPerSection + j;
            int v3 = (i + 1) * pointsPerSection + next;

            mesh.addTriangle(v0, v2, v1);
            mesh.addTriangle(v1, v2, v3);
        }
    }

    mesh.computeNormals();
    return mesh;
}
