#include "CMAESOptimizer.h"
#include <random>
#include <algorithm>
#include <cmath>

CMAESOptimizer::CMAESOptimizer(int dimension, double sigma)
    : dimension(dimension), sigma(sigma), populationSize(4 + (int)(3 * std::log(dimension))) {
    mean.resize(dimension, 0.0);
    covarianceMatrix.resize(dimension, std::vector<double>(dimension, 0.0));

    for (int i = 0; i < dimension; i++) {
        covarianceMatrix[i][i] = 1.0;
    }
}

std::vector<double> CMAESOptimizer::optimize(ObjectiveFunction func, int maxIterations) {
    std::random_device rd;
    std::mt19937 gen(rd());
    std::normal_distribution<double> dist(0.0, 1.0);

    std::vector<double> bestSolution = mean;
    double bestFitness = func(mean);

    for (int iter = 0; iter < maxIterations; iter++) {
        std::vector<std::vector<double>> population(populationSize);
        std::vector<double> fitness(populationSize);

        // Generate population
        for (int i = 0; i < populationSize; i++) {
            population[i].resize(dimension);
            for (int j = 0; j < dimension; j++) {
                population[i][j] = mean[j] + sigma * dist(gen);
            }
            fitness[i] = func(population[i]);

            if (fitness[i] < bestFitness) {
                bestFitness = fitness[i];
                bestSolution = population[i];
            }
        }

        // Update mean (simplified)
        std::fill(mean.begin(), mean.end(), 0.0);
        int eliteCount = populationSize / 2;

        std::vector<int> indices(populationSize);
        for (int i = 0; i < populationSize; i++) indices[i] = i;
        std::sort(indices.begin(), indices.end(),
                  [&](int a, int b) { return fitness[a] < fitness[b]; });

        for (int i = 0; i < eliteCount; i++) {
            for (int j = 0; j < dimension; j++) {
                mean[j] += population[indices[i]][j] / eliteCount;
            }
        }

        // Adapt step size (simplified)
        sigma *= 0.99;
    }

    return bestSolution;
}
