#include "RBFOptimizer.h"
#include <random>
#include <cmath>
#include <algorithm>

RBFOptimizer::RBFOptimizer(int dimension) : dimension(dimension) {
    lowerBounds.resize(dimension, -5.0);
    upperBounds.resize(dimension, 5.0);
}

void RBFOptimizer::setBounds(const std::vector<double>& lower, const std::vector<double>& upper) {
    lowerBounds = lower;
    upperBounds = upper;
}

double RBFOptimizer::rbfKernel(const std::vector<double>& x1, const std::vector<double>& x2) {
    double dist = 0.0;
    for (size_t i = 0; i < x1.size(); i++) {
        double diff = x1[i] - x2[i];
        dist += diff * diff;
    }
    return std::exp(-dist / (2.0 * dimension));
}

std::vector<double> RBFOptimizer::selectNextPoint(const std::vector<std::vector<double>>& evaluated) {
    std::random_device rd;
    std::mt19937 gen(rd());

    std::vector<double> nextPoint(dimension);
    for (int i = 0; i < dimension; i++) {
        std::uniform_real_distribution<double> dist(lowerBounds[i], upperBounds[i]);
        nextPoint[i] = dist(gen);
    }

    return nextPoint;
}

std::vector<double> RBFOptimizer::optimize(ObjectiveFunction func, int maxEvaluations) {
    std::vector<std::vector<double>> evaluatedPoints;
    std::vector<double> evaluatedValues;

    std::random_device rd;
    std::mt19937 gen(rd());

    // Initial sampling
    for (int i = 0; i < std::min(10, maxEvaluations); i++) {
        std::vector<double> point(dimension);
        for (int j = 0; j < dimension; j++) {
            std::uniform_real_distribution<double> dist(lowerBounds[j], upperBounds[j]);
            point[j] = dist(gen);
        }
        double value = func(point);
        evaluatedPoints.push_back(point);
        evaluatedValues.push_back(value);
    }

    // Find best so far
    auto minIt = std::min_element(evaluatedValues.begin(), evaluatedValues.end());
    int bestIdx = std::distance(evaluatedValues.begin(), minIt);

    // Iterative optimization
    for (int iter = evaluatedPoints.size(); iter < maxEvaluations; iter++) {
        std::vector<double> nextPoint = selectNextPoint(evaluatedPoints);
        double value = func(nextPoint);

        evaluatedPoints.push_back(nextPoint);
        evaluatedValues.push_back(value);

        if (value < evaluatedValues[bestIdx]) {
            bestIdx = evaluatedPoints.size() - 1;
        }
    }

    return evaluatedPoints[bestIdx];
}
