#pragma once
#include "../math/Vec3.h"
#include "SDFPrimitives.h"
#include <functional>

class SDFEngine {
public:
    using SDFFunction = std::function<float(const Vec3&)>;

private:
    SDFFunction sdfFunc;
    Vec3 boundsMin;
    Vec3 boundsMax;
    float threshold;

public:
    SDFEngine() : boundsMin(-5, -5, -5), boundsMax(5, 5, 5), threshold(0.001f) {
        // Default SDF: sphere
        sdfFunc = [](const Vec3& p) { return SDFPrimitives::sphere(p, 1.0f); };
    }

    void setSDF(SDFFunction func) {
        sdfFunc = func;
    }

    void setBounds(const Vec3& min, const Vec3& max) {
        boundsMin = min;
        boundsMax = max;
    }

    void setThreshold(float t) {
        threshold = t;
    }

    float evaluate(const Vec3& p) const {
        return sdfFunc(p);
    }

    Vec3 estimateNormal(const Vec3& p, float epsilon = 0.001f) const {
        float d = evaluate(p);
        return Vec3(
            evaluate(Vec3(p.x + epsilon, p.y, p.z)) - d,
            evaluate(Vec3(p.x, p.y + epsilon, p.z)) - d,
            evaluate(Vec3(p.x, p.y, p.z + epsilon)) - d
        ).normalized();
    }

    Vec3 rayMarch(const Vec3& origin, const Vec3& direction, float maxDist = 100.0f, int maxSteps = 100) const {
        float t = 0.0f;
        for (int i = 0; i < maxSteps; i++) {
            Vec3 p = origin + direction * t;
            float d = evaluate(p);
            if (d < threshold) {
                return p;
            }
            t += d;
            if (t > maxDist) {
                break;
            }
        }
        return Vec3(0, 0, 0); // No intersection
    }

    const Vec3& getBoundsMin() const { return boundsMin; }
    const Vec3& getBoundsMax() const { return boundsMax; }
    float getThreshold() const { return threshold; }
};
